// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

VPBPlay::VPBPlay(VPBTrunk *trunk, int h) :
Service((Trunk *)trunk, keythreads.priAudio()), AudioFile()
{
	handle = h;
	reset = false;
}

VPBPlay::~VPBPlay()
{
	char buffer[12];
	Terminate();
	sprintf(buffer, "%d", getPosition());
	trunk->setSymbol(SYM_PLAYED, buffer);
	Close();
	if(reset)
	{
		vpb_play_terminate(handle);
		dspReset();
	}
}

char *VPBPlay::getContinuation(void)
{
	return getPlayfile();
}

void VPBPlay::Initial(void)
{
	int codec;
	char buffer[32];
	char *fn = getPlayfile();
	struct stat ino;
	struct tm *dt;
	struct tm tbuf;
	char *ann;
	
	trunk->getName(buffer);
	stat(fn, &ino);
	Open(fn);	
	if(!isOpen())
	{
		slog(SLOG_ERROR) << fn << ": cannot open" << endl;
		Failure();
	}

	dt = localtime_r(&ino.st_ctime, &tbuf);
	sprintf(buffer, "%04d%02d%02d,%02d%02d%02d",
		dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday,
		dt->tm_hour, dt->tm_min, dt->tm_sec);
	trunk->setSymbol(SYM_CREATED, buffer);
	ann = getAnnotation();
	if(ann)
		trunk->setSymbol(SYM_ANNOTATION, ann);
	
	switch(getEncoding())
	{
	case G721_ADPCM_ENCODING:
	case OKI_ADPCM_ENCODING:
		codec = VPB_OKIADPCM;
		bufsize = 80;
		samples = 160;
		break;
	case G723_3BIT_ENCODING:
		codec = VPB_OKIADPCM24;
		samples = 160;
		bufsize = 60;
		break;
	case PCM16_AUDIO_ENCODING:
		codec = VPB_LINEAR;
		bufsize = 320;
		samples = 160;
		break;
	case MULAW_AUDIO_ENCODING:
		codec = VPB_MULAW;
		bufsize = 160;
		samples = 160;
		break;
	case ALAW_AUDIO_ENCODING:
		codec = VPB_ALAW;
		bufsize = 160;
		samples = 160;
		break;
	default:
		slog(SLOG_ERROR) << buffer << ": unsupported codec required" << endl;
		Failure();
	}

	reset = true;
	vpb_play_buf_start(handle, codec);
}

void VPBPlay::Run(void)
{
	char name[33];
	char buffer[bufsize];
	audioerror_t status = AUDIO_SUCCESS;
	int len = 0;

	trunk->getName(name);

	while(status == AUDIO_SUCCESS && !stopped)
	{	
		status = getSamples(buffer, samples);
		if(status == AUDIO_READ_INCOMPLETE || status == AUDIO_SUCCESS)
			if(vpb_play_buf_sync(handle, buffer, bufsize) != VPB_OK)
				status = AUDIO_WRITE_INCOMPLETE;
	}
	if(status == AUDIO_READ_FAILURE)
		slog(SLOG_ERROR) << name << ": failed playback" << endl;
	vpb_play_buf_finish(handle);
	reset = false;
	Success();
}

bool VPBTrunk::playHandler(TrunkEvent *event)
{
	switch(event->id)
	{
	case TRUNK_SERVICE_SUCCESS:
		TrunkSignal(TRUNK_SIGNAL_STEP);
		handler = &VPBTrunk::stepHandler;
		return true;
	case TRUNK_SERVICE_FAILURE:
		setSymbol(SYM_ERROR, "play-failed");
		TrunkSignal(TRUNK_SIGNAL_ERROR);
		handler = &VPBTrunk::stepHandler;
		return true;
	case TRUNK_ENTER_STATE:
		debug->DebugState(this, "play");
		flags.dsp = DSP_MODE_VOICE;
		status[id] = 'p';
		if(!flags.offhook)
		{
			flags.offhook = true;
			vpb_sethook_async(handle, VPB_OFFHOOK);
			setTimer(getPickupTimer());
			return true;
		}
	case TRUNK_TIMER_EXPIRED:
		setDTMFDetect();
		thread = new VPBPlay(this, handle);
		thread->Start();
		return true;
	}
	return false;
}

