// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with the Aculab telephony libraries to produce a executable image
// without requiring Aculab's sources to be supplied in a free software
// license long as each source file so linked contains this exclusion
// and the unalrtered Aculab source files are also provided.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include "driver.h"

#define	EVENT_BUFFER_SIZE	16

char AculabTrunk::status[MAXPORT * MAXTIMESLOTS];

AculabTrunk::AculabTrunk(int p, int t) :
Trunk(p * MAXTIMESLOTS + t), Thread(NULL, 0)
{
	unsigned long opts;
	struct in_xparms in_xparms;

	handler = NULL;
	lastring = 0;
	port = p;
	ts = t;

	in_xparms.net = port;
	in_xparms.ts = ts;
	in_xparms.cnf = 0;

	if(call_openin(&in_xparms))
	{
		slog(SLOG_CRITICAL) << "aculab(" << port << "," << "): open failed" << endl;
		throw((Trunk *)this);
	}

	handle = in_xparms.handle;
	lcn = call_handle_2_chan(in_xparms.handle);
	aculabivr.setChannel(this);
	
	pipe(evbuf);
	opts = fcntl(evbuf[1], F_GETFL);
	fcntl(evbuf[1], F_SETFL, opts | O_NONBLOCK);
}

AculabTrunk::~AculabTrunk()
{
	Terminate();
	close(evbuf[0]);
	close(evbuf[1]);
}

void AculabTrunk::Run(void)
{
	char buf[EVENT_BUFFER_SIZE];
	struct pollfd pollfd;
	TrunkEvent *event = (TrunkEvent *)&buf;

	for(;;)
	{
		pollfd.fd = evbuf[0];
		pollfd.events = POLLIN;
		if(poll(&pollfd, 1, getTimer()) < 1)
		{
			endTimer();
			memset(event, 0, sizeof(TrunkEvent));
			event->id = TRUNK_TIMER_EXPIRED;
			postEvent(event);		
			continue;
		}
		if(read(evbuf[0], &buf, sizeof(buf)) != sizeof(buf))
		{
			slog(SLOG_ERROR) << "aculab(" << id << "): event failure in trunk" << endl;
			continue;
		}
		postEvent(event);
	}
}
	
void AculabTrunk::putEvent(TrunkEvent *evt)
{
	char buf[EVENT_BUFFER_SIZE];

	memcpy(buf, evt, sizeof(TrunkEvent));
	if(write(evbuf[1], &buf, sizeof(buf)) != sizeof(buf))
		slog(SLOG_ERROR) << "aculab(" << id << "): event overflow" << endl;
}	

void AculabTrunk::getName(char *buffer)
{
	sprintf(buffer, "aculab(%d,%d)", port, ts);
}

void AculabTrunk::Exit(void)
{
	// handler = &AculabTrunk::hangupHandler;
}

bool AculabTrunk::postEvent(TrunkEvent *event)
{
	bool rtn = true;
	trunkhandler_t prior;

	EnterMutex();
	switch(event->id)
	{
	case TRUNK_TIMER_EXPIRED:
		if(!getTimer())
			rtn = false;

		break;
	case TRUNK_DTMF_KEYUP:
		if(!flags.dtmf)
			rtn = false;
		break;
	}
	if(!rtn)
	{
		LeaveMutex();
		return false;
	}

	if(!handler)
	{
		slog(SLOG_WARNING) << "aculab(" << port << "," << ts << ")";
		slog() << ": no handler active; event=" << event->id << endl;
		LeaveMutex();
		return false;
	}

retry:
	debug->DebugEvent(this, event);
	prior = handler;
	rtn = (this->*handler)(event);
	if(rtn)
	{
		if(handler != prior)
		{
			event->id = TRUNK_ENTER_STATE;
			goto retry;
		}
		LeaveMutex();
		return true;
	}

	// default handler

	rtn = true;
	switch(event->id)
	{
	case TRUNK_RINGING_ON:
	case TRUNK_CALL_DETECT:
		++rings;
		break;
	case TRUNK_ENTER_STATE:
		if(flags.offhook)
			setDTMFDetect();
		else
			setDTMFDetect(false);
		endTimer();
		break;
	case TRUNK_CPA_DIALTONE:
	case TRUNK_LINE_WINK:
	case TRUNK_CALL_RELEASE:
	case TRUNK_STOP_DISCONNECT:
		TrunkSignal(TRUNK_SIGNAL_HANGUP);
		event->id = TRUNK_STOP_STATE;
		goto retry;
		break;
	case TRUNK_TIMER_EXPIRED:
		TrunkSignal(TRUNK_SIGNAL_TIMEOUT);
		event->id = TRUNK_STOP_STATE;
		goto retry;
		break;
	case TRUNK_DTMF_KEYUP:
		if(digits < 32)
			dtmf.bin.data[digits++] = digit[event->parm.dtmf.digit];
		dtmf.bin.data[digits] = 0;
		if(TrunkSignal((trunksignal_t)(event->parm.dtmf.digit + TRUNK_SIGNAL_0)))
		{
			event->id = TRUNK_STOP_STATE;
			goto retry;
		}
		break;
	case TRUNK_EXIT_SHELL:
		tgi.pid = 0;
		break;
	case TRUNK_STOP_STATE:
		endTimer();
//		handler = &AculabTrunk::stepHandler;
		break;
	case TRUNK_EXIT_STATE:
		break;
	case TRUNK_MAKE_BUSY:
//		handler = &AculabTrunk::busyHandler;
		break;
	case TRUNK_MAKE_IDLE:
//		handler = &AculabTrunk::idleHandler;
		break;
	default:
		rtn = false;
	}
	if(handler != prior)
	{
		event->id = TRUNK_ENTER_STATE;
		goto retry;
	}
	LeaveMutex();
	return rtn;
}

unsigned long AculabTrunk::getIdleTime(void)
{
	time_t now;

	time(&now);
//	if(handler == &AculabTrunk::idleHandler)
//		return now - idle;

	return 0;
}

