// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with  the Pika MonteCarlo libraries to produce a executable image
// without requiring MonteCarlo itself to be supplied in source form so
// long as each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include "ivrconfig.h"
#include <ctype.h>

#define	BAYONNE_VERSION	VERSION
#define	BAYONNE_SERVER	PACKAGE

#include "bayonne.h"

unsigned funid(char *sym)
{
	unsigned id = 0;

	while(*sym)
		id = (id << 1) ^ (*(sym++) & 0x1f);

	return id % KEYWORD_INDEX_SIZE;
}

aaScript::aaScript() :
ScriptCommand("/bayonne/script")
{
	// define primary Bayonne dialect
	static SCRKEYWORDS interp[] = {
		{"hangup", (scriptmethod_t)&Trunk::scrHangup, &ScriptCommand::chkNoArgs},
		{"function", (scriptmethod_t)&Trunk::scrFunction, &ScriptCommand::chkHasArgs},
		{"alog", (scriptmethod_t)&Trunk::scrAlog, &ScriptCommand::chkHasArgs},
		{"cdr", (scriptmethod_t)&Trunk::scrAudit, &ScriptCommand::chkHasArgs},
		{"import", (scriptmethod_t)&Trunk::scrImport, &ScriptCommand::chkHasArgs},
		{"export", (scriptmethod_t)&Trunk::scrExport, &ScriptCommand::chkHasArgs},
		{"debug", (scriptmethod_t)&Trunk::scrDebug, &ScriptCommand::chkHasArgs},
		{"sleep", (scriptmethod_t)&Trunk::scrSleep, &ScriptCommand::chkHasArgs},
		{"wait", (scriptmethod_t)&Trunk::scrWait, &ScriptCommand::chkHasArgs},
		{"answer", (scriptmethod_t)&Trunk::scrAnswer, &ScriptCommand::chkIgnore},
		{"collect", (scriptmethod_t)&Trunk::scrCollect, &ScriptCommand::chkHasArgs},
		{"flash", (scriptmethod_t)&Trunk::scrCollect, &ScriptCommand::chkIgnore},
		{"play", (scriptmethod_t)&Trunk::scrPlay, &ScriptCommand::chkHasArgs},
		{"record", (scriptmethod_t)&Trunk::scrRecord, &ScriptCommand::chkHasArgs},
		{"append", (scriptmethod_t)&Trunk::scrAppend, &ScriptCommand::chkHasArgs},
		{"tone", (scriptmethod_t)&Trunk::scrTone, &ScriptCommand::chkHasArgs},
		{"dial", (scriptmethod_t)&Trunk::scrDial, &ScriptCommand::chkHasArgs},
		{"transfer", (scriptmethod_t)&Trunk::scrTransfer, &ScriptCommand::chkHasArgs},
		{"pause", (scriptmethod_t)&Trunk::scrPause, &ScriptCommand::chkHasArgs},
		{"libexec", (scriptmethod_t)&Trunk::scrLibexec, &ScriptCommand::chkHasArgs},
		{"pack", (scriptmethod_t)&Trunk::scrPack, &ScriptCommand::chkHasArgs},
		{"unpack", (scriptmethod_t)&Trunk::scrUnpack, &ScriptCommand::chkHasArgs},
		{"speak", (scriptmethod_t)&Trunk::scrSpeak, &ScriptCommand::chkHasArgs},
		{"lookup", (scriptmethod_t)&Trunk::scrLookup, &ScriptCommand::chkHasArgs},
		{"retry", (scriptmethod_t)&Trunk::scrRetry, &ScriptCommand::chkHasArgs},
		{"login", (scriptmethod_t)&Trunk::scrLogin, &ScriptCommand::chkHasArgs},
		{"logout", (scriptmethod_t)&Trunk::scrLogout, &ScriptCommand::chkNoArgs},
		{"submit", (scriptmethod_t)&Trunk::scrSubmit, &ScriptCommand::chkHasArgs},
		{"session", (scriptmethod_t)&Trunk::scrSession, &ScriptCommand::chkHasArgs},
		{NULL, NULL, NULL}};

	// pre-load some useful compiler "$constants", some of which
	// may be overriden in the [script] area.  These are used not
	// only to configure default behavior, but also to define
	// operating country and dialing plan requirements.  The
	// dialing plan options shown here is based on a centrex style
	// PBX operating in the US.

	static KEYDEF keys[] = {
		{"timeout", "60"},
		{"interdigit", "6"},
		{"extdigits", "4"},
		{"localdigits", "7"},
		{"natdigits", "10"},
		{"xferextension", "FW"},
		{"dialextension", "W"},
		{"diallocal", "W9"},
		{"xferlocal", "FW9"},
		{"dialnational", "W81"},
		{"xfernational", "FW81"},
		{"dialinternational", "W8011"},
		{"xferinternational", "FW8011"},
		{"localprefix", "732"},
		{"dialoperator", "0"},
		{"fastpickup", "2"},
		{"slowpickup", "5"},
		{"country", "1"},
		{"ringtime", "6"},
		{"language", "english"},
		{"extension", ".au"},
		{"voice", "UsEngM"},
		{NULL, NULL}};

	SCRKEYWORDS keywords[2];
	char **keylist = NULL;

	Keydata::Load("~bayonne/script");
	setValue("node", keyserver.getNode());
	setValue("version", BAYONNE_VERSION);
	setValue("server", BAYONNE_SERVER);
	setValue("driver", plugins.getDriverName());
	Keydata::Load(keys);

	memset(functions, 0, sizeof(functions));

	// set standard aascript trap identifiers

	Trap("timeout");
	Trap("dtmf");
	Trap("0");
	Trap("1");
	Trap("2");
	Trap("3");
	Trap("4");
	Trap("5");
	Trap("6");
	Trap("7");
	Trap("8");
	Trap("9");
	Trap("star");
	Trap("pound");	
	Trap("a");
	Trap("b");
	Trap("c");
	Trap("d");
	Trap("dialtone");
	Trap("busytone");
	Trap("cancel");

	Load(interp);

	if(database)
		keylist = database->getKeywords();

	if(!keylist)
		return;

	while(*keylist)
	{
		keywords[0].keyword = *keylist;
		keywords[0].method = (scriptmethod_t)&Trunk::scrSession;
		keywords[0].check = &ScriptCommand::chkHasArgs;
		keywords[1].keyword = NULL;
		keywords[1].method = (scriptmethod_t)NULL;
		keywords[1].check = (scriptcheck_t)NULL;
		Load(keywords);
		++keylist;
	}
}

functioncall_t aaScript::getFunction(char *name)
{
	functions_t *fun = functions[funid(name)];
	
	while(fun)
	{
		if(!stricmp(fun->name, name))
			return fun->function;
		fun = fun->next;
	}
	return (functioncall_t)NULL;
}

void aaScript::addFunction(char *name, functioncall_t function)
{
	char *p;
	unsigned id = funid(name);
	functions_t *fun = (functions_t *)alloc(sizeof(functions_t) + strlen(name));
	fun->function = function;
	p = fun->name;
	while(*name)
		*(p++) = tolower(*(name++));
	*p = 0;
	fun->next = functions[id];
	functions[id] = fun;
}

unsigned long aaScript::getTrapMask(const char *trapname)
{
	unsigned long mask;
	
	if(!strcmp(trapname, "hangup"))
		return 0x00000001;

	if(!strcmp(trapname, "!"))
		return 0x0000fff8;

	mask = ScriptCommand::getTrapMask(trapname);
	if(mask == 0x00000008)
		return 0x0000fff8;

	if(mask & 0x0000fff0)
		mask |= 0x00000008;

	return mask;
}

aaImage::aaImage(aaScript *script) :
ScriptImage((ScriptCommand *)script)
{
	char path[1024];
	char *pp = getenv("HOME");

	strcpy(path, keypaths.getScriptFiles());
	if(canAccess(path))
		scanDir(path);

	if(!pp)
		return;

	strcpy(path, pp);
	strcat(path, ".bayonne");
	if(isDir(path))
		if(canAccess(path))
			scanDir(path);

	Commit();
};

void aaImage::scanDir(char *path)
{
	char *scrname;
	char *pp = path + strlen(path);
	Dir dir(path);

	*(pp++) = '/';
	for(;;)
	{
		scrname = dir.getName();
		if(!scrname)
			break;
		if(!isScript(scrname))
			continue;

		strcpy(pp, scrname);
		Compile(path);
	}
}

bool aaImage::isScript(char *scrname)
{
	char *ext = strrchr(scrname, '.');
	if(!ext)
		return false;

	if(!stricmp(ext, ".scr"))
		return true;

	return false;
}
