/*
    blocks.c  ---  routines responsible for creating blocks which 
    comprise the cube, and manipulating them, and querying their 
    status.
    Copyright (C) 1998, 2003  John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
static const char RCSID[]="$Id: blocks.c,v 1.5 2003/10/26 04:32:51 jmd Exp $";

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <assert.h>

#include "txfm.h"
#include "blocks.h"
#include "gnubik.h"

static int CUBE_DIM, BLOCK_QTY;


static Matrix *block_transformation;
static point *block_location;
static int perimeter ;

static char error_msg[132];
#define SET_ERR(MSG) ( strcpy(error_msg,MSG) )


struct FaceAttributes {
  /* An array of vectors showing the orientation of the blocks. 
     There are two vectors per face.  These are used to orientate the
     mouse pointer, to detect when the cube has been solved and to
     provide feedback to clients querying the state. 
  */
  vector quadrants[4];

  /* The normal vector is orthogonal to the face of the block. 
   When all normals are in the same direction, the cube colours are correct, 
   but not necessarily their orientations */
  vector normal;

  /* The position of the centre of the face */
  point centre;
};

struct BlockAttributes {

  /* Unique id for this block */
  int id;               

  /* Which faces are on the surface of the cube, and should therefore 
     be rendered to the framebuffer. */
  int visible_faces;    

  struct FaceAttributes face[6];
};

static struct BlockAttributes *block_attributes =0;

static void set_face_centre(int block, int face, const point p) ;

int 
get_visible_faces(int id)
{
  return block_attributes[id].visible_faces;
}

/* Cube co-ordinates have their origin at the centre of the
cube, and their units are equivalent to one half of the length
of one edge of a block */


/* Return x^n (integral values ) */
long
ipow(int x, int n)
{
  int i;
  long result=1;
  
  for (i = 0 ; i < n ; ++i ) 
    result *=x;

  return result;
    
}

/* Get the transformation of block number  `block_id' from the origin, and
store it in transform.
Return  0 on success, -1 on error */
int
get_block_transform(int block_id, Matrix transform)
{
	int i;

	if ( block_id < 0 || block_id >= BLOCK_QTY) {
		SET_ERR("Reference to invalid block");
		return(-1);
	}

	for( i = 0 ; i < MATRIX_DIM * MATRIX_DIM ; i++)
		transform[i] = block_transformation[block_id][i];

	return 0;
}


/* return the identity of the block at location x,y,z (cube co-ordinates) 
Returns -1 on error */
int
block_at_location(int x, int y, int z)
{
	int i ;

	int odd_dim = (CUBE_DIM % 2) ;

	if ( abs(x) != CUBE_DIM -1 && abs(y) != CUBE_DIM -1 && abs(z) != CUBE_DIM -1 ) {
		SET_ERR("Reference to block not on cube perimeter");
		return(-1);
	}

	if ( abs(x) % 2 == odd_dim  || abs(y) % 2 == odd_dim  || abs(z) % 2 == odd_dim  ) {
		SET_ERR("Reference to non aligned block position");
		return(-1);
	}


	/* Search the blocks for the one we're looking for.
	Some kind of hashing scheme would be better, but this is a small list, 
	and a non time critical application */
	for(i=0; i < BLOCK_QTY ; i++) {
		if( x == block_location[i][0] &&  y == block_location[i][1]
				 &&  z == block_location[i][2] )
			return i ;
	}
		
	
	return -1;
}
	







/* This func initialises the positions of the blocks which comprise the cube.  
The enumeration scheme I have chosen goes around four surfaces of the cube, 
then fills in the ends.  Thus, for a 4x4x4 cube it looks like:

----------------------------------------------------------------------------
View this diagram with 132 coloumns!

                                                                                       |  60 |  61 |  62 |  63 |
                                                          |  44 |  45 |  46 |  47 |    |  56 |  57 |  58 |  59 |
                             |  28 |  29 |  30 |  31 |    |  40 |  41 |  42 |  43 |    |  52 |  53 |  54 |  55 |
|  12 |  13 |  14 |  15 |    |  24 |  25 |  26 |  27 |    |  36 |  37 |  38 |  39 |    |  48 |  49 |  50 |  51 |
|   8 |   9 |  10 |  11 |    |  20 |  21 |  22 |  23 |    |  32 |  33 |  34 |  35 |
|   4 |   5 |   6 |   7 |    |  16 |  17 |  18 |  19 |
|   0 |   1 |   2 |   3 |

cube_dimension, is the size of the cube to be created.
The function places the number of blocks created into block_qty,
and returns that also, or 0 on error
*/
int 
create_blocks(int cube_dimension, int *block_qty)
{

  int i;
	

  CUBE_DIM = cube_dimension;

  BLOCK_QTY = CUBE_DIM*CUBE_DIM*CUBE_DIM;
  switch ( CUBE_DIM ) {
  case 0:
    perimeter = 0;
    break;
  case 1:
    perimeter = 1;
    break;
  default:
    if ( CUBE_DIM < 0 ) {
      SET_ERR("invalid dimension value");
      return 0 ;
    }
    perimeter = (CUBE_DIM -1 ) * 4;

    break;
  }

  if (BLOCK_QTY > 0 ) {
    if ( NULL == (block_location = (point*) malloc ( BLOCK_QTY * sizeof (point) ) ) ) {
      fprintf(stderr,"Cannot malloc at %s:%d\n",__FILE__,__LINE__);
      exit(-1);
    }
    if ( NULL == (block_transformation = (Matrix*) malloc ( BLOCK_QTY * sizeof (Matrix) ) ) ) {
      fprintf(stderr,"Cannot malloc at %s:%d\n",__FILE__,__LINE__);
      exit(-1);
    }
    if ( NULL == ( block_attributes =  (struct BlockAttributes *) malloc( BLOCK_QTY * sizeof (struct  BlockAttributes) ) ) ){
      fprintf(stderr,"Cannot malloc at %s:%d\n",__FILE__,__LINE__);
      exit(-1);
    }
  }

  for( i = 0 ; i < BLOCK_QTY ; i++) {
    int k; 
    /* initialise all translations to the identity matrix */
    for(k=0 ; k < MATRIX_DIM ; k++ ) {
      int j;
      for(j=0;j < MATRIX_DIM ; j++)
	block_transformation[i][k+ MATRIX_DIM*j] = (k==j) ? 1: 0;
    }
  }

  /* Now attend to the attributes */
  for( i = 0 ; i < BLOCK_QTY ; i++) {
    block_attributes[i].visible_faces = 0;
    block_attributes[i].id = i;
  }


#if DEBUG
  /* Fill them all with -99 so I can see what's going on */
  for( i = 0 ; i < BLOCK_QTY ; i++) {
    int j;
    for(j=0;j < MATRIX_DIM ; j++)
      block_location[i][j]= -99;
  }
#endif


  /* flaging only certain faces as visible, allows us to avoid rendering 
     invisible surfaces, and thus slowing down animation */
	
  for(i=0; i < BLOCK_QTY ; i++) {

    if ( 0 ==          ( i / (CUBE_DIM*CUBE_DIM) )  ) 
      block_attributes[i].visible_faces |= FACE_0 ;

    if ( CUBE_DIM-1 == ( i / (CUBE_DIM*CUBE_DIM) )  ) 
      block_attributes[i].visible_faces |= FACE_1 ;

    if ( 0 ==          ( i / (CUBE_DIM) % (CUBE_DIM)))
      block_attributes[i].visible_faces |= FACE_2 ;

    if ( CUBE_DIM-1 == ( i / (CUBE_DIM) % (CUBE_DIM)))
      block_attributes[i].visible_faces |= FACE_3 ;

    if ( 0 ==          ( i% (CUBE_DIM)))
      block_attributes[i].visible_faces |= FACE_4 ;

    if ( CUBE_DIM-1 == ( i% (CUBE_DIM)))
      block_attributes[i].visible_faces |= FACE_5 ;

  }

  /* Set the location of all the block numbers */

  for(i=0; i < BLOCK_QTY ; i++) {
    int j ;
    for ( j = 0 ; j < 3 ; ++j ) { 
      block_location[i][j] = (i / ipow(CUBE_DIM,j) )  % CUBE_DIM ;
    }
  }

  /* Set all the transformation matrices to reflect 
     their transformations from a block sitting at (0,0,0 ) */
  for( i=0; i < BLOCK_QTY; i++) {
    int j;
    for( j=0; j < 3; j++){
      /* Now convert to cube co-ordinates */
      block_location[i][j] *=  2;
      block_location[i][j] -=  CUBE_DIM -1 ;
    }
    /* And these co-ordinates are homogenous, so just indicate that 
       they are points, not vectors */
    block_location[i][3] =  1;

    for(j=0;j < MATRIX_DIM ; j++)
      block_transformation[i][(j+ 12)] = block_location[i][j] ;
		
  }


  /* Set all the face centres */
  for( i = 0 ; i < BLOCK_QTY ; ++i ) { 
    int f;
    point centre ;
    
    centre[3]=1.0;


    for ( f = 0 ; f < 6 ; ++f ) { 

      switch (f) { 
      case 0:
	centre[0]=centre[1]=0;
	centre[2]=-1.0;
	break;
      case 1:
	centre[0]=centre[1]=0;
	centre[2]=+1.0;
	break;
      case 3:
	centre[0]=centre[2]=0;\
	centre[1]=1.0;
	break;
      case 2:
	centre[0]=centre[2]=0;
	centre[1]=-1.0;
	break;
      case 4:
	centre[1]=centre[2]=0;
	centre[0]=-1.0;
	break;
      case 5:
	centre[1]=centre[2]=0;
	centre[0]=1.0;
	break;
      }

      set_face_centre(i,f,centre);
    
    }
  }
	
#if DEBUG
  {
    int j;
    /* and print them out for me */
    for( i = 0 ; i < BLOCK_QTY ; i++) {
      printf("Point %d:\t",i);
      for ( j=0; j < 3 ; j++)
	printf("%2d ",block_location[i][j]);
      printf("\n");
    }
  }
  for(i= -(CUBE_DIM - 1); i <= CUBE_DIM -1 ; i+=2 )
    showSlice(i);

  for( i = 0 ; i < BLOCK_QTY ; i++) {
    printf("Transform %d:\n",i);
    showMatrix(block_transformation[i]);
  }
#endif
	
  *block_qty = BLOCK_QTY ;

  return BLOCK_QTY;
}


/* Actually this function never gets called.  But it is here for completeness
sake */
void destroy_blocks()
{
  free ( block_location );
  free ( block_transformation );
}




/* a couple of simple little funcs to give trig functions for values expressed in multiples of 90 degrees */
static int COS ( int quarters); 
static int SIN ( int quarters);


/* Rotate the slice identified by the block whose centre is at block, and axis, 
about that axis, through a  degree specified by turns, which is in quarters of 
complete revolutions */
int
rotate_slice(int block_id, int axis, int turns)
{
	int i;

	int rotating_block_qty;

	int *blocks_in_this_slice;

	/* Create a matrix describing the rotation of we are about to perform.
	We do this be starting with the identity matrix */
	Matrix rotation = {
		1,0,0,0,
		0,1,0,0,
		0,0,1,0,
		0,0,0,1
	};

	/* and then assigning value to the active elements */
	rotation[(axis+1)%3 + MATRIX_DIM * ((axis+1)%3) ] = COS(turns);
	rotation[(axis+2)%3 + MATRIX_DIM * ((axis+2)%3) ] = COS(turns);
	rotation[(axis+1)%3 + MATRIX_DIM * ((axis+2)%3) ] =  SIN(turns);
	rotation[(axis+2)%3 + MATRIX_DIM * ((axis+1)%3) ] = -SIN(turns);


	if (block_id < 0 || block_id >= BLOCK_QTY ) {
		SET_ERR("Invalid block identity");
		return(-1);
	}
		

	/* get the value of rotating_block_qty. If were rotating a slice in the
	middle of the cube, then there are less blocks in it */
	blocks_in_this_slice  = identify_blocks(block_id, axis, &rotating_block_qty);
	if ( blocks_in_this_slice == NULL ) {
		return(-1);
	}
	


	for( i = 0 ; i < rotating_block_qty; i ++ ) {
		int rotating_block =  blocks_in_this_slice[i] ;

		/* Update the position of each block */
		transform(rotation,block_location[rotating_block] );

		/* and update the matrix representing its position and orientation*/
		pre_mult(rotation,block_transformation[rotating_block]);
	}

	return 0;

} /* end rotate_slice()*/





/* return a pointer to an array of block numbers which are in the slice
identified by block_id, and axis.  This array is dynamically allocated,
and its size is placed in block_qty */
int *
identify_blocks(int block_id, int axis , int *block_qty)
{
  int i,j;
  static int *blocks_in_slice= NULL;
  point block;
  int blocks_in_slice_qty ;

  int odd_dim = (CUBE_DIM % 2) ;

  for(i=0; i< 3 ; i++)
    block[i] = block_location[block_id][i];
  block[3] = 1;
  blocks_in_slice_qty = CUBE_DIM *CUBE_DIM;

  if ( abs(block[0]) % 2 == odd_dim  || abs(block[1]) % 2 == odd_dim  || abs(block[2]) % 2 == odd_dim  ) {
    SET_ERR("Reference to non aligned block position");
    return(NULL);
  }

  /* Free up the memory if allocated */
  if (blocks_in_slice != NULL ) {
    free ( blocks_in_slice);
    blocks_in_slice = NULL;
  }

  /* Now identify all blocks in this slice */
  switch (axis) {
  case 0:
  case 1:
  case 2:
    if ( NULL == ( blocks_in_slice = (int *) malloc( CUBE_DIM*CUBE_DIM * sizeof (int) ) ) ) {
      fprintf(stderr,"Cannot malloc at %s:%d\n",__FILE__,__LINE__);
      exit(-1);
    }
    j=0;
    for (i=0; i < BLOCK_QTY ; i++) {
      if ( block_location[i][axis] == block[axis] ) {
	blocks_in_slice[j++] = i;
      }
    }
    if( j != blocks_in_slice_qty ) {
      SET_ERR("Block quantity mismatch");
      return(NULL);
    }
    break;
  default:
    SET_ERR("invalid axis");
    return(NULL);
  }

#if DEBUG
  printf("Blocks within slice (total %d) :",blocks_in_slice_qty);
  plot(block);
  printf("axis %d \n", axis);
  for( i = 0 ; i < blocks_in_slice_qty ; i++ ) {
    printf("#%d ", blocks_in_slice[i]);
  }
  putchar('\n');
#endif

  *block_qty = blocks_in_slice_qty;

  return blocks_in_slice;

}  /* identify_blocks() */


static int
COS ( int quarters)
/* Crude cosine function for angles expressed in quarters of complete 
revolutions */
{
	if ( quarters < 0 )
		quarters = -quarters;

	switch ( quarters % 4 ) {
		case 0:
			return 1;
		case 2:
			return -1;
		case 1:
		case 3:
			return 0;
	}

	/* Error value */
	return -999;
}

/* Crude sinusoid function, for angles expressed in quarters of complete 
revolutions */
static int
SIN ( int quarters)
{
	int retval;

	switch ( abs(quarters) % 4 ) {
		case 1:
			retval =  1;
			break;
		case 3:
			retval =  -1;
			break;
		case 0:
		case 2:
			retval =  0;
			break;
		default:
			/* Error value */
			retval = -999;
			break;
	}

	if ( quarters < 0 )
		retval = -retval;
		
	return retval;
}


/* This function is used during debugging and development.  It shows a graphic
representation of a z  slice of the cube */
void
showSlice(int slice)
{
	int x,y;

	printf("Slice %d\n",slice);
	for(y=(CUBE_DIM-1); y >= -(CUBE_DIM-1) ; y-=2) {
		printf("|");
		for(x=-(CUBE_DIM-1); x <= CUBE_DIM-1; x+=2) {
			int id = block_at_location(x,y,slice);
			if ( id != -1 )
				printf(" %3d |", id );
			else
				printf("     |");
				
		}
		printf("\n");
	}

}



/* Error printer */
void
bl_err(const char *s)
{
  printf("%s: %s\n",s,error_msg);
}


int
get_number_of_blocks(void)
{
  return BLOCK_QTY;
}



void
showBlock(int block)
{
  int face;



  struct BlockAttributes *attr=block_attributes+block;

  assert(block == attr->id);

  printf("Block %d\n",block);

  for(face = 0 ; face < 6 ; ++face ) { 
    int quadrant;
    unsigned int mask = 0x01 << face;
    if ( mask & attr->visible_faces) { 
      printf("Direction of face %d\n",face);
      for ( quadrant = 0 ; quadrant < 4 ; ++quadrant ) {
	printf("q%d:",quadrant);
	plot(attr->face[face].quadrants[quadrant]);
      }
    }
    else { 
      printf("Face %d not visible \n",face);
    }
  }
}


/* Set the centre point of the face to p */
void
set_face_centre(int block, int face, const point p)
{
  point *dest;

  dest = &block_attributes[block].face[face].centre;

  memcpy(dest,p,sizeof(point));

}

void 
get_face_centre(int block, int face, point p)
{

  memcpy(p,block_attributes[block].face[face].centre,
	 sizeof(vector));

  transform(block_transformation[block],p);

  printf("Face centre %d,%d is \n",block,face);
  plot(p);

  
}


/* Set the normal vector for block/face */
void 
set_normal_vector(int block, int face,const vector v)
{
  Matrix view;
  vector *dest;

  glGetFloatv(GL_MODELVIEW_MATRIX,view);


  dest = &block_attributes[block].face[face].normal;

  memcpy(dest,v,sizeof(vector));

  transform(view,*dest);
  
}

/* Retrieve the normal vector for block/face */
void 
get_normal_vector(int block, int face,vector v)
{
  memcpy(v,block_attributes[block].face[face].normal,
	 sizeof(vector));
  
}


/* Set the vector for block/face/quadrant to v */
void 
set_quadrant_vector(int block, int face, int quadrant, const vector v)
{
  Matrix view;
  vector *dest;

  glGetFloatv(GL_MODELVIEW_MATRIX,view);


  dest = &block_attributes[block].face[face].quadrants[quadrant] ;

  memcpy(dest,v,sizeof(vector));

  transform(view,*dest);

}


/* Return the quadrant vector in v */
void 
get_quadrant_vector(int block, int face, int quadrant, vector v)
{

  memcpy(v,block_attributes[block].face[face].quadrants[quadrant],
	 sizeof(vector));
  
}



int
equal(const vector v1, const vector v2)
{
  int i;
 
  for(i = 0 ; i < MATRIX_DIM ; ++i ) { 
    if ( v1[i] != v2[i] ) 
      return 0;
  }
  return 1;
}


/* The cube is solved iff for all faces the quadrant vectors 
   point in the same direction.
   If however all the normals point in the same dir, but the quadrants do not, 
   then the colours are all on the right faces, but not correctly orientated */
enum cube_status
cubeStatusCheck(void)
{
  int face ; 
  int i;

  for ( face = 0 ; face < 6 ; ++face)
  {
    vector q1;
    vector q0;
    unsigned int mask = 0x01 << face;
    
    int x =0; 

    for ( i = 0 ; i < BLOCK_QTY ; ++i ) { 
      vector *v0 ; 
      vector *v1 ; 

      
      /* Ignore faces which are inside the cube */
      if ( ! ( block_attributes[i].visible_faces & mask ) ) 
	continue;

      v0 = &block_attributes[i].face[face].quadrants[0];
      v1 = &block_attributes[i].face[face].quadrants[1];

      if ( x == 0 ) {
	memcpy(q0,&block_attributes[i].face[face].quadrants[0],sizeof(vector));
	memcpy(q1,&block_attributes[i].face[face].quadrants[1],sizeof(vector));
      }
      ++x;

      if ( ! equal(q0,*v0) ) {
	goto part_check;
      }
      if ( ! equal(q1,*v1) ) {
	goto part_check;
      }

    }
  }
  /* Cube is fully solved */
  return SOLVED;

  /* The cube is not solved, but perhaps the colours are right, 
   * and the orientation is wrong */
 part_check:

  for ( face = 0 ; face < 6 ; ++face)
  {
    vector q0;
    unsigned int mask = 0x01 << face;
    
    int x =0; 

    for ( i = 0 ; i < BLOCK_QTY ; ++i ) { 
      vector *v0 ; 
      
      /* Ignore faces which are inside the cube */
      if ( ! ( block_attributes[i].visible_faces & mask ) ) 
	continue;

      v0 = &block_attributes[i].face[face].normal;


      if ( x == 0 ) {
	memcpy(q0,&block_attributes[i].face[face].normal,sizeof(vector));
      }
      ++x;

      if ( ! equal(q0,*v0) ) {
	return NOT_SOLVED;
      }
    }
  }

  return HALF_SOLVED;
}
