/*
 * Electric(tm) VLSI Design System
 *
 * File: dberror.c
 * Database error handler
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "database.h"
#include "usr.h"

INTBIG db_lasterror;			/* last error message */
INTSML db_printerrors;			/* flag for printing internal errors */

char *db_errortype[] =
{
	N_("no error"),								/* DBNOERROR */
	N_("no memory"),							/* DBNOMEM */
	N_("bad transposition"),					/* DBBADTRANS */
	N_("bad rotation"),							/* DBBADROT */
	N_("bad prototype"),						/* DBBADPROTO */
	N_("bad parent"),							/* DBBADPARENT */
	N_("invalid instance"),						/* DBBADINST */
	N_("invalid name"),							/* DBBADNAME */
	N_("bad width"),							/* DBBADWIDTH */
	N_("bad end A node/port"),					/* DBBADENDAN */
	N_("bad end B node/port"),					/* DBBADENDBN */
	N_("bad end A connection"),					/* DBBADENDAC */
	N_("bad end B connection"),					/* DBBADENDBC */
	N_("bad end A position"),					/* DBBADENDAP */
	N_("bad end B position"),					/* DBBADENDBP */
	N_("bad new width"),						/* DBBADNEWWID */
	N_("bad facet"),							/* DBBADFACET */
	N_("bad library"),							/* DBBADLIB */
	N_("bad size"),								/* DBBADSIZE */
	N_("bad object type"),						/* DBBADOBJECT */
	N_("bad sub port"),							/* DBBADSUBPORT */
	N_("still has arcs"),						/* DBHASARCS */
	N_("still has ports"),						/* DBHASPORTS */
	N_("facet has instances"),					/* DBHASINSTANCES */
	N_("recursive call"),						/* DBRECURSIVE */
	N_("arc not in port"),						/* DBNOTINPORT */
	N_("conflicts with primitive"),				/* DBCONFLICT */
	N_("port mismatch"),						/* DBPORTMM */
	N_("duplicate name"),						/* DBDUPLICATE */
	N_("primitive prototype"),					/* DBPRIMITIVE */
	N_("bad transformation matrix"),			/* DBBADTMAT */
	N_("variable does not exist"),				/* DBNOVAR */
	N_("variable cannot be changed"),			/* DBVARFIXED */
	N_("variable cannot be displayable array"),	/* DBVARARRDIS */
	N_("this is the last technology"),			/* DBLASTECH */
	N_("technology is still in use"),			/* DBTECINUSE */
	N_("sliding not allowed")					/* DBNOSLIDING */
};

/*
 * database error routine codes
 */
char *db_routines[] =
{
	"addstringtoinfstr",	/*  1 */
	"addtechnology",		/*  2 */
	"addtoinfstr",			/*  3 */
	"allocarcinst",			/*  4 */
	"alloccell",			/*  5 */
	"allocgeom",			/*  6 */
	"alloclibrary",			/*  7 */
	"allocnodeinst",		/*  8 */
	"allocnodeproto",		/*  9 */
	"allocpolygon",			/* 10 */
	"allocportarcinst",		/* 11 */
	"allocportexpinst",		/* 12 */
	"allocportproto",		/* 13 */
	"allocstring",			/* 14 */
	"alloctechnology",		/* 15 */
	"allocview",			/* 16 */
	"copynodeproto",		/* 17 */
	"delind",				/* 18 */
	"delindkey",			/* 19 */
	"delvalkey",			/* 20 */
	"describevariable",		/* 21 */
	"extendpolygon",		/* 22 */
	"initinfstr",			/* 23 */
	"initobjlist",			/* 24 */
	"insind",				/* 25 */
	"insindkey",			/* 26 */
	"killarcinst",			/* 27 */
	"killlibrary",			/* 28 */
	"killnodeinst",			/* 29 */
	"killnodeproto",		/* 30 */
	"killportproto",		/* 31 */
	"killtechnology",		/* 32 */
	"makekey",				/* 33 */
	"modifyarcinst",		/* 34 */
	"moveportproto",		/* 35 */
	"newarcinst",			/* 36 */
	"newlibrary",			/* 37 */
	"newnodeinst",			/* 38 */
	"newnodeproto",			/* 39 */
	"newportproto",			/* 40 */
	"newview",				/* 41 */
	"replacearcinst",		/* 42 */
	"replacenodeinst",		/* 43 */
	"returninfstr",			/* 44 */
	"setind",				/* 45 */
	"setindkey",			/* 46 */
	"setval",				/* 47 */
	"setvalkey",			/* 48 */
	"transmult",			/* 49 */
	"xform",				/* 50 */
};

/*
 * routine to report a database error
 */
INTBIG db_error(INTBIG code)
{
	db_lasterror = code;
	if (db_printerrors != 0) telldatabaseerror();
	return(-1);
}

/*
 * routine to print an error message
 */
void telldatabaseerror(void)
{
	REGISTER INTSML routine, error;

	routine = (db_lasterror >> 16) - 1;
	error = db_lasterror & 0xFFFF;
	if (error == DBNOERROR) ttyputmsg(_("No error")); else
		ttyputmsg("%s: %s", db_routines[routine], _(db_errortype[error]));
}

/******************** ERROR REPORTING ********************/

/*
 * These are the routines to log errors:
 *   initerrorlogging(char *s)                   initialize for subsystem "s"
 *   e = logerror(char *msg, NODEPROTO *f, k)    log message "msg" in facet "f", sort key "k"
 *   addgeomtoerror(void *e, GEOM *g, INTSML s)  add geom "g" to error "e" (show if "s" nonzero)
 *   addlinetoerror(void *e, x1, y1, x2, y2)     add line to error "e"
 *   addpolytoerror(void *e, POLYGON *p)         add polygon to error "e"
 *   addpointtoerror(void *e, x, y)              add point to error "e"
 * To report errors, call:
 *   termerrorlogging()                          complete error accumulation
 *   sorterrors()                                sort errors by key
 *   n = numerrors()                             returns number of errors
 *   s = reportnexterror(show, &g1, &g2)         report next error
 *   s = reportpreverror()                       report previous error
 *   s = reportcurrenterror(show, &g1, &g2)      report current error
 */

typedef enum {ERRORTYPEGEOM, ERRORTYPEFACET, ERRORTYPELINE, ERRORTYPEPOINT} ERRORHIGHLIGHTTYPE;

#define NOERRORHIGHLIGHT ((ERRORHIGHLIGHT *)-1)

typedef struct
{
	ERRORHIGHLIGHTTYPE type;
	GEOM              *geom;
	INTBIG             showgeom;
	INTBIG             x1, y1;
	INTBIG             x2, y2;
} ERRORHIGHLIGHT;


#define NOERRORLIST ((ERRORLIST *)-1)

typedef struct iErrorList
{
	char              *message;
	NODEPROTO         *facet;
	INTBIG             sortkey;
	INTBIG             numhighlights;
	ERRORHIGHLIGHT   **highlights;
	struct iErrorList *preverrorlist;
	struct iErrorList *nexterrorlist;
} ERRORLIST;

ERRORLIST *db_errorlistfree = NOERRORLIST;
ERRORLIST *db_firsterrorlist = NOERRORLIST;
ERRORLIST *db_curerrorlist = NOERRORLIST;
ERRORLIST *db_nexterrorlist = NOERRORLIST;
ERRORLIST *db_preverrorlist = NOERRORLIST;
char       db_errorsystem[100] = "";

INTSML db_addtoerrorlist(ERRORLIST *el, ERRORHIGHLIGHT *eh);

/*
 * Routine to free all previously stored errors and initialize the system.
 */
void initerrorlogging(char *system)
{
	REGISTER ERRORLIST *el;
	REGISTER INTBIG i;

	while (db_firsterrorlist != NOERRORLIST)
	{
		el = db_firsterrorlist;
		db_firsterrorlist = el->nexterrorlist;
		efree(el->message);
		for(i=0; i<el->numhighlights; i++)
			efree((char *)el->highlights[i]);
		if (el->numhighlights > 0) efree((char *)el->highlights);
		efree((char *)el);
	}
	db_curerrorlist = NOERRORLIST;
	db_nexterrorlist = NOERRORLIST;
	db_preverrorlist = NOERRORLIST;
	strncpy(db_errorsystem, system, 100);
}

/*
 * Routine to create an error message with the text "message" applying to facet "facet".
 * Returns a pointer to the message (0 on error) which can be used to add highlights.
 */
void *logerror(char *message, NODEPROTO *facet, INTBIG sortkey)
{
	REGISTER ERRORLIST *el;

	if (db_errorlistfree != NOERRORLIST)
	{
		el = db_errorlistfree;
		db_errorlistfree = el->nexterrorlist;
	} else
	{
		el = (ERRORLIST *)emalloc(sizeof (ERRORLIST), db_cluster);
		if (el == 0) return(0);
	}
	if (allocstring(&el->message, message, db_cluster) != 0) return(0);
	el->facet = facet;
	el->sortkey = sortkey;
	el->numhighlights = 0;
	el->preverrorlist = NOERRORLIST;
	el->nexterrorlist = db_firsterrorlist;
	if (db_firsterrorlist != NOERRORLIST) db_firsterrorlist->preverrorlist = el;
	db_firsterrorlist = el;
	return((void *)el);
}

/*
 * Routine to add "geom" to the error in "errorlist".  Returns nonzero on error.
 */
INTSML addgeomtoerror(void *errorlist, GEOM *geom, INTSML showit)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;

	el = (ERRORLIST *)errorlist;
	eh = (ERRORHIGHLIGHT *)emalloc(sizeof (ERRORHIGHLIGHT), db_cluster);
	if (eh == 0) return(1);
	eh->type = ERRORTYPEGEOM;
	eh->geom = geom;
	eh->showgeom = showit;
	if (db_addtoerrorlist(el, eh) != 0) return(1);
	return(0);	
}

/*
 * Routine to add facet "facet" to the error in "errorlist".  Returns nonzero on error.
 */
INTSML addfacettoerror(void *errorlist, NODEPROTO *facet)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;

	el = (ERRORLIST *)errorlist;
	eh = (ERRORHIGHLIGHT *)emalloc(sizeof (ERRORHIGHLIGHT), db_cluster);
	if (eh == 0) return(1);
	eh->type = ERRORTYPEFACET;
	eh->geom = (GEOM *)facet;
	if (db_addtoerrorlist(el, eh) != 0) return(1);
	return(0);	
}

/*
 * Routine to add line (x1,y1)=>(x2,y2) to the error in "errorlist".
 * Returns nonzero on error.
 */
INTSML addlinetoerror(void *errorlist, INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;

	el = (ERRORLIST *)errorlist;
	eh = (ERRORHIGHLIGHT *)emalloc(sizeof (ERRORHIGHLIGHT), db_cluster);
	if (eh == 0) return(1);
	eh->type = ERRORTYPELINE;
	eh->x1 = x1;
	eh->y1 = y1;
	eh->x2 = x2;
	eh->y2 = y2;
	if (db_addtoerrorlist(el, eh) != 0) return(1);
	return(0);	
}
/*
 * Routine to add polygon "poly" to the error in "errorlist".
 * Returns nonzero on error.
 */
INTSML addpolytoerror(void *errorlist, POLYGON *poly)
{
	REGISTER INTBIG i, prev;
	INTBIG lx, hx, ly, hy;

	if (isbox(poly, &lx, &hx, &ly, &hy) != 0)
	{
		if (addlinetoerror(errorlist, lx, ly, lx, hy) != 0) return(1);
		if (addlinetoerror(errorlist, lx, hy, hx, hy) != 0) return(1);
		if (addlinetoerror(errorlist, hx, hy, hx, ly) != 0) return(1);
		if (addlinetoerror(errorlist, hx, ly, lx, ly) != 0) return(1);
	} else
	{
		for(i=0; i<poly->count; i++)
		{
			if (i == 0) prev = poly->count-1; else prev = i-1;
			if (addlinetoerror(errorlist, poly->xv[prev], poly->yv[prev],
				poly->xv[i], poly->yv[i]) != 0) return(1);
		}
	}
	return(0);	
}

/*
 * Routine to add point (x,y) to the error in "errorlist".
 * Returns nonzero on error.
 */
INTSML addpointtoerror(void *errorlist, INTBIG x, INTBIG y)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;

	el = (ERRORLIST *)errorlist;
	eh = (ERRORHIGHLIGHT *)emalloc(sizeof (ERRORHIGHLIGHT), db_cluster);
	if (eh == 0) return(1);
	eh->type = ERRORTYPEPOINT;
	eh->x1 = x;
	eh->y1 = y;
	if (db_addtoerrorlist(el, eh) != 0) return(1);
	return(0);	
}

INTSML db_addtoerrorlist(ERRORLIST *el, ERRORHIGHLIGHT *eh)
{
	REGISTER ERRORHIGHLIGHT **neweh;
	REGISTER INTBIG i;

	neweh = (ERRORHIGHLIGHT **)emalloc((sizeof (ERRORHIGHLIGHT *)) * (el->numhighlights+1),
		db_cluster);
	if (neweh == 0) return(1);
	for(i=0; i<el->numhighlights; i++)
		neweh[i] = el->highlights[i];
	neweh[el->numhighlights] = eh;
	if (el->numhighlights > 0) efree((char *)el->highlights);
	el->highlights = neweh;
	el->numhighlights++;
	return(0);
}

/*
 * Routine called when all errors are logged.  Initializes pointers for replay of errors.
 */
void termerrorlogging(void)
{
	REGISTER INTBIG errs, olderrs;
	REGISTER VARIABLE *var;

	db_curerrorlist = NOERRORLIST;
	db_nexterrorlist = db_firsterrorlist;
	db_preverrorlist = NOERRORLIST;

	/* save the number of errors in "%V" */
	errs = numerrors();
	var = getval((INTBIG)us_aid, VAID, VINTEGER, "USER_local_capv");
	if (var == NOVARIABLE) olderrs = -1; else
		olderrs = var->addr;
	if (errs != olderrs)
	{
		setval((INTBIG)us_aid, VAID, "USER_local_capv", errs, VINTEGER|VDONTSAVE);
	}
}

void sorterrors(void)
{
	REGISTER ERRORLIST *ep1, *ep2, *prev, *next;
	REGISTER INTBIG sorted;

	sorted = 0;
	while (sorted == 0)
	{
		sorted = 1;
		for(ep1 = db_firsterrorlist; ep1 != NOERRORLIST; ep1 = ep1->nexterrorlist)
		{
			ep2 = ep1->nexterrorlist;
			if (ep2 == NOERRORLIST) continue;

			if (ep1->sortkey <= ep2->sortkey) continue;

			/* reverse the errors */
			prev = ep1->preverrorlist;
			next = ep2->nexterrorlist;
			if (prev == NOERRORLIST)
			{
				db_firsterrorlist = ep2;
				ep2->preverrorlist = NOERRORLIST;
			} else
			{
				prev->nexterrorlist = ep2;
				ep2->preverrorlist = prev;
			}
			ep1->nexterrorlist = next;
			if (next != NOERRORLIST)
			{
				next->preverrorlist = ep1;
			}
			ep1->preverrorlist = ep2;
			ep2->nexterrorlist = ep1;
			ep1 = ep2;
			sorted = 0;
		}
	}
}

/*
 * Routine to return the number of logged errors.
 */
INTBIG numerrors(void)
{
	REGISTER INTBIG errors;
	REGISTER ERRORLIST *el;

	errors = 0;
	for(el = db_firsterrorlist; el != NOERRORLIST; el = el->nexterrorlist)
		errors++;
	return(errors);
}

/*
 * Routine to advance to the next error and report it.
 */
char *reportnexterror(INTBIG showhigh, GEOM **g1, GEOM **g2)
{
	if (db_nexterrorlist == NOERRORLIST)
	{
		(void)initinfstr();
		(void)formatinfstr(_("No more %s errors"), db_errorsystem);
		return(returninfstr());
	}
	db_curerrorlist = db_nexterrorlist;
	db_nexterrorlist = db_curerrorlist->nexterrorlist;
	db_preverrorlist = db_curerrorlist->preverrorlist;
	return(reportcurrenterror(showhigh, g1, g2));
}

/*
 * Routine to back up to the previous error and report it.
 */
char *reportpreverror(void)
{
	if (db_preverrorlist == NOERRORLIST)
	{
		(void)initinfstr();
		(void)formatinfstr(_("No more %s errors"), db_errorsystem);
		return(returninfstr());
	}
	db_curerrorlist = db_preverrorlist;
	db_nexterrorlist = db_curerrorlist->nexterrorlist;
	db_preverrorlist = db_curerrorlist->preverrorlist;
	return(reportcurrenterror(1, 0, 0));
}

/*
 * Routine to return the error message associated with the current error.
 * Highlights associated graphics if "showhigh" is nonzero.  Fills "g1" and "g2"
 * with associated geometry modules (if nonzero).
 */
#define MAXFACETS 5

char *reportcurrenterror(INTBIG showhigh, GEOM **g1, GEOM **g2)
{
	REGISTER ERRORLIST *el;
	REGISTER NODEPROTO *facet;
	NODEPROTO *facetlist[MAXFACETS];
	REGISTER INTBIG i, j, consize, numfacets;
	INTBIG lx, hx, ly, hy;
	REGISTER ERRORHIGHLIGHT *eh;
	REGISTER GEOM *geom1, *geom2;
	REGISTER WINDOWPART *w;

	el = db_curerrorlist;
	if (el == NOERRORLIST)
	{
		(void)initinfstr();
		(void)formatinfstr(_("No %s errors"), db_errorsystem);
		return(returninfstr());
	}

	/* turn off highlighting */
	if (showhigh != 0)
	{
		(void)askaid(us_aid, "clear");

		/* first figure out which facets need to be displayed */
		numfacets = 0;
		for(i=0; i<el->numhighlights; i++)
		{
			eh = el->highlights[i];
			facet = el->facet;
			switch (eh->type)
			{
				case ERRORTYPEGEOM:
					if (eh->showgeom == 0) facet = NONODEPROTO; else
						facet = geomparent(eh->geom);
					break;
				case ERRORTYPEFACET:
					facet = (NODEPROTO *)eh->geom;
					break;
			}
			if (facet == NONODEPROTO) continue;
			for(j=0; j<numfacets; j++)
				if (facetlist[j] == facet) break;
			if (j < numfacets) continue;
			if (numfacets >= MAXFACETS) break;
			facetlist[numfacets] = facet;
			numfacets++;
		}

		/* be sure that all requested facets are shown */
		for(i=0; i<numfacets; i++)
		{
			for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
				if (w->curnodeproto == facetlist[i]) break;
			if (w != NOWINDOWPART) continue;

			/* not displayed: find a window to use */
			for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
			{
				for(j=0; j<numfacets; j++)
					if (w->curnodeproto == facetlist[j]) break;
				if (j >= numfacets) break;
			}
			if (w == NOWINDOWPART) w = us_wantnewwindow(0);
			el_curwindowpart = w;
			us_fullview(facetlist[i], &lx, &hx, &ly, &hy);
			us_switchtofacet(facetlist[i], lx, hx, ly, hy, NONODEINST, NOPORTPROTO, 0, 0);
		}
	}

	/* now show the errors */
	geom1 = geom2 = NOGEOM;
	for(i=0; i<el->numhighlights; i++)
	{
		eh = el->highlights[i];
		switch (eh->type)
		{
			case ERRORTYPEGEOM:
				if (geom1 == NOGEOM) geom1 = eh->geom; else
					if (geom2 == NOGEOM) geom2 = eh->geom;
				if (showhigh != 0 && eh->showgeom != 0)
					(void)askaid(us_aid, "show-object", (INTBIG)eh->geom);
				break;
			case ERRORTYPELINE:
				if (showhigh != 0)
					(void)askaid(us_aid, "show-line", eh->x1, eh->y1, eh->x2, eh->y2,
						el->facet);
				break;
			case ERRORTYPEPOINT:
				if (showhigh != 0)
				{
					consize = lambdaoffacet(el->facet) * 5;
					(void)askaid(us_aid, "show-line", eh->x1-consize, eh->y1-consize,
						eh->x1+consize, eh->y1+consize, el->facet);
					(void)askaid(us_aid, "show-line", eh->x1-consize, eh->y1+consize,
						eh->x1+consize, eh->y1-consize, el->facet);
				}
				break;
		}
	}

	/* return geometry if requested */
	if (g1 != 0) *g1 = geom1;
	if (g2 != 0) *g2 = geom2;

	/* return the error message */
	(void)initinfstr();
	(void)addstringtoinfstr(db_errorsystem);
	(void)addstringtoinfstr(": ");
	(void)addstringtoinfstr(el->message);
	return(returninfstr());
}
