/* mesh.h -- mesh handling routines header
//

   Written and Copyright (C) 1994-1999 by Michael J. Gourlay

This file is part of Xmorph.

Xmorph is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Xmorph is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Xmorph; see the file LICENSE.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.

*/




/* SWIG: Simplified Wrapper Interface Generator */
#ifdef SWIG
%module mesh
%{
#include "mesh.h"



/* mesh_tcl_result: TCL Result string for mesh functions
//
// DESCRIPTION
// This is a sneaky way to return character string results to TCL while
// not having to worry about memory leaks, and while still using SWIG's
// wrapping abilities.  I.e., I don't have to worry about writing my
// own TCL wrapper code and registering it with TCL.
//
// NOTES
// The drawback to this method is potentially that returning a pointer to
// a string could cause problems if that pointer is abused, i.e., if the
// contents of that string are assumed to remain constant by the TCL code.
*/
#define MESH_TCL_RESULT_MAX_LEN 2048
static char mesh_tcl_result[MESH_TCL_RESULT_MAX_LEN];



/* NAME
//   meshLine: return a line of mesh point coordinate pairs as a string
//
//
// ARGUMENTS
//   this (in): mesh pointer
//
//   line_index (in): index of the row or column, depending on 'direction'
//
//   direction (in): 1 for row (horizontal line), 2 for column (vertical line)
//
//
// DESCRIPTION
//   meshLine returns a list of coordinate pairs of all of the mesh
//   points along a line of the given line_index and direction.
//
//
// RETURN VALUE
//   Returns the address of the result string, which is assumed to be
//   passed back to the TCL interpretter as a TCL list.
//   The string used is the global string mesh_tcl_result.  Using a
//   constant address eliminated potential memory leak problems.
//
//
// SEE ALSO
//   mesh_tcl_result
*/
char *
meshLine(const MeshT *this, const int line_index, const int direction)
{
  int vi;
  int nv;
  int str_len = 0;
  char float_string[64];
  int point_index = -10000;

  if(line_index < 0) {
    /* line_index is out of range */
    return NULL;
  }

  if(1 == direction) {
    nv = this->nx;
    if(line_index >= this->ny) {
      /* line_index is out of range */
      return NULL;
    }
  } else if (2 == direction) {
    nv = this->ny;
    if(line_index >= this->nx) {
      /* line_index is out of range */
      return NULL;
    }
  } else {
    /* Invalid value for direction */
    return NULL;
  }

  /* Empty the result string */
  mesh_tcl_result[0] = '\0';

  for(vi = 0; vi < nv; vi++) {
    if(1 == direction) {
      point_index = line_index * this->nx + vi;
    } else if (2 == direction) {
      point_index = vi * this->nx + line_index;
    }
    sprintf(float_string, " %.0f %.0f",
      this->x[point_index], this->y[point_index]);
    str_len += strlen(float_string);
    if(str_len >= MESH_TCL_RESULT_MAX_LEN) {
      fprintf(stderr, "meshLine: mesh_tcl_result length exceeded\n");
      return NULL;
    }
    strcat(mesh_tcl_result, float_string);
  }
  return mesh_tcl_result;
}
%}
#endif /* SWIG */




#ifndef _MESH_H__INCLUDED_
#define _MESH_H__INCLUDED_




typedef struct {
  long    nx;   /* number of mesh points in the x-direction */
  long    ny;   /* number of mesh points in the y-direction */
  double *x;    /* 2D array of mesh point x-values */
  double *y;    /* 2D array of mesh point y-values */
} MeshT;




#ifdef SWIG
%addmethods MeshT {
  MeshT(void)           { return meshNew(0, 0); }

  ~MeshT()              { meshDelete(self); }

  int alloc(int nx, int ny)
                        { return meshAlloc(self, nx, ny); }

  void free()           { meshFree(self); }

  void print()          { meshPrint(self); }

  void interpolate(const MeshT *m1P, const MeshT *m2P, float tween_param)
                        {meshInterpolate(self, m1P, m2P, tween_param);}

  void reset(int img_width, int img_height)
                        { meshReset(self, img_width, img_height); }

  void scale(int img_width, int img_height)
                        { meshScale(self, img_width, img_height); }

  char *row(int line_index)
                        { return meshLine(self, line_index, 1); }

  char *col(int line_index)
                        { return meshLine(self, line_index, 2); }

  int pick(int mouse_x, int mouse_y, int component, float proximity)
        { return meshPick(self, mouse_x, mouse_y, component, proximity); }

  void store()          { meshStore(self); }

  void recover()        { meshRetrieve(self); }

  int lineModify(MeshT *other, int mouse_x, int mouse_y, char line_type, char action)
        { return meshLineMouseModify(self, other, mouse_x, mouse_y,
                                     line_type, action);
        }

  int read(const char *filename)
                        { return meshRead(self, filename); }

  int write(const char *filename)
                        { return meshWrite(self, filename); }

  void match(const MeshT *other)
                        { meshMatch(self, other); }

  float pointGet(int xi, int yi, int component)
        { if(0 == component) return self->x[yi * self->nx + xi];
          else if(1 == component) return self->y[yi * self->nx + xi];
          /* Invalid value for component */
          else return -1.0;
        }

  void set(int xi, int yi, float new_x, float new_y)
        {  meshSet(self, xi, yi, new_x, new_y); }
};
#endif




/* MP_PICK_DIST: farthest you may be from a Mesh point to pick it */
#define MP_PICK_DIST 8




void meshInit(MeshT *this);

int meshAlloc(MeshT *this, int nx, int ny);

MeshT * meshNew(const int nx, const int ny);

void meshFree(MeshT *this);

void meshDelete(MeshT *this);

void meshPrint(const MeshT *this);

int meshCompatibilityCheck(const MeshT *this, const MeshT *other);

void meshInterpolate(MeshT *moP, const MeshT *m1P, const MeshT *m2P, float tween_param);

void meshReset(MeshT *this, const int img_width, const int img_height);

void meshScale(MeshT *this, const int img_width, const int img_height) ;

int meshFunctionalize(MeshT *this, int img_width, int img_height);

long int meshPointNearest(const MeshT *this, int px, int py, int *mi, int *mj, int *dx, int *dy);

int meshPick(const MeshT *this, int mouse_x, int mouse_y, int component, float proximity);

void meshSet(MeshT *this, int xi, int yi, float new_x, float new_y);

void meshCopy(MeshT *this, const MeshT *source) ;

void meshBackupIndexSet(int backup_index) ;

void meshBackupFree(void) ;

void meshStore(const MeshT *this) ;

void meshRetrieve(MeshT *this) ;

int meshLineAdd(MeshT *this, const int mi, const float mt, const int type);

int meshLineDelete(MeshT *this, int mi, int type);

int meshLineMouseModify(MeshT *this, MeshT *other, int mouse_x, int mouse_y, char line_type, char action);

int meshRead(MeshT *this, const char *filename);

int meshWrite(const MeshT *this, char *filename);

void meshMatch(MeshT *this, const MeshT *other);


#ifdef GIMP
void set_src_mesh_name (char *fname);
char *get_src_mesh_name (void);

void set_dst_mesh_name (char *fname);
char *get_dst_mesh_name (void);
#endif


#endif /* _MESH_H__INCLUDED_ */
