/*
   GNU xhippo: a GTK-based playlist manager.
   Copyright 1998, 1999, 2000 Adam Sampson <azz@gnu.org>

   Please report bugs to bug-xhippo@gnu.org.

   This file is part of GNU xhippo.

   GNU xhippo is free software; you can redistribute and/or modify it
   under the terms of that license as published by the Free Software
   Foundation; either version 2 of the License, or (at your option)
   any later version.
   
   GNU xhippo is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with GNU xhippo; see the file COPYING. If not, write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
   MA 02111-1307 USA, or see http://www.gnu.org/.

*/

#include "xhippo.h"

/* Prototypes for static functions. */
static void add_info_window_row(GtkWidget *table, gint pos,
				gchar *text1, gchar *text2);

/* Handle selections from the list. */
void handle_list_select(GtkWidget *gtklist, gint row, gint column,
			GdkEventButton *event, gpointer data) {
  get_songinfo(row)->played = 1;
  start_playing(row);
  update_window();
}

/* Handle right clicks which bring up the popup menu. */
void handle_menu_popup(GtkWidget *widget, GdkEventButton *event, 
		       gpointer data) {
  gint state = 1;

  /* Return if this isn't an RMB click. */
  if (event->button != 3) return;
  
  last_row_hit = -1;
  if (widget == filelist) {
    /* The click was on the list. Get the selected file. */
    gtk_clist_get_selection_info(GTK_CLIST(filelist),
				 event->x, event->y,
				 &last_row_hit, NULL);
    
  } else if (widget == eventbox) {
    /* The click was on the status bar. If there's a song playing then
       use that. */
    if (playing) last_row_hit = last_played;
  }

  /* If the click isn't on the list, disable the menu items. */
  if (last_row_hit < 0) state = 0;

  gtk_widget_set_sensitive(GTK_WIDGET(info_item), state);
  gtk_widget_set_sensitive(GTK_WIDGET(up_item), state);
  gtk_widget_set_sensitive(GTK_WIDGET(down_item), state);
  gtk_widget_set_sensitive(GTK_WIDGET(delete_item), state);

  gtk_menu_popup(GTK_MENU(popupmenu), NULL, NULL, NULL, NULL, 
		 event->button, event->time);
}

/* Handle the "Cancel" button in the file selector. */
void handle_fileselector_cancel(GtkWidget *widget, gpointer data) {
  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}

/* Handle the "OK" button in the Load Playlist file selector. */
void handle_load_ok(GtkWidget *widget, gpointer data) {
  clear_playlist();
  read_playlist(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselector)));
  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}

/* Handle the "OK" button in the Save Playlist file selector. */
void handle_save_ok(GtkWidget *widget, gpointer data) {
  FILE *f = fopen(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselector)), "w");
  int i;

  if (!f) {
    status("Unable to save playlist.");
    return;
  }

  for (i=0; i<listcount; i++) fprintf(f, "%s\n", get_songinfo(i)->name);
  fclose(f);

  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}

/* Handle the "OK" button in the Add file selector. */
void handle_add_ok(GtkWidget *widget, gpointer data) {
  add_file(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselector)));
  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}

#ifdef HAVE_NFTW
/* Handle the "OK" button in the Add Directory file selector. */
void handle_add_dir_ok(GtkWidget *widget, gpointer data) {
  add_directory(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselector)));
  gtk_widget_destroy(fileselector);
  fileselector = NULL;
}
#endif

/* Called when "Next" is pressed. */
void handle_next(GtkWidget *widget, gpointer data) {
  pick_next();
}

/* Called when "Prev" is pressed. */
void handle_prev(GtkWidget *widget, gpointer data) {
  pick_prev();
}

/* Called when attribute_mini has changed, to alter the window size. */
void set_mini() {
  if (attribute_mini) {
    gtk_widget_hide(list_box);
    gtk_widget_set_usize(window, attribute_width, -1);
  } else {
    gtk_widget_show(list_box);

    /* Fake the adjustment being changed, so that the scrollbar gets
       placed in the right position when we come out of mini
       mode. This is necessary because if the adjustment changes
       (read: the list was scrolled) while the scrollbar is hidden,
       the scrollbar will ignore the message it gets and thus will
       come up in the wrong position when we unhide it. */
    gtk_adjustment_value_changed(vadj);

    gtk_widget_set_usize(window, attribute_width, attribute_height);
  }
}

/* Called when the "Mini" checkbox is toggled. */
void handle_minitoggle(GtkWidget *widget, gpointer data) {
  attribute_mini = 0;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(minicheckbox))) {
    attribute_mini = 1;
  }
  set_mini();
  update_window();
}

/* Called when the "Random" checkbox is toggled. */
void handle_randomtoggle(GtkWidget *widget, gpointer data) {
  mode_play_ordered = 1;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(randomcheckbox))) {
    mode_play_ordered = 0;
  }
}

/* Called when "Restart" is pressed. */
void handle_restart(GtkWidget *widget, gpointer data) {
  start_playing(last_played);
  update_window();
}

/* Called when "Stop" is pressed. */
void handle_stop(GtkWidget *widget, gpointer data) {
  stop_playing();
  update_window();
}

/* Called when "Pause" is pressed. */
void handle_pause(GtkWidget *widget, gpointer data) {
  pause_playing();
  update_window();
}

/* Handle the "delete_event" event on an info window. data points to
   the songinfo structure. */
gint handle_info_delete_event(GtkWidget *widget, GdkEvent *event, 
			      gpointer data) {
  return(FALSE); /* Close the window, as FALSE */
}

/* Handle the "destroy" event on an info window. data points to the
   songinfo structure. */
void handle_info_destroy(GtkWidget *widget, gpointer data) {
  /* Clear the pointer to the window. */
  ((songinfo *)data)->info_window = NULL;
}

/* Add a row to the info window. */
static void add_info_window_row(GtkWidget *table, gint pos,
			 gchar *text1, gchar *text2) {
  GtkWidget *label, *text;

  label = gtk_label_new(text1);
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
  gtk_table_attach(GTK_TABLE(table), label,
		   0, 1, pos, pos + 1,
		   GTK_FILL, GTK_FILL, 4, 0);
  gtk_widget_show(label);

  text = gtk_label_new(text2);
  gtk_label_set_justify(GTK_LABEL(text), GTK_JUSTIFY_LEFT);
  gtk_table_attach(GTK_TABLE(table), text,
		   1, 2, pos, pos + 1,
		   GTK_EXPAND, GTK_EXPAND, 4, 0);
  gtk_widget_show(text);
}

/* Handle the items in the popup menu. */
void handle_menu_info(GtkWidget *widget, gpointer data) {
  songinfo *sinfo;
  GtkWidget *table;
  gchar buf[STRBUFSIZE], *p;

  sinfo = get_songinfo(last_row_hit);

  /* If the window already exists, don't bother. */
  if (sinfo->info_window) return;

  /* Make sure we've got the information we need. */
  read_song_info(sinfo, NULL);

  sinfo->info_window = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_title(GTK_WINDOW(sinfo->info_window), _("Song information"));
  gtk_signal_connect(GTK_OBJECT(sinfo->info_window), "delete_event",
		     GTK_SIGNAL_FUNC(handle_info_delete_event), sinfo);
  gtk_signal_connect(GTK_OBJECT(sinfo->info_window), "destroy",
		     GTK_SIGNAL_FUNC(handle_info_destroy), sinfo);

  table = gtk_table_new(4, 2, FALSE);
  gtk_container_add(GTK_CONTAINER(sinfo->info_window), table);

  add_info_window_row(table, 0, _("Name:"), sinfo->display_name);
  add_info_window_row(table, 1, _("Filename:"), sinfo->name);
  snprintf(buf, STRBUFSIZE, "%s", ctime(&sinfo->mtime));
  /* ctime appends a carriage return; strip it off. */
  while ((p = strrchr(buf, '\n'))) *p = '\0';
  add_info_window_row(table, 2, _("Last modified:"), buf);
  snprintf(buf, STRBUFSIZE, "%lu", sinfo->size);
  add_info_window_row(table, 3, _("File size:"), buf);
  gtk_widget_show(table);
  gtk_widget_show(sinfo->info_window);
}

void handle_menu_up(GtkWidget *widget, gpointer data) {
  if (last_row_hit > 0) move_row(last_row_hit, last_row_hit - 1);
}

void handle_menu_down(GtkWidget *widget, gpointer data) {
  if (last_row_hit < listcount - 1) move_row(last_row_hit, last_row_hit + 1);
}

void handle_menu_delete(GtkWidget *widget, gpointer data) {
  /* If we're playing the removed song, stop; otherwise adjust the
     current song number and all the previous links to match the list. */
  if (last_row_hit == last_played) {
    stop_playing();
  } else if (last_played > last_row_hit) {
    int i;

    --last_played;
    for (i = 0; i < listcount; i++) {
      songinfo *s = get_songinfo(i);
      if (s->previous > last_row_hit) --s->previous;
    }
  }
  gtk_clist_remove(GTK_CLIST(filelist), last_row_hit);
  update_window();
}

/* Handle user-supplied items in the menu. */
void handle_menu_user(gpointer data) {
  usercommand *uc = (usercommand *)data;
  gchar cmdbuf[STRBUFSIZE];

  snprintf(cmdbuf, STRBUFSIZE, uc->command,
	   last_row_hit == -1 ? "" : get_songinfo(last_row_hit)->name);
  system(cmdbuf);
}

/* Called when "Load" is pressed. */
void handle_menu_load(GtkWidget *widget, gpointer data) {
  gchar name[STRBUFSIZE];

  get_playlist_dir(name, STRBUFSIZE);
  if (fileselector) return;
  fileselector = gtk_file_selection_new(_("Load playlist"));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileselector), name);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->ok_button),
		     "clicked", (GtkSignalFunc) handle_load_ok,
		     fileselector);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->cancel_button),
		     "clicked", (GtkSignalFunc) handle_fileselector_cancel,
		     fileselector);
  gtk_widget_show(fileselector);
}

void handle_menu_save(GtkWidget *widget, gpointer data) {
  gchar name[STRBUFSIZE];

  get_playlist_dir(name, STRBUFSIZE);
  if (fileselector) return;
  fileselector = gtk_file_selection_new(_("Save playlist"));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileselector), name);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->ok_button),
		     "clicked", (GtkSignalFunc) handle_save_ok,
		     fileselector);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->cancel_button),
		     "clicked", (GtkSignalFunc) handle_fileselector_cancel,
		     fileselector);
  gtk_widget_show(fileselector);
}

void handle_menu_add(GtkWidget *widget, gpointer data) {
  if (fileselector) return;
  fileselector = gtk_file_selection_new(_("Add song"));
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->ok_button),
		     "clicked", (GtkSignalFunc) handle_add_ok,
		     fileselector);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->cancel_button),
		     "clicked", (GtkSignalFunc) handle_fileselector_cancel,
		     fileselector);
  gtk_widget_show(fileselector);
}

#ifdef HAVE_NFTW
void handle_menu_add_dir(GtkWidget *widget, gpointer data) {
  if (fileselector) return;
  fileselector = gtk_file_selection_new(_("Add directory"));
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->ok_button),
		     "clicked", (GtkSignalFunc) handle_add_dir_ok,
		     fileselector);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fileselector)
				->cancel_button),
		     "clicked", (GtkSignalFunc) handle_fileselector_cancel,
		     fileselector);
  gtk_widget_show(fileselector);
}
#endif

/* Handlers for the sort menu items. */
void handle_menu_sort_name(GtkWidget *widget, gpointer data) {
  sortplaylist(sort_compare_name, FALSE);
}
void handle_menu_sort_swapped_name(GtkWidget *widget, gpointer data) {
  sortplaylist(sort_compare_swapped_name, FALSE);
}
void handle_menu_sort_mtime(GtkWidget *widget, gpointer data) {
  sortplaylist(sort_compare_mtime, TRUE);
}

void handle_menu_clear() {
  clear_playlist();
}

/* Handle the "delete_event" event. */
gint handle_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data) {
  /* Save the window position, if necessary. */
  if (mode_save_window_pos) save_window_state();

  return(FALSE); /* Close the window, as FALSE */
}

/* Handle the "destroy" event. */
void handle_destroy(GtkWidget *widget, gpointer data) {
  /* Stop the player. */
  stop_playing();

  /* Then quit the interface. */
  gtk_main_quit();
}

/* Handle "configure" events (moves & resizes) of the main window. */
void handle_configure(GtkWidget *widget, gpointer data) {
  /* Don't bother if we're making the window small */
  if (attribute_mini) return;
  gdk_window_get_root_origin(widget->window, 
			     &attribute_xpos, &attribute_ypos);
  gdk_window_get_geometry(widget->window, NULL, NULL,
			  &attribute_width, &attribute_height,
			  NULL);
}

/* Handle files dropped on our window. */
void handle_dnd_drop(GtkWidget *widget, GdkDragContext *context,
		     gint x, gint y,
		     GtkSelectionData *data, guint info,
		     guint time) {
  if (data->data) {
#ifdef USEGNOME
    /* Use the GNOME functions to parse the list. */
    GList *urls = gnome_uri_list_extract_uris(data->data);
    gchar *s;

    while (urls) {
      s = (gchar *)urls->data;
      if (strlen(s) > 5 && !strncmp(s, "file:", 5)) add_file(s+5);
      urls = urls->next;
    }
    gnome_uri_list_free_strings(urls);
#else
    /* The format of data->data is a list of URIs, seperated by
       and terminating in \r\n. We're only interested in the file:
       URIs. */
    gchar *s = strdup(data->data), *p, *q;

    p = s;
    while ((q = strchr(p, '\r'))) {
      *q = '\0';
      if (!(*++q == '\n')) break;
      *q = '\0';
      if (strlen(p) > 5 && !strncmp(p, "file:", 5)) {
	if (filetype(p + 5) == 2) {
	  add_directory(p + 5);
	} else {
	  add_file(p + 5);
	}
      }
      p = q + 1;
    }
    
    free(s);
#endif
  }   
}

#ifdef USEGNOMEMENUS
void handle_about(GtkWidget *widget, void *data) {
  GtkWidget *about;
  static const gchar *authors[] = { "Adam Sampson", NULL };
  
  about = gnome_about_new("GNU Xhippo", VERSION, "(c) 1998, 1999, 2000, 2001 "
			  "Adam Sampson",
			   authors, "Xhippo is a generic music player"
			   " for UNIX systems.", NULL);
  gtk_widget_show(about);
}
#endif
