/************************************************************************/
/* File		objectfile.cpp						*/
/*									*/
/* Purpose	This C++ program file contains the implementation of	*/
/*		the ObjectFileRecHead class and the ObjectFileHeader	*/
/*		class. These two classes are defined in the header file	*/
/*		objectfile.h. These two classes support the ObjectFile	*/
/*		class template which is also defined in objectfile.h.	*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Thursday, May 2, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <string.h>			// memcpy function.
#include "objectfile.h"			// Class definitions.

/************************************************************************/
/*									*/
/*									*/
/*									*/
/************************************************************************/

status Read_String_From_ObjectFile(File file, const int filehandle, String& string)
   {
      status		result		= OK;
      int		length;
      char*		buffer;

      if (file.Read_File(filehandle, (void*)&length, sizeof(int)) == ERROR)
         {
	    file.Set_Error_Info_String("Could not read string length from file");
	    result	= ERROR;
	 }
      else
         {
	    buffer	= new char[length];
	    if (file.Read_File(filehandle, (void*)buffer, length) == ERROR)
	       {
		  file.Set_Error_Info_String("Could not read string data from file");
		  result	= ERROR;
	       }
	    else
	       {
	          string	= buffer;
	       }
	    delete [] buffer;
	 }
      return(result);
   }

/************************************************************************/
/*									*/
/*									*/
/*									*/
/************************************************************************/

status Write_String_To_ObjectFile(File file, const int filehandle, String& string)
   {
      status		result		= OK;
      int		length;

      length		= string.Length() + 1;
      if (file.Write_File(filehandle, (void*)&length, sizeof(int)) == ERROR)
         {
	    file.Set_Error_Info_String("Could not write string length from file");
	    result	= ERROR;
	 }
      else if (file.Write_File(filehandle, (void*)string.Data(), length) == ERROR)
	 {
	    file.Set_Error_Info_String("Could not write string data from file");
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/*									*/
/*									*/
/*									*/
/************************************************************************/

status Read_Integer_From_ObjectFile(File file, const int filehandle, int& number)
   {
      status		result		= OK;

      if (file.Read_File(filehandle, (void*)&number, sizeof(int)) == ERROR)
         {
	    file.Set_Error_Info_String("Could not read integer from file");
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/*									*/
/*									*/
/*									*/
/************************************************************************/

status Write_Integer_To_ObjectFile(File file, const int filehandle, int& number)
   {
      status		result		= OK;

      if (file.Write_File(filehandle, (void*)&number, sizeof(int)) == ERROR)
         {
	    file.Set_Error_Info_String("Could not write integer from file");
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/*									*/
/*									*/
/*									*/
/************************************************************************/

status Read_Float_From_ObjectFile(File file, const int filehandle, float& number)
   {
      status		result		= OK;

      if (file.Read_File(filehandle, (void*)&number, sizeof(float)) == ERROR)
         {
	    file.Set_Error_Info_String("Could not read float from file");
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/*									*/
/*									*/
/*									*/
/************************************************************************/

status Write_Float_To_ObjectFile(File file, const int filehandle, float& number)
   {
      status		result		= OK;

      if (file.Write_File(filehandle, (void*)&number, sizeof(float)) == ERROR)
         {
	    file.Set_Error_Info_String("Could not write float from file");
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	ObjectFileRecHead()					*/
/*									*/
/* Purpose	This is the default constructor for an			*/
/*		ObjectFileRecHead object. The ObjectFileRecHead object	*/
/*		is used to support the ObjectFile class template. This	*/
/*		constructor does not do anything.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None							*/
/************************************************************************/

ObjectFileRecHead::ObjectFileRecHead()
   {
   }

/************************************************************************/
/* Function	ObjectFileRecHead(const ObjectFileRecHead& record)	*/
/*									*/
/* Purpose	This is the copy constructor for an ObjectFileRecHead	*/
/*		object. The copy constructor simply makes a copy of	*/
/*		itself.							*/
/*									*/
/* Input	This function expects the variable 'record' to contain	*/
/*		a reference to the ObjectFileRecHead object that is to	*/
/*		be copied.						*/
/*									*/
/* Output	This function will make itself a copy of the object	*/
/*		that is passed to it in the variable 'record'.		*/
/************************************************************************/

ObjectFileRecHead::ObjectFileRecHead(const ObjectFileRecHead& record)
   {
      itsdeleted	= record.itsdeleted;
      itsrecordsize	= record.itsrecordsize;
      itsnextdeleted	= record.itsnextdeleted;
   }

/************************************************************************/
/* Function	~ObjectFileRecHead()					*/
/*									*/
/* Purpose	This is the default destructor for an ObjectFileRecHead	*/
/*		object. This constructor does not do anything.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None							*/
/************************************************************************/

ObjectFileRecHead::~ObjectFileRecHead()
   {
   }

/************************************************************************/
/* Function	condition Is_Deleted(void)				*/
/*									*/
/* Purpose	This function can be used to determine if a record is	*/
/*		deleted or not. If this function returns true then the	*/
/*		record is deleted.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If the record is deleted then this function will return	*/
/*		true. This function will return false if the record is	*/
/*		not deleted.						*/
/************************************************************************/

condition ObjectFileRecHead::Is_Deleted(void) const
   {
      return(itsdeleted);
   }

/************************************************************************/
/* Function	void Set_Deleted(const condition deleted)		*/
/*									*/
/* Purpose	This function can be used to set the flag that is used	*/
/*		to determine if a record is deleted or not.		*/
/*									*/
/* Input	This function expects the variable 'deleted' to contain	*/
/*		the new value for the flag that is used to determine if	*/
/*		a record is deleted or not.				*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void ObjectFileRecHead::Set_Deleted(const condition deleted)
   {
      itsdeleted	= deleted;
   }

/************************************************************************/
/* Function	int Get_Record_Size(void)				*/
/*									*/
/* Purpose	This function can be used to get the size of the record	*/
/*		(in bytes).						*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the size of the record (in	*/
/*		bytes).							*/
/************************************************************************/

int ObjectFileRecHead::Get_Record_Size(void) const
   {
      return(itsrecordsize);
   }

/************************************************************************/
/* Function	void Set_Record_Size(const int recordsize)		*/
/*									*/
/* Purpose	This function can be used to set the size of the record	*/
/*		(in bytes).						*/
/*									*/
/* Input	This function expects the variable 'recordsize' to	*/
/*		contain the new size (in bytes) of the record.		*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void ObjectFileRecHead::Set_Record_Size(const int recordsize)
   {
      itsrecordsize	= recordsize;
   }

/************************************************************************/
/* Function	FilePointer Get_Next_Deleted(void)			*/
/*									*/
/* Purpose	This function can be used to get the pointer to the	*/
/*		next deleted record in the chain of deleted records.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return a pointer to the next record	*/
/*		in the chain of deleted records. If this pointer == 0	*/
/*		then this is the last record in the chain.		*/
/************************************************************************/

FilePointer ObjectFileRecHead::Get_Next_Deleted(void) const
   {
      return(itsnextdeleted);
   }

/************************************************************************/
/* Function	void Set_Next_Deleted(const FilePointer nextdeleted)	*/
/*									*/
/* Purpose	This function can be used to set the pointer to the	*/
/*		next deleted record in the chain of deleted records.	*/
/*									*/
/* Input	This function expects the variable 'nextdeleted' to	*/
/*		contain a pointer to the next record in the chain of	*/
/*		deleted records. If this pointer == 0 then this is the	*/
/*		last record in the chain of deleted records.		*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void ObjectFileRecHead::Set_Next_Deleted(const FilePointer nextdeleted)
   {
      itsnextdeleted	= nextdeleted;
   }

/************************************************************************/
/* Function	status Read_Record_Header(File& file,			*/
/*		   const int filehandle, const FilePointer filepointer)	*/
/*									*/
/* Purpose	This function will read in a record's header		*/
/*		information from a file. After this function has read	*/
/*		in the header information, the file will be positioned	*/
/*		to the beginning of the record's data.			*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		reference to a file object. The variable 'filehandle'	*/
/*		must contain the file handle to use when accessing the	*/
/*		physical file. The variable 'filepointer' must contain	*/
/*		the pointer to the record who's header is to be read	*/
/*		in.							*/
/*									*/
/* Output	If this function was able to read in the record's	*/
/*		header information then it will return OK and the	*/
/*		physical file will be positioned to read in the		*/
/*		record's data. If this function was not able to read in	*/
/*		the record's header information then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status ObjectFileRecHead::Read_Record_Header(File& file,
   const int filehandle, const FilePointer filepointer)
   {
      status		result		= OK;

      /* Position the physical file pointer to point to the record's	*/
      /* header information in the file.				*/

      if (file.Position_File(filehandle, filepointer) == ERROR)
         {
	    /* Could not read record header.				*/

	    file.Set_Error_Info_String(
	       "Positioning for record header read in ObjectFile file");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* Now read in the record's header information from the file.	*/

      else if (file.Read_File(filehandle, (void*)this,
         sizeof(ObjectFileRecHead)) == ERROR)
         {
	    /* Could not read record header.				*/

	    file.Set_Error_Info_String(
	       "Reading record header from ObjectFile file");
	    file.Report_Error();
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_Record_Header(File& file,			*/
/*		   const int filehandle, const FilePointer filepointer)	*/
/*									*/
/* Purpose	This function will write a record's header information	*/
/*		to a file. After this function has written the header	*/
/*		information, the file will be positioned to the		*/
/*		beginning of the record's data.				*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		reference to a file object. The variable 'filehandle'	*/
/*		must contain the file handle to use when accessing the	*/
/*		physical file. The variable 'filepointer' must contain	*/
/*		the pointer to the record who's header is to be written	*/
/*		to the file.						*/
/*									*/
/* Output	If this function was able to write the record's header	*/
/*		information then it will return OK and the physical	*/
/*		file will be positioned to write out the record's data.	*/
/*		If this function was not able to write the record's	*/
/*		header information then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status ObjectFileRecHead::Write_Record_Header(File& file,
   const int filehandle, const FilePointer filepointer)
   {
      status		result		= OK;

      if (file.Position_File(filehandle, filepointer) == ERROR)
         {
	    /* Could not write record header.				*/

	    file.Set_Error_Info_String(
	       "Positioning for record header write in ObjectFile file");
	    file.Report_Error();
	    result	= ERROR;
	 }
      else if (file.Write_File(filehandle, (void*)this,
         sizeof(ObjectFileRecHead)) == ERROR)
         {
	    /* Could not write record header.				*/

	    file.Set_Error_Info_String(
	       "Writing record header to ObjectFile file");
	    file.Report_Error();
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	ObjectFileHeader()					*/
/*									*/
/* Purpose	This is the default constructor for an			*/
/*		ObjectFileHeader object. The ObjectFileHeader object	*/
/*		is used to support the ObjectFile class template. This	*/
/*		constructor does not do anything.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None							*/
/************************************************************************/

ObjectFileHeader::ObjectFileHeader()
   {
      register int	index;
      int		size;

      size		= (signed)sizeof(itsreserved);
      for (index = 0; index < size; index++)
         {
	    itsreserved[index]	= '*';
	 }
   }

/************************************************************************/
/* Function	~ObjectFileHeader()					*/
/*									*/
/* Purpose	This is the default destructor for an ObjectFileHeader	*/
/*		object. This constructor does not do anything.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None							*/
/************************************************************************/

ObjectFileHeader::~ObjectFileHeader()
   {
   }

/************************************************************************/
/* Function	void Get_Header_Message(char* message)			*/
/*									*/
/* Purpose	This function can be used to get the contents of the	*/
/*		message area from a file's header. The message area is	*/
/*		a 64 byte buffer that is written to the beginning of	*/
/*		every single file.					*/
/*									*/
/* Input	This function expects the variable 'message' to point	*/
/*		to a 64 byte message buffer.				*/
/*									*/
/* Output	This function will transfer exactly 64 bytes from the	*/
/*		buffer pointed to by the variable 'message' to the	*/
/*		ObjectFileHeader object's message area.			*/
/************************************************************************/

void ObjectFileHeader::Get_Header_Message(char* message)
   {
      memcpy(message, itsmessage, 64);
   }

/************************************************************************/
/* Function	void Set_Header_Message(const char* message)		*/
/*									*/
/* Purpose	This function can be used to set the contents of the	*/
/*		message area in a file's header. The message area is a	*/
/*		64 byte buffer that is written to the beginning of	*/
/*		every single file.					*/
/*									*/
/* Input	This function expects the variable 'message' to point	*/
/*		to a 64 byte message buffer.				*/
/*									*/
/* Output	This function will transfer exactly 64 bytes from the	*/
/*		ObjectFileHeader object's message area to the buffer	*/
/*		pointed to by the variable 'message'.			*/
/************************************************************************/

void ObjectFileHeader::Set_Header_Message(const char* message)
   {
      memcpy(itsmessage, message, 64);
   }

/************************************************************************/
/* Function	FilePointer Get_First_Deleted(void) const		*/
/*									*/
/* Purpose	This function can be used to get the pointer to the	*/
/*		first deleted record in the chain of deleted records.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return a pointer to the first record	*/
/*		in the chain of deleted records. If this pointer == 0	*/
/*		then there are no records in the chain.			*/
/************************************************************************/

FilePointer ObjectFileHeader::Get_First_Deleted(void) const
   {
      return(itsfirstdeleted);
   }

/************************************************************************/
/* Function	void Set_First_Deleted(const FilePointer firstdeleted)	*/
/*									*/
/* Purpose	This function can be used to set the pointer to the	*/
/*		first deleted record in the chain of deleted records.	*/
/*									*/
/* Input	This function expects the variable 'firstdeleted' to	*/
/*		contain a pointer to the first record in the chain of	*/
/*		deleted records. If this pointer == 0 then there are no	*/
/*		records in the chain.					*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void ObjectFileHeader::Set_First_Deleted(const FilePointer firstdeleted)
   {
      itsfirstdeleted	= firstdeleted;
   }

/************************************************************************/
/* Function	FilePointer Get_Next_Available(void) const		*/
/*									*/
/* Purpose	This function can be used to get a pointer to the next	*/
/*		available position in a file where a new record can be	*/
/*		added.							*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return a pointer to the next		*/
/*		available position in a file where a new record can be	*/
/*		added.							*/
/************************************************************************/

FilePointer ObjectFileHeader::Get_Next_Available(void) const
   {
      return(itsnextavailable);
   }

/************************************************************************/
/* Function	void Set_Next_Available(const FilePointer nextavailable)*/
/*									*/
/* Purpose	This function can be used to set a pointer to the next	*/
/*		available position in a file where a new record can be	*/
/*		added.							*/
/*									*/
/* Input	This function expects the variable 'nextavailable' to	*/
/*		point to the next available position in a file where a	*/
/*		new record can be added.				*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void ObjectFileHeader::Set_Next_Available(const FilePointer nextavailable)
   {
      itsnextavailable	= nextavailable;
   }

/************************************************************************/
/* Function	int Get_User_Header_Size(void) const			*/
/*									*/
/* Purpose	This function will return the size (in bytes) of the	*/
/*		user's header area. This is the size that was specified	*/
/*		when the file was created.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the size (in bytes) of the	*/
/*		user's header area.					*/
/************************************************************************/

int ObjectFileHeader::Get_User_Header_Size(void) const
   {
      return(itsuserheadersize);
   }

/************************************************************************/
/* Function	void Set_User_Header_Size(const int headersize)		*/
/*									*/
/* Purpose	This function will set the size (in bytes) of the	*/
/*		user's header area. This size should be specified	*/
/*		when the file is created.				*/
/*									*/
/* Input	This function expects the variable 'headersize' to	*/
/*		contain the size (in bytes) of the user's header area.	*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void ObjectFileHeader::Set_User_Header_Size(const int headersize)
   {
      itsuserheadersize	= headersize;
   }

/************************************************************************/
/* Function	status Read_ObjectFile_Header(File& file,		*/
/*		   const int filehandle, void* userheader)		*/
/*									*/
/* Purpose	This function is responsible for reading the header	*/
/*		information that is stored at the beginning of every	*/
/*		ObjectFile file. The header is read in from the file	*/
/*		when the ObjectFile file is opened. If the variable	*/
/*		'userheader' contains a pointer then the user's header	*/
/*		information is also read in from the file.		*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		reference to a file object. The variable 'filehandle'	*/
/*		must contain the physical file handle of the file that	*/
/*		is to have its header information read in. If the	*/
/*		variable 'userheader' contains a pointer then the	*/
/*		user's header area is also read in from the file.	*/
/*									*/
/* Output	If this function was able to read in the file's header	*/
/*		information then this function will return OK. If the	*/
/*		variable 'userheader' contained a pointer then the	*/
/*		memory that it points to will contain the user's header	*/
/*		information. If this function was not able to read in	*/
/*		the file's header information then it will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status ObjectFileHeader::Read_ObjectFile_Header(File& file,
   const int filehandle, void* userheader)
   {
      status		result		= OK;

      /* Position the physical file to the beginning of the file.	*/

      if (file.Position_File(filehandle, 0) == ERROR)
         {
	    /* Could not read file header.				*/

	    file.Set_Error_Info_String(
	       "Positioning to read ObjectFile file header");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* Read in the file's header from the physical file.		*/

      else if (file.Read_File(filehandle, (void*)this,
         sizeof(ObjectFileHeader)) == ERROR)
         {
	    /* Could not read file header.				*/

	    file.Set_Error_Info_String(
	       "Reading header from ObjectFile file");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* If this file contains a user header then write it too.		*/

      else if (userheader != (void*)0)
         {
	    /* Read the user header from the front of the file.		*/

	    if (file.Read_File(filehandle, userheader,
	       itsuserheadersize) == ERROR)
	       {
	          /* Could not read file header.			*/

	          file.Set_Error_Info_String(
	             "Reading user header from ObjectFile file");
	          file.Report_Error();
	          result	= ERROR;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_ObjectFile_Header(File& file,		*/
/*		   const int filehandle, void* userheader)		*/
/*									*/
/* Purpose	This function will write out the header information to	*/
/*		the beginning of an ObjectFile file. If a pointer to a	*/
/*		buffer that contains the user's header was specified in	*/
/*		the variable 'userheader' then the buffer is written to	*/
/*		the user's header area of the file.			*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		reference to a file object. The variable 'filehandle'	*/
/*		must contain the physical file handle of the file that	*/
/*		is to have its header information written out. Finally,	*/
/*		if the variable 'userheader' contains a pointer then	*/
/*		the buffer that it points to will be written to the	*/
/*		user's header area of the file.				*/
/*									*/
/* Output	If this function was able to write the file's header	*/
/*		information then this function will return OK. If this	*/
/*		function was not able to write the file's header	*/
/*		information then it will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status ObjectFileHeader::Write_ObjectFile_Header(File& file,
   const int filehandle, void* userheader)
   {
      status		result		= OK;

      /* Position the physical file to the beginning of the file.	*/

      if (file.Position_File(filehandle, 0) == ERROR)
         {
	    /* Could not write file header.				*/

	    file.Set_Error_Info_String(
	       "Positioning to write ObjectFile file header");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* Write out the file's header to the physical file.		*/

      else if (file.Write_File(filehandle, (void*)this,
         sizeof(ObjectFileHeader)) == ERROR)
         {
	    /* Could not write file header.				*/

	    file.Set_Error_Info_String(
	       "Writing header to ObjectFile file");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* If this file contains a user header then write it too.		*/

      else if (userheader != (void*)0)
         {
	    /* Write the user header to the front of the file.		*/

	    if (file.Write_File(filehandle, userheader,
	       itsuserheadersize) == ERROR)
	       {
	          /* Could not write file header.			*/

	          file.Set_Error_Info_String(
	             "Writing user header to ObjectFile file");
	          file.Report_Error();
	          result	= ERROR;
	       }
	 }
      return(result);
   }
