/************************************************************************/
/* File		dir.cpp							*/
/*									*/
/* Purpose	This C++ program file implements the Dir class. The Dir	*/
/*		class provides a C++ interface to the physical		*/
/*		directory routines.					*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00						*/
/*									*/
/* Date		Wednseday, May 1, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <sys/stat.h>			// File stat function.
#include <unistd.h>			// Standard unix.
#include <errno.h>			// OS error codes.
#include <stdlib.h>			// getenv().
#include <string.h>			// Standard C string library.
#include "dir.h"			// Dir class.

/* Static	The following variable is the error message array that	*/
/*		is specific to the Dir class. The default		*/
/*		UtilityModule error values are overwritten by the Dir	*/
/*		object constructor.					*/

static char *errormessages[] =
   {
      "No error",			// DirNoError
      "Path buffer overflow",		// DirPathBuff
      "Could not create directory",	// DirNoMake
      "Path is not a directory",	// DirNotDir
      "Could not get $HOME directory",	// DirNoHome
      "Could not get CWD",		// DirNoCWD
      "Could not change directory",	// DirNoChange
      "Could not remove directory",	// DirNoDel
   };

/************************************************************************/
/* Function	Dir()							*/
/*									*/
/* Purpose	This is the default constructor for a Dir object. This	*/
/*		constructor will prepare the object to report errors if	*/
/*		it needs to.						*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function alters inherited UtilityModule variables.	*/
/************************************************************************/

Dir::Dir()
   {
      itsmodulename	= "Dir";
      itsmaxerror	= DirMaxError;
      itserrormessages	= errormessages;
   }

/************************************************************************/
/* Function	~Dir()							*/
/*									*/
/* Purpose	This is the default destructor for a Dir object. It	*/
/*		does not do anything.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

Dir::~Dir()
   {
   }

/************************************************************************/
/* Function	condition Does_Directory_Exist(const char* directory)	*/
/*									*/
/* Purpose	This function can be used to determine if a directory	*/
/*		already exists or not. If the directory does exist then	*/
/*		this function will return true. This function will	*/
/*		return false if the directory does not exist or if	*/
/*		there was an error.					*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the path of the directory that is to be tested.	*/
/*									*/
/* Output	This function will return true if the directory does	*/
/*		exist. This function will return false if the directory	*/
/*		does not exist or if there was an error. If there was	*/
/*		an error, the error will not be saved and the error	*/
/*		will not be reported.					*/
/************************************************************************/

condition Dir::Does_Directory_Exist(const char* directory)
   {
      condition		result		= false;
      struct stat	statbuff;

      if (stat(directory, &statbuff) == 0)
         {
	    if (S_ISDIR(statbuff.st_mode) != 0)
	       {
	          result	= true;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	condition Does_Directory_Exist(const String& directory)	*/
/*									*/
/* Purpose	This function can be used to determine if a directory	*/
/*		already exists or not. If the directory does exist then	*/
/*		this function will return true. This function will	*/
/*		return false if the directory does not exist or if	*/
/*		there was an error.					*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the path of the directory that is to be tested.	*/
/*									*/
/* Output	This function will return true if the directory does	*/
/*		exist. This function will return false if the directory	*/
/*		does not exist or if there was an error. If there was	*/
/*		an error, the error will not be saved and the error	*/
/*		will not be reported.					*/
/************************************************************************/

condition Dir::Does_Directory_Exist(const String& directory)
   {
      return(Does_Directory_Exist(directory.Data()));
   }

/************************************************************************/
/* Function	status Get_Current_Working_Directory(String& directory)	*/
/*									*/
/* Purpose	This function will get the current working directory.	*/
/*		The current working directory path will be returned in	*/
/*		the variable 'directory'.				*/
/*									*/
/* Input	This function requires a reference to a String object	*/
/*		that will receive the current working directory path.	*/
/*									*/
/* Output	If this function was able to get the current working	*/
/*		directory then it will return OK and the current	*/
/*		working directory will be stored in the String object	*/
/*		referenced by the variable 'directory'. If this		*/
/*		function was not able to get the current working	*/
/*		directory then it will return ERROR. The error		*/
/*		condition will be saved but it will not be reported.	*/
/*		You will need to call Dir::Report_Error() if you want	*/
/*		the error reported.					*/
/************************************************************************/

status Dir::Get_Current_Working_Directory(String& directory)
   {
      status		result		= OK;
      char		fullpath[1024];

      if (getcwd(fullpath, 1024) == (char*)0)
         {
	    itsoserror		= errno;
	    itserror		= DirNoCWD;
	    result		= ERROR;
	 }
      else
         {
	    directory	= fullpath;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Change_Directory(const char* directory)		*/
/*									*/
/* Purpose	This function will change the current working directory	*/
/*		to the directory that is specified in the variable	*/
/*		'directory'.						*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the directory path that the current working	*/
/*		directory is to be change to. The directory must exist.	*/
/*									*/
/* Output	If this function is able to change the current working	*/
/*		directory (CWD) to the directory that was specified in	*/
/*		the variable 'directory' then it will return OK. If	*/
/*		this function was not able to change the CWD then it	*/
/*		will return ERROR. The error condition will be saved	*/
/*		but it will not be reported. You must call		*/
/*		Dir::Report_Error() if you want to report the error.	*/
/************************************************************************/

status Dir::Change_Directory(const char* directory)
   {
      status		result		= OK;

      if (chdir(directory) != 0)
         {
	    /* Could not change directory.				*/

	    itserrorinfo	= "Attempted to change directory to ";
	    itserrorinfo	+= directory;
	    itsoserror		= errno;
	    itserror		= DirNoChange;
	    result		= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Change_Directory(const String& directory)	*/
/*									*/
/* Purpose	This function will change the current working directory	*/
/*		to the directory that is specified in the variable	*/
/*		'directory'.						*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the directory path that the current working	*/
/*		directory is to be change to. The directory must exist.	*/
/*									*/
/* Output	If this function is able to change the current working	*/
/*		directory (CWD) to the directory that was specified in	*/
/*		the variable 'directory' then it will return OK. If	*/
/*		this function was not able to change the CWD then it	*/
/*		will return ERROR. The error condition will be saved	*/
/*		but it will not be reported. You must call		*/
/*		Dir::Report_Error() if you want to report the error.	*/
/************************************************************************/

status Dir::Change_Directory(const String& directory)
   {
      return(Change_Directory(directory.Data()));
   }

/************************************************************************/
/* Function	status Create_Directory(const char* directory,		*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This function will create a directory if it does not	*/
/*		already exist. The variable 'mode' should contain the	*/
/*		permission bits for the created directory(s). The value	*/
/*		of DirDefaultMode is used if 'mode' is not included.	*/
/*		The following constants are defined in dir.h for your	*/
/*		use:							*/
/*									*/
/*		Constant		Value	Description		*/
/*									*/
/*		DirModeOwnRWX		0700	Owner read/write/exec.	*/
/*		DirModeOwnRead		0400	Owner can read.		*/
/*		DirModeOwnWrite		0200	Owner can write.	*/
/*		DirModeOwnExec		0100	Owner can execute.	*/
/*		DirModeGrpRWX		0070	Group read/write/exec.	*/
/*		DirModeGrpRead		0040	Group can read.		*/
/*		DirModeGrpWrite		0020	Group can write.	*/
/*		DirModeGrpExec		0010	Group can execute.	*/
/*		DirModeAllRWX		0007	All read/write/exec.	*/
/*		DirModeAllRead		0004	All others can read.	*/
/*		DirModeAllWrite		0002	All others can write.	*/
/*		DirModeAllExec		0001	All others can execute.	*/
/*									*/
/*		Create the directory mode by combining the modes that	*/
/*		you want. (e.g. DirModeOwnRead | DirModeOwnWrite).	*/
/*									*/
/*		NOTE: Depending on your 'umask' value, the directory	*/
/*		may be created with a different mode than you		*/
/*		requested. (See Dir::Get_Linux_Mask())			*/
/*									*/
/* Input	This function expects 'directory' to contain the path	*/
/*		of the directory to be created. 'mode' must constain	*/
/*		the read/write/exec permissions for the directory that	*/
/*		is to be created. The value of DirDefaultMode is used	*/
/*		if 'mode' is not included.				*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the named directory or if the named directory already	*/
/*		existed. This function will return ERROR if it could	*/
/*		not create the named directory. Dir::Report_Error() can	*/
/*		be used to report any errors by this function.		*/
/************************************************************************/

status Dir::Create_Directory(const char* directory, const int mode)
   {
      status		result		= OK;

      if (Does_Directory_Exist(directory) == false)
         {
            if (mkdir(directory, mode) != 0)
               {
	          /* Could not Create directory.			*/

	          itserrorinfo	= "Attempted to create directory ";
	          itserrorinfo	+= directory;
	          itsoserror	= errno;
	          itserror	= DirNoMake;
	          result	= ERROR;
		 }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Create_Directory(const String& directory,	*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This function will create a directory if it does not	*/
/*		already exist. The variable 'mode' should contain the	*/
/*		permission bits for the created directory(s). The value	*/
/*		of DirDefaultMode is used if 'mode' is not included.	*/
/*		The following constants are defined in dir.h for your	*/
/*		use:							*/
/*									*/
/*		Constant		Value	Description		*/
/*									*/
/*		DirModeOwnRWX		0700	Owner read/write/exec.	*/
/*		DirModeOwnRead		0400	Owner can read.		*/
/*		DirModeOwnWrite		0200	Owner can write.	*/
/*		DirModeOwnExec		0100	Owner can execute.	*/
/*		DirModeGrpRWX		0070	Group read/write/exec.	*/
/*		DirModeGrpRead		0040	Group can read.		*/
/*		DirModeGrpWrite		0020	Group can write.	*/
/*		DirModeGrpExec		0010	Group can execute.	*/
/*		DirModeAllRWX		0007	All read/write/exec.	*/
/*		DirModeAllRead		0004	All others can read.	*/
/*		DirModeAllWrite		0002	All others can write.	*/
/*		DirModeAllExec		0001	All others can execute.	*/
/*									*/
/*		Create the directory mode by combining the modes that	*/
/*		you want. (e.g. DirModeOwnRead | DirModeOwnWrite).	*/
/*									*/
/*		NOTE: Depending on your 'umask' value, the directory	*/
/*		may be created with a different mode than you		*/
/*		requested. (See Dir::Get_Linux_Mask())			*/
/*									*/
/* Input	This function expects 'directory' to contain the path	*/
/*		of the directory to be created. 'mode' must constain	*/
/*		the read/write/exec permissions for the directory that	*/
/*		is to be created. The value of DirDefaultMode is used	*/
/*		if 'mode' is not included.				*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the named directory or if the named directory already	*/
/*		existed. This function will return ERROR if it could	*/
/*		not create the named directory. Dir::Report_Error() can	*/
/*		be used to report any errors by this function.		*/
/************************************************************************/

status Dir::Create_Directory(const String& directory, const int mode)
   {
      return(Create_Directory(directory.Data(), mode));
   }

/************************************************************************/
/* Function	status Remove_Directory(const char* directory)		*/
/*									*/
/* Purpose	This function will remove the directory that is passed	*/
/*		to it in the variable 'directory'. The directory that	*/
/*		is to be removed must be empty.				*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the path of the directory that is to be		*/
/*		removed.						*/
/*									*/
/* Output	If this function was able to remove the directory then	*/
/*		it will return OK. If this function was not able to	*/
/*		remove the directory then it will return ERROR. The	*/
/*		error condition will be saved but it will not be	*/
/*		reported. You can use Dir::Report_Error() if you want	*/
/*		to report the error.					*/
/************************************************************************/

status Dir::Remove_Directory(const char* directory)
   {
      status		result		= OK;

      if (rmdir(directory) != 0)
         {
	    /* Could not remove directory.				*/

	    itserrorinfo	= "Attempted to remove directory ";
	    itserrorinfo	+= directory;
	    itsoserror		= errno;
	    itserror		= DirNoDel;
	    result		= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Remove_Directory(const String& directory)	*/
/*									*/
/* Purpose	This function will remove the directory that is passed	*/
/*		to it in the variable 'directory'. The directory that	*/
/*		is to be removed must be empty.				*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the path of the directory that is to be		*/
/*		removed.						*/
/*									*/
/* Output	If this function was able to remove the directory then	*/
/*		it will return OK. If this function was not able to	*/
/*		remove the directory then it will return ERROR. The	*/
/*		error condition will be saved but it will not be	*/
/*		reported. You can use Dir::Report_Error() if you want	*/
/*		to report the error.					*/
/************************************************************************/

status Dir::Remove_Directory(const String& directory)
   {
      return(Remove_Directory(directory.Data()));
   }

/************************************************************************/
/* Function	status Get_Home_Directory(String& directory)		*/
/*									*/
/* Purpose	This function will get the home directory path from the	*/
/*		user's environment variable. The directory path will be	*/
/*		returned in 'directory'.				*/
/*									*/
/* Input	This function does not depend on input.			*/
/*									*/
/* Output	If this function was able to get the home directory	*/
/*		path from the environment variable then it will store	*/
/*		the path in 'directory' and return OK. If it could not	*/
/*		get the home directory path, this function will return	*/
/*		ERROR. Dir::Report_Error() can be used to report any	*/
/*		errors by this function.				*/
/************************************************************************/

status Dir::Get_Home_Directory(String& directory)
   {
      status		result		= OK;

      directory		= getenv("HOME");	// Get HOME directory.
      if (directory.Length() == 0)
         {
	    itsoserror		= errno;
	    itserror		= DirNoHome;
	    result		= ERROR;
         }
      return(result);
   }

/************************************************************************/
/* Function	status Create_Recursive_Directory(			*/
/*		const char* directory, const int mode)			*/
/*									*/
/* Purpose	This function will create a directory if it does not	*/
/*		already exist. This is true even if several		*/
/*		subdirectories need to be created. The variable 'mode'	*/
/*		should contain the permission bits for the created	*/
/*		directory(s). The value of DirDefaultMode is used if	*/
/*		'mode' is not included. The following constants are	*/
/*		defined in dir.h for your use:				*/
/*									*/
/*		Constant		Value	Description		*/
/*									*/
/*		DirModeOwnRWX		0700	Owner read/write/exec.	*/
/*		DirModeOwnRead		0400	Owner can read.		*/
/*		DirModeOwnWrite		0200	Owner can write.	*/
/*		DirModeOwnExec		0100	Owner can execute.	*/
/*		DirModeGrpRWX		0070	Group read/write/exec.	*/
/*		DirModeGrpRead		0040	Group can read.		*/
/*		DirModeGrpWrite		0020	Group can write.	*/
/*		DirModeGrpExec		0010	Group can execute.	*/
/*		DirModeAllRWX		0007	All read/write/exec.	*/
/*		DirModeAllRead		0004	All others can read.	*/
/*		DirModeAllWrite		0002	All others can write.	*/
/*		DirModeAllExec		0001	All others can execute.	*/
/*									*/
/*		Create the directory mode by combining the modes that	*/
/*		you want. (e.g. DirModeOwnRead | DirModeOwnWrite).	*/
/*									*/
/*		NOTE: Depending on your 'umask' value, the directory	*/
/*		may be created with a different mode than you		*/
/*		requested. (See Dir::Get_Linux_Mask())			*/
/*									*/
/* Input	This function expects 'directory' to contain the path	*/
/*		of the directory to be created. 'mode' must constain	*/
/*		the read/write/exec permissions for the directory that	*/
/*		is to be created. The value of DirDefaultMode is used	*/
/*		if 'mode' is not included.				*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the named directory or if the named directory already	*/
/*		existed. This function will return ERROR if it could	*/
/*		not create the named directory. Dir::Report_Error() can	*/
/*		be used to report any errors by this function.		*/
/************************************************************************/

status Dir::Create_Recursive_Directory(const char* directory,
   const int mode)
   {
      status		result		= OK;	// Return value.
      register int	length;			// 'directory' length.
      register int	index		= 0;	// Array index.
      char		fullpath[1024];		// Work dir path.
      struct stat	statbuff;		// File stat buffer.

      length		= strlen(directory);

      /* Loop through each character in the directory path. Each time	*/
      /* that a '/' is found, check the full path to see if it is an	*/
      /* existing directory. If it is not, then create it and continue	*/
      /* through the loop until the end of the directory string.	*/

      while (index < length)
	 {
	    /* Check that this iteration will not overflow the buffer.	*/

	    if (index == 1024)
	       {
	          /* Directory path buffer overflow.			*/

	          itserror	= DirPathBuff;
	          result	= ERROR;
	          break;
	       }

	    /* Move one character to the path buffer and then check the	*/
	    /* character to see if it terminates a subdirectory.	*/

	    fullpath[index]	= directory[index];
	    if (fullpath[index] == '/' || index + 1 == length)
	       {
	          /* Terminate the path buffer and try to obtain the	*/
		  /* file stats for the current path.			*/

		  fullpath[index + 1]	= '\0';
		  if (stat(fullpath, &statbuff) != 0)
		     {
		        /* Directory does not exist. This means that	*/
			/* we need to try and create the directory.	*/

			if (mkdir(fullpath, mode) != 0)
			   {
			      /* Could not create directory.		*/

			      itserrorinfo	= "Attempted to create";
			      itserrorinfo	+= " directory ";
			      itserrorinfo	+= fullpath;
			      itsoserror	= errno;
			      itserror		= DirNoMake;
			      result		= ERROR;
			      break;
			   }
		     }

		  /* If we made it here then we obtained stats for the	*/
		  /* current path and we need to make sure that the	*/
		  /* path is actually a directory.			*/

		  else if (S_ISDIR(statbuff.st_mode) == 0)
		     {
		        /* Path is not a directory.			*/

			itserrorinfo	= "Attempted to create";
			itserrorinfo	= " directory ";
			itserrorinfo	+= fullpath;
		        itserror	= DirNotDir;
		        result		= ERROR;
		        break;
		     }
	       }
	    index++;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Create_Recursive_Directory(			*/
/*		const String& directory, const int mode)		*/
/*									*/
/* Purpose	This function will create a directory if it does not	*/
/*		already exist. This is true even if several		*/
/*		subdirectories need to be created. The variable 'mode'	*/
/*		should contain the permission bits for the created	*/
/*		directory(s). The value of DirDefaultMode is used if	*/
/*		'mode' is not included. The following constants are	*/
/*		defined in dir.h for your use:				*/
/*									*/
/*		Constant		Value	Description		*/
/*									*/
/*		DirModeOwnRWX		0700	Owner read/write/exec.	*/
/*		DirModeOwnRead		0400	Owner can read.		*/
/*		DirModeOwnWrite		0200	Owner can write.	*/
/*		DirModeOwnExec		0100	Owner can execute.	*/
/*		DirModeGrpRWX		0070	Group read/write/exec.	*/
/*		DirModeGrpRead		0040	Group can read.		*/
/*		DirModeGrpWrite		0020	Group can write.	*/
/*		DirModeGrpExec		0010	Group can execute.	*/
/*		DirModeAllRWX		0007	All read/write/exec.	*/
/*		DirModeAllRead		0004	All others can read.	*/
/*		DirModeAllWrite		0002	All others can write.	*/
/*		DirModeAllExec		0001	All others can execute.	*/
/*									*/
/*		Create the directory mode by combining the modes that	*/
/*		you want. (e.g. DirModeOwnRead | DirModeOwnWrite).	*/
/*									*/
/*		NOTE: Depending on your 'umask' value, the directory	*/
/*		may be created with a different mode than you		*/
/*		requested. (See Dir::Get_Linux_Mask())			*/
/*									*/
/* Input	This function expects 'directory' to contain the path	*/
/*		of the directory to be created. 'mode' must constain	*/
/*		the read/write/exec permissions for the directory that	*/
/*		is to be created. The value of DirDefaultMode is used	*/
/*		if 'mode' is not included.				*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the named directory or if the named directory already	*/
/*		existed. This function will return ERROR if it could	*/
/*		not create the named directory. Dir::Report_Error() can	*/
/*		be used to report any errors by this function.		*/
/************************************************************************/

status Dir::Create_Recursive_Directory(const String& directory,
   const int mode)
   {
      return(Create_Recursive_Directory(directory.Data(), mode));
   }
