/************************************************************************/
/* File		dataindex.cpp						*/
/*									*/
/* Purpose	This C++ program file contains the class implementation	*/
/*		for the DataIndexRec class. The DataIndexRec class	*/
/*		supports the DataIndex class. The DataIndexRec class is	*/
/*		the ObjectFile record that is used by the DataIndex	*/
/*		class. The DataIndexRec class is defined in the header	*/
/*		file dataindex.h.					*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, May 26, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include "dataindex.h"		// DataIndexRec class.

/************************************************************************/
/* Function	DataIndexRec()						*/
/*									*/
/* Purpose	This is the DataIndexRec default constructor. This	*/
/*		constructor does not do anything.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

DataIndexRec::DataIndexRec()
   {
   }

/************************************************************************/
/* Function	DataIndexRec(const DataIndexRec& object)		*/
/*									*/
/* Purpose	This is the default DataIndexRec copy constructor. This	*/
/*		constructor will make a deep copy of the object.	*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		a reference to the object that is to be copied.		*/
/*									*/
/* Output	None.							*/
/************************************************************************/

DataIndexRec::DataIndexRec(const DataIndexRec& object)
   {
      itskey		= object.itskey;
      itsvalue		= object.itsvalue;
      itscolor		= object.itscolor;
      itsleftnode	= object.itsleftnode;
      itsrightnode	= object.itsrightnode;
   }

/************************************************************************/
/* Function	~DataIndexRec()						*/
/*									*/
/* Purpose	This is the ObjectFileRec default destructor. The	*/
/*		default destructor does nothing.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

DataIndexRec::~DataIndexRec()
   {
   }

/************************************************************************/
/* Function	int Record_Size(void)					*/
/*									*/
/* Purpose	The DataIndexRec class is used as an ObjectFile record.	*/
/*		An ObjectFile record must supply this function to the	*/
/*		ObjectFile class so that the ObjectFile class can	*/
/*		determine how big a record is (in bytes).		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the number of bytes that are	*/
/*		required to hold the object.				*/
/************************************************************************/

int DataIndexRec::Record_Size(void)
   {
      return(sizeof(int) + itskey.Length() + 1 + (sizeof(FilePointer) * 3)
         + sizeof(RBColor));
   }

/************************************************************************/
/* Function	status Read_From_File(File& file, const int filehandle)	*/
/*									*/
/* Purpose	The DataIndexRec class is uses as an ObjectFile record.	*/
/*		An ObjectFile record must supply this function to the	*/
/*		ObjectFile class so that the ObjectFile class can read	*/
/*		the object in from the file.				*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		reference to a File class object. This function expects	*/
/*		the variable 'filehandle' to contain the physical file	*/
/*		handle to use when reading.				*/
/*									*/
/* Output	If the object was able to read itself in from the file	*/
/*		then it will return OK. If the object was not able to	*/
/*		read itself in from the file then it will return ERROR	*/
/*		All errors by this function are saved by the File	*/
/*		class object but they are not reported to stderr.	*/
/************************************************************************/

status DataIndexRec::Read_From_File(File& file, const int filehandle)
   {
      status		result		= OK;
      int		length;
      char*		buffer;

      /* The first thing to read in is the length of the key's string.	*/

      if (file.Read_File(filehandle, (void*)&length, sizeof(int))
         == ERROR)
         {
	    file.Set_Error_Info_String("Read DataIndex record failed");
	    file.Report_Error();
	    result	= ERROR;
	 }
      else
         {
	    /* Create a temporary buffer that can hold the key.		*/

	    buffer	= new char[length + 1];
	    if (file.Read_File(filehandle, (void*)buffer, length + 1)
	       == ERROR)
               {
	          file.Set_Error_Info_String(
		     "Read DataIndex record failed");
	          file.Report_Error();
	          result	= ERROR;
	       }
	    else
	       {
	          /* Store the string in the member variable.		*/

	          itskey	= buffer;

		  /* Read in the key's value.				*/

		  if (file.Read_File(filehandle, (void*)&itsvalue,
		     sizeof(FilePointer)) == ERROR)
		     {
		        file.Set_Error_Info_String(
			   "Read DataIndex record failed");
			file.Report_Error();
			result	= ERROR;
		     }

		  /* Read in the key's color.				*/

		  else if (file.Read_File(filehandle, (void*)&itscolor,
		     sizeof(RBColor)) == ERROR)
		     {
		        file.Set_Error_Info_String(
			   "Read DataIndex record failed");
			file.Report_Error();
			result	= ERROR;
		     }

		  /* Read in the key's left node pointer.		*/

		  else if (file.Read_File(filehandle, (void*)&itsleftnode,
		     sizeof(FilePointer)) == ERROR)
		     {
		        file.Set_Error_Info_String(
			   "Read DataIndex record failed");
			file.Report_Error();
			result	= ERROR;
		     }

		  /* Read in the key's right node pointer.		*/
		  else if (file.Read_File(filehandle,
		     (void*)&itsrightnode, sizeof(FilePointer)) == ERROR)
		     {
		        file.Set_Error_Info_String(
			   "Read DataIndex record failed");
			file.Report_Error();
			result	= ERROR;
		     }
	       }
	    delete buffer;		// Free temporary buffer.
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_To_File(File& file, const int filehandle)	*/
/*									*/
/* Purpose	The DataIndexRec class is uses as an ObjectFile record.	*/
/*		An ObjectFile record must supply this function to the	*/
/*		ObjectFile class so that the ObjectFile class can write	*/
/*		the object out to the file.				*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		reference to a File class object. This function expects	*/
/*		the variable 'filehandle' to contain the physical file	*/
/*		handle to use when writing.				*/
/*									*/
/* Output	If the object was able to write itself out to the file	*/
/*		then it will return OK. If the object was not able to	*/
/*		write itself out to the file then it will return ERROR	*/
/*		All errors by this function are saved by the File	*/
/*		class object but they are not reported to stderr.	*/
/************************************************************************/

status DataIndexRec::Write_To_File(File& file, const int filehandle)
   {
      status		result		= OK;
      int		length;

      /* Get the length of the key's string in a variable so it can be	*/
      /* written out to the file.					*/

      length		= itskey.Length();

      /* Write the length of the key's character string to the file.	*/

      if (file.Write_File(filehandle, (void*)&length, sizeof(int))
         == ERROR)
         {
	    file.Set_Error_Info_String(
	       "Write DataIndex record failed");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* Write the key's character string to the file.			*/

      else if (file.Write_File(filehandle, (void*)itskey.Data(),
         length + 1) == ERROR)
         {
	    file.Set_Error_Info_String(
	       "Write DataIndex record failed");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* Write the key's value to the file.				*/

      else if (file.Write_File(filehandle, (void*)&itsvalue,
         sizeof(FilePointer)) == ERROR)
         {
	    file.Set_Error_Info_String(
	       "Write DataIndex record failed");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* Write the key's color to the file.				*/

      else if (file.Write_File(filehandle, (void*)&itscolor,
      sizeof(RBColor)) == ERROR)
         {
	    file.Set_Error_Info_String(
	       "Write DataIndex record failed");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* Write the key's left node to the file.				*/

      else if (file.Write_File(filehandle, (void*)&itsleftnode,
      sizeof(FilePointer)) == ERROR)
         {
	    file.Set_Error_Info_String(
	       "Write DataIndex record failed");
	    file.Report_Error();
	    result	= ERROR;
	 }

      /* Write the key's right node to the file.			*/

      else if (file.Write_File(filehandle, (void*)&itsrightnode,
         sizeof(FilePointer)) == ERROR)
         {
	    file.Set_Error_Info_String(
	       "Write DataIndex record failed");
	    file.Report_Error();
	    result	= ERROR;
	 }
      return(result);
   }

/************************************************************************/
/* Function	String Get_Record_Key(void) const			*/
/*									*/
/* Purpose	This function will return a record's key.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return a String object that contains	*/
/*		a record's key.						*/
/************************************************************************/

String DataIndexRec::Get_Record_Key(void) const
   {
      return(itskey);
   }

/************************************************************************/
/* Function	void Set_Record_Key(const char* key)			*/
/*									*/
/* Purpose	This function will set a record's key to the value of	*/
/*		the variable 'key'.					*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		new value for a record's key.				*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndexRec::Set_Record_Key(const char* key)
   {
      itskey		= key;
   }

/************************************************************************/
/* Function	void Set_Record_Key(const String& key)			*/
/*									*/
/* Purpose	This function will set a record's key to the value of	*/
/*		the variable 'key'.					*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		new value for a record's key.				*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndexRec::Set_Record_Key(const String& key)
   {
      itskey		= key;
   }

/************************************************************************/
/* Function	FilePointer Get_Value(void) const			*/
/*									*/
/* Purpose	This function will return a record's value.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the value of a key's record.	*/
/************************************************************************/

FilePointer DataIndexRec::Get_Value(void) const
   {
      return(itsvalue);
   }

/************************************************************************/
/* Function	void Set_Value(const FilePointer value)			*/
/*									*/
/* Purpose	This function will set a record's value.		*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		the new value for a key's record.			*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndexRec::Set_Value(const FilePointer value)
   {
      itsvalue		= value;
   }

/************************************************************************/
/* Function	RBColor Get_Color(void) const				*/
/*									*/
/* Purpose	This function will return a key's red/black color. The	*/
/*		colors are defined in dataindex.h. The colors are:	*/
/*									*/
/*		Color		Value					*/
/*									*/
/*		RBColorRed	true					*/
/*		RBColorBlack	false					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return a key's red/black color.	*/
/************************************************************************/

RBColor DataIndexRec::Get_Color(void) const
   {
      return(itscolor);
   }

/************************************************************************/
/* Function	void Set_Color(const RBColor color)			*/
/*									*/
/* Purpose	This function will set a key's red/black color. The	*/
/*		colors are defined in dataindex.h. The colors are:	*/
/*									*/
/*		Color		Value					*/
/*									*/
/*		RBColorRed	true					*/
/*		RBColorBlack	false					*/
/*									*/
/* Input	This function expects the variable 'color' to contain	*/
/*		the new red/black color for a key's record.		*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndexRec::Set_Color(const RBColor color)
   {
      itscolor		= color;
   }

/************************************************************************/
/* Function	FilePointer Get_Left_Node(void) const			*/
/*									*/
/* Purpose	This function will return a key's left node value.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return a key's left node value.	*/
/************************************************************************/

FilePointer DataIndexRec::Get_Left_Node(void) const
   {
      return(itsleftnode);
   }

/************************************************************************/
/* Function	void Set_Left_Node(const FilePointer leftnode)		*/
/*									*/
/* Purpose	This function will set a key's left node value.		*/
/*									*/
/* Input	This function expects the variable 'leftnode' to	*/
/*		contain the key's new left node value.			*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndexRec::Set_Left_Node(const FilePointer leftnode)
   {
      itsleftnode	= leftnode;
   }

/************************************************************************/
/* Function	FilePointer Get_Right_Node(void) const			*/
/*									*/
/* Purpose	This function will return a key's right node value.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return a key's right node value.	*/
/************************************************************************/

FilePointer DataIndexRec::Get_Right_Node(void) const
   {
      return(itsrightnode);
   }

/************************************************************************/
/* Function	void Set_Right_Node(const FilePointer rightnode)	*/
/*									*/
/* Purpose	This function will set a key's right node value.	*/
/*									*/
/* Input	This function expects the variable 'rightnode' to	*/
/*		contain the key's new right node value.			*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndexRec::Set_Right_Node(const FilePointer rightnode)
   {
      itsrightnode	= rightnode;
   }
