/************************************************************************/
/* File		dataindexadd.cpp					*/
/*									*/
/* Purpose	This C++ program file contains the DataIndex class	*/
/*		member functions that add a key to a DataIndex object.	*/
/*		All of the public and private member functions that are	*/
/*		required to add a key to a DataIndex object are located	*/
/*		in this file.						*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, May 26, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include "dataindex.h"			// DataIndex class.

/************************************************************************/
/* Function	status Add_Key(const char* key, const FilePointer value)*/
/*									*/
/* Purpose	This function is responsible for adding a new key and	*/
/*		its associated value to the red/black tree. If the key	*/
/*		already exists in the tree then this function will	*/
/*		return an error.					*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		new key value to add to the red/black tree. The		*/
/*		variable 'value' should contain the ObjectFile record	*/
/*		number that is associated with the key.			*/
/*									*/
/* Output	If this function is able to add the new key to the	*/
/*		red/black tree then it will return OK. If this function	*/
/*		was not able to add the new key to the red/black tree	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr. NOTE: If this	*/
/*		function cannot add the new key to the index file then	*/
/*		it may lock the index file.				*/
/************************************************************************/

status DataIndex::Add_Key(const char* key, const FilePointer value)
   {
      status		result		= OK;
      FilePointer	pointer;

      /* Begin by making sure that the index file is not locked.	*/

      if (itsheader.Is_Index_Locked() == true)
         {
	    /* Index is corrupted.					*/

	    itserrorinfo	= "Attempted to add ";
	    itserrorinfo	+= key;
	    itserrorinfo	+= " to file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadIndex;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Find the location to insert the new key into the tree.		*/

      else if (locate_insertion_point(key, pointer) == ERROR)
         {
	    /* There was an error locating an insertion point. If the	*/
	    /* variable 'record' equals zero then it was an I/O error.	*/

	    if (pointer == 0)
	       {
	          /* If we made it here then we had an I/O error and we	*/
		  /* need to lock the index file.			*/

                  itsheader.Set_Locked_Index_State(true);
	       }
	 }

      /* Insert the new key into the index file and into the red/black	*/
      /* tree structure.						*/

      else if (insert_key(key, value) == ERROR)
         {
	    /* Could not add key to index.				*/

            itsheader.Set_Locked_Index_State(true);
	 }

      /* Rebalance the red/black tree if it needs it.			*/

      else if (rebalance_after_insert() == ERROR)
         {
	    /* Could not add key to index.				*/

            itsheader.Set_Locked_Index_State(true);
	 }
      else
         {
	    /* Increment the node count and write the index file's	*/
	    /* header back to the file to record the change.		*/

	    itsheader.Increment_Node_Count();
            if (itsindexfile.Write_ObjectFile_Header() == ERROR)
               {
	          /* Could not add key to index file.			*/

	          itserrorinfo	= "Could not write header to index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexBadAdd;
	          result	= ERROR;
	          Report_Error();
	          itsheader.Set_Locked_Index_State(true);
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Add_Key(const String& key,			*/
/*		   const FilePointer value)				*/
/*									*/
/* Purpose	This function is responsible for adding a new key and	*/
/*		its associated value to the red/black tree. If the key	*/
/*		already exists in the tree then this function will	*/
/*		return an error.					*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		new key value to add to the red/black tree. The		*/
/*		variable 'value' should contain the ObjectFile record	*/
/*		number that is associated with the key.			*/
/*									*/
/* Output	If this function is able to add the new key to the	*/
/*		red/black tree then it will return OK. If this function	*/
/*		was not able to add the new key to the red/black tree	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr. NOTE: If this	*/
/*		function cannot add the new key to the index file then	*/
/*		it may lock the index file.				*/
/************************************************************************/

status DataIndex::Add_Key(const String& key, const FilePointer value)
   {
      return(Add_Key(key.Data(), value));
   }

/************************************************************************/
/* Function	status locate_insertion_point(const char* key,		*/
/*		   FilePointer& record)					*/
/*									*/
/* Purpose	This function is responsible for locating the position	*/
/*		in the tree to insert a new key. A stack is used to	*/
/*		keep track of the branch of nodes to the new location.	*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		key that is being inserted into the tree.		*/
/*									*/
/* Output	If this function is able to locate the position in the	*/
/*		tree to insert a new key then this function will return	*/
/*		OK and the stack will contain the branch of nodes that	*/
/*		lead up to the new location. Also, the variable		*/
/*		'record' will be set to 0. If the private member	*/
/*		variable 'itsindex' equals -1 then the new key will be	*/
/*		the root key inserted into the tree. If this function	*/
/*		is not able to locate the position in the tree to	*/
/*		insert the new key, or if the key already exists in the	*/
/*		tree, then this function will return ERROR. If the	*/
/*		error was that the key alreaddy exists in the tree then	*/
/*		the variable 'record' will contain the record pointer	*/
/*		of the key in the tree. All errors by this function are	*/
/*		reported to stderr.					*/
/************************************************************************/

status DataIndex::locate_insertion_point(const char* key,
   FilePointer& record)
   {
      status		result		= OK;
      relation		difference;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Prepare to keep track of the nodes that lead up to the new	*/
      /* insertion point.						*/

      itsindex		= -1;
      record		= 0;
      workpointer	= itsheader.Get_Root_Node();

      /* The following loop will locate the position to insert a new	*/
      /* key into the red/black tree data structure.			*/

      while (workpointer != 0)
         {
	    /* Read in the next index record to compare from the index	*/
	    /* file.							*/

	    if (read_index_record(workobject, workpointer) == ERROR)
	       {
	          /* Could not locate insertion point.			*/

		  itserrorinfo	= "Attempted to locate insertion point "
				  "for \"";
		  itserrorinfo	+= key;
		  itserrorinfo	+= "\" in index file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= DataIndexLocateInsert;
		  result	= ERROR;
		  Report_Error();
		  break;
	       }

	    /* Compare the record that was just read in with the key	*/
	    /* that was passed to this function.			*/

	    difference	= workobject.Get_Record_Key().Compare(key);
	    if (difference == EQUAL)
	       {
	          /* Key already exists.				*/

		  record	= workpointer;
		  itserrorinfo	= "Attempted to locate insertion point "
				  "for \"";
		  itserrorinfo	+= key;
		  itserrorinfo	+= "\" in index file ";
		  itserrorinfo	+= itsfilename;
		  itserror	= DataIndexKeyDup;
		  result	= ERROR;
		  Report_Error();
		  break;
	       }

	    /* Save the record that we just read in on the stack.	*/

	    push_record(workobject, workpointer, difference);

	    /* Prepare to read in the next record to compare.		*/

	    if (difference == LESS)
	       {
	          workpointer	= workobject.Get_Right_Node();
	       }
	    else
	       {
	          workpointer	= workobject.Get_Left_Node();
	       }
	 }				// while loop.
      return(result);
   }

/************************************************************************/
/* Function	status insert_key(const char* key,			*/
/*		   const FilePointer value)				*/
/*									*/
/* Purpose	This function is responsible for inserting a new key	*/
/*		into a DataIndex file. This function assumes that the	*/
/*		location to insert the new key has already been		*/
/*		determined by the 'locate_insertion_point; function.	*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		new unique key to add to the index file. The variable	*/
/*		'value' should contain the ObjectFile record number	*/
/*		that is indexed by the given key.			*/
/*									*/
/* Output	If this function is able to insert the new key into the	*/
/*		index file then it will return OK. If this function was	*/
/*		not able to add the new key to the index file then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataIndex::insert_key(const char* key, const FilePointer value)
   {
      status		result		= OK;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Prepare a new DataIndex record with the values that were	*/
      /* passed to this function.					*/

      workobject.Set_Record_Key(key);
      workobject.Set_Value(value);
      workobject.Set_Color(RBColorRed);
      workobject.Set_Left_Node(0);
      workobject.Set_Right_Node(0);

      /* Add the new record to the object file. (The new record is	*/
      /* added to the physical file using the ObjectFile function but	*/
      /* the record still needs to be inserted into the red/black tree)	*/

      if (itsindexfile.Add_ObjectFile_Record(workobject, workpointer)
         == ERROR)
         {
	    /* Could not insert key.					*/

	    itserrorinfo	= "Attempted to insert key \"";
	    itserrorinfo	+= key;
	    itserrorinfo	+= "\" into index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoInsert;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the variable 'itsindex' equals -1 then this is the first	*/
      /* key in the tree (the root node).				*/

      else if (itsindex == -1)
         {
	    /* inserting the root node.					*/

	    itsheader.Set_Root_Node(workpointer);
	 }

      /* Else, the last node that was placed on the stack when the	*/
      /* insertion point was located is the parent of the new node.	*/

      else
         {
	    /* The direction was determined when the insertion point	*/
	    /* was located.						*/

	    if (itsdifferences[itsindex] == LESS)
	       {
	          itsobjects[itsindex].Set_Right_Node(workpointer);
	       }
	    else
	       {
	          itsobjects[itsindex].Set_Left_Node(workpointer);
	       }

	    /* Write the altered record to the index file, inserting	*/
	    /* the new record into the tree structure.			*/

	    if (write_index_record(itsobjects[itsindex],
	       itspointers[itsindex]) == ERROR)
               {
	          /* Could not insert key.				*/

	          itserrorinfo	= "Attempted to insert key \"";
	          itserrorinfo	+= key;
	          itserrorinfo	+= "\" into index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoInsert;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status insert_case_one_both_sides(DataIndexRec& object,	*/
/*		   FilePointer pointer)					*/
/*									*/
/* Purpose	This function assists the function that inserts a new	*/
/*		key into the red/black tree. This function is called	*/
/*		when the uncle of the key that is being rebalanced is	*/
/*		red. This function will rearrange the color scheme to	*/
/*		maintain the black height of both sides of the tree.	*/
/*									*/
/* Input	This function expects the variable 'object' to be a	*/
/*		reference to the object of the key that is being	*/
/*		rebalanced. This function expects the variable		*/
/*		'pointer' to contain the record pointer to the newly	*/
/*		inserted key in the index file.				*/
/*									*/
/* Output	If this function is able to rearrange the color scheme	*/
/*		to maintain the black height then it will return OK. If	*/
/*		this function was not able to rearrange the color	*/
/*		scheme then it will return ERROR. All errors by this	*/
/*		function are reported to stderr.			*/
/************************************************************************/

status DataIndex::insert_case_one_both_sides(DataIndexRec& object,
   FilePointer pointer)
   {
      status		result		= OK;

      /* The object's color is set to black; the node on the top of the	*/
      /* stack is set to black; and the node that is second from the	*/
      /* top of the stack is set to red.				*/

      object.Set_Color(RBColorBlack);
      itsobjects[itsindex].Set_Color(RBColorBlack);
      itsobjects[itsindex - 1].Set_Color(RBColorRed);

      /* Write the object that is being rebalanced back to the file.	*/

      if (write_index_record(object, pointer) == ERROR)
	 {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Write the object on the top of the stack back to the file.	*/

      else if (write_index_record(itsobjects[itsindex],
         itspointers[itsindex]) == ERROR)
	 {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Write the object that is second from the top of the stack.	*/

      else if (write_index_record(itsobjects[itsindex - 1],
         itspointers[itsindex - 1]) == ERROR)
	 {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Remove two nodes from the top of the stack before returning.	*/

      itsindex		-= 2;
      return(result);
   }

/************************************************************************/
/* Function	status insert_case_two_right_side(DataIndexRec& object,	*/
/*		   FilePointer pointer)					*/
/*									*/
/* Purpose	This function will perform a left rotation around the	*/
/*		object's grandparent.					*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the object that is being rebalanced in the red/black	*/
/*		tree. The variable 'pointer' must contain the index	*/
/*		file record number for the referenced object.		*/
/*									*/
/* Output	If this function is able to perform a left rotation	*/
/*		around the object's grandparent then it will return OK.	*/
/*		If this function was not able to perform a left		*/
/*		rotation around the object's grandparent then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataIndex::insert_case_two_right_side(DataIndexRec& object,
   FilePointer pointer)
   {
      status		result		= OK;

      /* The grandparents color equals RED.				*/

      itsobjects[itsindex - 1].Set_Color(RBColorRed);

      /* The object's color equals BLACK.				*/

      object.Set_Color(RBColorBlack);

      /* The grandparent's right node equals the object's left node.	*/

      itsobjects[itsindex - 1].Set_Right_Node(object.Get_Left_Node());

      /* The object's left node equals the pointer to the grandparent.	*/

      object.Set_Left_Node(itspointers[itsindex - 1]);

      /* Write the object back to the index file to record the change.	*/

      if (write_index_record(object, pointer) == ERROR)
	 {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Write the grandparent back to the index file.			*/

      else if (write_index_record(itsobjects[itsindex - 1],
         itspointers[itsindex - 1]) == ERROR)
	 {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the stack index equals one then we need to change the root	*/
      /* node of the tree.						*/

      if (itsindex == 1)
         {
	    itsheader.Set_Root_Node(pointer);
	 }

      /* Else, we need to change the pointer to the grandparent's node.	*/

      else
         {
	    if (itsdifferences[itsindex - 2] == LESS)
	       {
	          /* Change the right node that pointed to grandparent.	*/

	          itsobjects[itsindex - 2].Set_Right_Node(pointer);
	       }
	    else
	       {
	          /* Change the left node that pointed to grandparent.	*/

	          itsobjects[itsindex - 2].Set_Left_Node(pointer);
	       }

	    /* Write the record to the index file to record the change.	*/

	    if (write_index_record(itsobjects[itsindex - 2],
	       itspointers[itsindex - 2]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status insert_case_three_right_side(			*/
/*		   DataIndexRec& object, FilePointer& pointer)		*/
/*									*/
/* Purpose	This function will rotate the tree right at the		*/
/*		object's parent. This has the effect of changing case 3	*/
/*		into case 2. This function will then return the		*/
/*		object's parent as the object.				*/
/*									*/
/* Input	This function expects the variable 'object' to be a	*/
/*		reference to the object that is being rebalanced. The	*/
/*		variable 'pointer' must contain the index file record	*/
/*		number for the object.					*/
/*									*/
/* Output	If this function is able to transform case 3 into case	*/
/*		2 then this function will return OK and the variable	*/
/*		'object' will contain the parent of the record that was	*/
/*		passed to this function in the variable 'object'. Also,	*/
/*		the variable 'pointer' will contain the index file	*/
/*		record number for the record in 'object'. If this	*/
/*		function is not able to transform case 3 into case 2	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status DataIndex::insert_case_three_right_side(DataIndexRec& object,
   FilePointer& pointer)
   {
      status		result		= OK;

      /* Read in the left node of the record on the top of stack.	*/

      pointer		= itsobjects[itsindex].Get_Left_Node();
      if (read_index_record(object, pointer) == ERROR)
	 {
	    /* Could not delete key.				*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* The left node of the record on the top of the stack is	*/
	    /* set to the right node of the object.			*/

	    itsobjects[itsindex].Set_Left_Node(object.Get_Right_Node());

	    /* The object's right node is set to the pointer to the	*/
	    /* record on the top of the stack.				*/

	    object.Set_Right_Node(itspointers[itsindex]);

	    /* The right node of the object that is second from the top	*/
	    /* on the stack is set to point to the object.		*/

	    itsobjects[itsindex - 1].Set_Right_Node(pointer);

	    /* Write the object to the index file to record the change.	*/

	    if (write_index_record(object, pointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Write the record on the top of the stack.		*/

	    else if (write_index_record(itsobjects[itsindex],
	       itspointers[itsindex]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Write the record that is second from the top of stack.	*/

	    else if (write_index_record(itsobjects[itsindex - 1],
	       itspointers[itsindex - 1]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status insert_case_two_left_side(DataIndexRec& object,	*/
/*		   FilePointer pointer)					*/
/*									*/
/* Purpose	This function will perform a right rotation around the	*/
/*		object's grandparent.					*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		the object that is being rebalanced in the red/black	*/
/*		tree. The variable 'pointer' must contain the index	*/
/*		file record number for the referenced object.		*/
/*									*/
/* Output	If this function is able to perform a right rotation	*/
/*		around the object's grandparent then it will return OK.	*/
/*		If this function was not able to perform a right	*/
/*		rotation around the object's grandparent then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataIndex::insert_case_two_left_side(DataIndexRec& object,
   FilePointer pointer)
   {
      status		result		= OK;

      /* The grandparents color equals RED.				*/

      itsobjects[itsindex - 1].Set_Color(RBColorRed);

      /* The object's color equals BLACK.				*/

      object.Set_Color(RBColorBlack);

      /* The grandparent's left node equals the object's right node.	*/

      itsobjects[itsindex - 1].Set_Left_Node(object.Get_Right_Node());

      /* The object's right node equals the pointer to the grandparent.	*/

      object.Set_Right_Node(itspointers[itsindex - 1]);

      /* Write the object back to the index file to record the change.	*/

      if (write_index_record(object, pointer) == ERROR)
	 {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* Write the grandparent back to the index file.			*/

      else if (write_index_record(itsobjects[itsindex - 1],
         itspointers[itsindex - 1]) == ERROR)
	 {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }

      /* If the stack index equals one then we need to change the root	*/
      /* node of the tree.						*/

      if (itsindex == 1)
         {
	    itsheader.Set_Root_Node(pointer);
	 }

      /* Else, we need to change the pointer to the grandparent's node.	*/

      else
         {
	    if (itsdifferences[itsindex - 2] == LESS)
	       {
	          /* Change the right node that pointed to grandparent.	*/

	          itsobjects[itsindex - 2].Set_Right_Node(pointer);
	       }
	    else
	       {
	          /* Change the left node that pointed to grandparent.	*/

	          itsobjects[itsindex - 2].Set_Left_Node(pointer);
	       }

	    /* Write the record to the index file to record the change.	*/

	    if (write_index_record(itsobjects[itsindex - 2],
	       itspointers[itsindex - 2]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status insert_case_three_left_side(			*/
/*		   DataIndexRec& object, FilePointer& pointer)		*/
/*									*/
/* Purpose	This function will rotate the tree left at the object's	*/
/*		parent. This has the effect of changing case 3 into	*/
/*		case 2. This function will then return the object's	*/
/*		parent as the object.					*/
/*									*/
/* Input	This function expects the variable 'object' to be a	*/
/*		reference to the object that is being rebalanced. The	*/
/*		variable 'pointer' must contain the index file record	*/
/*		number for the object.					*/
/*									*/
/* Output	If this function is able to transform case 3 into case	*/
/*		2 then this function will return OK and the variable	*/
/*		'object' will contain the parent of the record that was	*/
/*		passed to this function in the variable 'object'. Also,	*/
/*		the variable 'pointer' will contain the index file	*/
/*		record number for the record in 'object'. If this	*/
/*		function is not able to transform case 3 into case 2	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status DataIndex::insert_case_three_left_side(DataIndexRec& object,
   FilePointer& pointer)
   {
      status		result		= OK;

      /* Read in the right node of the record on the top of stack.	*/

      pointer		= itsobjects[itsindex].Get_Right_Node();
      if (read_index_record(object, pointer) == ERROR)
	 {
	    /* Could not delete key.					*/

	    itserrorinfo	= "Attempted to delete key \"";
	    itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	    itserrorinfo	+= "\" from index file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexNoDelete;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    /* The right node of the record on the top of the stack is	*/
	    /* set to the left node of the object.			*/

	    itsobjects[itsindex].Set_Right_Node(object.Get_Left_Node());

	    /* The object's left node is set to the pointer to the	*/
	    /* record on the top of the stack.				*/

	    object.Set_Left_Node(itspointers[itsindex]);

	    /* The left node of the object that is second from the top	*/
	    /* on the stack is set to point to the object.		*/

	    itsobjects[itsindex - 1].Set_Left_Node(pointer);

	    /* Write the object to the index file to record the change.	*/

	    if (write_index_record(object, pointer) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Write the record on the top of the stack.		*/

	    else if (write_index_record(itsobjects[itsindex],
	       itspointers[itsindex]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }

	    /* Write the record that is second from the top of stack.	*/

	    else if (write_index_record(itsobjects[itsindex - 1],
	       itspointers[itsindex - 1]) == ERROR)
	       {
	          /* Could not delete key.				*/

	          itserrorinfo	= "Attempted to delete key \"";
	          itserrorinfo	+= itsdeletedobject.Get_Record_Key();
	          itserrorinfo	+= "\" from index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexNoDelete;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status rebalance_after_insert(void)			*/
/*									*/
/* Purpose	This function is responsible for rebalancing the tree	*/
/*		after a new key has been added. This is done in a loop	*/
/*		that checks the record on the top of the stack for a	*/
/*		rule 1 violation. If rebalancing is required then this	*/
/*		function divides the task into a left side rebalancing	*/
/*		and a right side rebalancing.				*/
/*									*/
/* Input	This function assumes that the stack contains the	*/
/*		branch of nodes that lead up to a newly inserted key.	*/
/*									*/
/* Output	If this function was able to rebalance the red/black	*/
/*		tree after a new key has been added then this function	*/
/*		will return OK. If this function was not able to	*/
/*		rebalance the red/black tree then it will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/************************************************************************/

status DataIndex::rebalance_after_insert(void)
   {
      status		result		= OK;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* The rebalancing loop will continue while the private member	*/
      /* variable 'itsindex' is greater than 0 and while the color of	*/
      /* the record on the top of the stack is equal to RED.		*/

      while (itsindex > 0 && itsobjects[itsindex].Get_Color()
         == RBColorRed)
         {
	    /* Determine if we need to rebalance the left side or the	*/
	    /* right side of the red/black tree.			*/

	    if (itsdifferences[itsindex - 1] == LESS)
	       {
	          /* Right side rebalancing. Get the left node of the	*/
		  /* grandparent on the stack.				*/

		  workpointer	= itsobjects[itsindex - 1].
				  Get_Left_Node();

		  /* If the pointer actually contains a record number...*/

		  if (workpointer != 0)
		     {
		        /* ...then read in the record.			*/

	                if (read_index_record(workobject, workpointer)
			   == ERROR)
	                   {
	                      /* Could not delete key.			*/

	                      itserrorinfo	= "Attempted to delete "
						  "key \"";
	                      itserrorinfo	+= itsdeletedobject.
						   Get_Record_Key();
	                      itserrorinfo	+= "\" from index file ";
	                      itserrorinfo	+= itsfilename;
	                      itserror		= DataIndexNoDelete;
	                      result		= ERROR;
	                      Report_Error();
			      break;
	                   }
		     }

		  /* If the pointer actually contains a record number	*/
		  /* and the color of that record is equal to RED...	*/

		  if (workpointer != 0 && workobject.Get_Color()
		     == RBColorRed)
		     {
		        /* Rearrange the tree's color scheme and then	*/
			/* go through the loop again.			*/

			result		= insert_case_one_both_sides(
					  workobject, workpointer);
		     }

		  /* Else, we need to rearrange the nodes and not just	*/
		  /* the tree's color scheme.				*/

		  else
		     {
		        /* If the saved difference of the record on the	*/
			/* top of the stack equals less then we need to	*/
			/* change the record that is being worked on.	*/

			if (itsdifferences[itsindex] == LESS)
			   {
			      /* Get the record pointer for the record	*/
			      /* that is on top of the stack and read	*/
			      /* it in as the new object to work with.	*/

			      workpointer	= itspointers[itsindex];
	                      if (read_index_record(workobject,
			         workpointer) == ERROR)
	                         {
	                            /* Could not delete key.		*/

	                            itserrorinfo = "Attempted to "
						   "delete key \"";
	                            itserrorinfo += itsdeletedobject.
						    Get_Record_Key();
	                            itserrorinfo += "\" from index file ";
	                            itserrorinfo += itsfilename;
	                            itserror	 = DataIndexNoDelete;
	                            result	 = ERROR;
	                            Report_Error();
			            break;
	                         }
			   }
			else
			   {
			      /* Transform case 3 into case 2 and fall	*/
			      /* through to the next transformation	*/

			      result	= insert_case_three_right_side(
					  workobject, workpointer);
			   }

			/* Perform a left rotation around the object's	*/
			/* grandparent.					*/

			if (result == OK)
			   {
			      result	= insert_case_two_right_side(
					  workobject, workpointer);
			   }

			/* Break out of the loop.			*/

		        break;
		     }
	       }
	    else
	       {
	          /* Left side rebalancing. Get the right node of the	*/
		  /* grandparent on the stack.				*/

		  workpointer	= itsobjects[itsindex - 1].
				  Get_Right_Node();

		  /* If the pointer actually contains a record number...*/

		  if (workpointer != 0)
		     {
		        /* ...then read in the record.			*/

	                if (read_index_record(workobject, workpointer)
			   == ERROR)
	                   {
	                      /* Could not delete key.			*/

	                      itserrorinfo	= "Attempted to delete "
						  "key \"";
	                      itserrorinfo	+= itsdeletedobject.
						   Get_Record_Key();
	                      itserrorinfo	+= "\" from index file ";
	                      itserrorinfo	+= itsfilename;
	                      itserror		= DataIndexNoDelete;
	                      result		= ERROR;
	                      Report_Error();
			      break;
	                   }
		     }

		  /* If the pointer actually contains a record number	*/
		  /* and the color of that record is equal to RED...	*/

		  if (workpointer != 0 && workobject.Get_Color()
		     == RBColorRed)
		     {
		        /* Rearrange the tree's color scheme and then	*/
			/* go through the loop again.			*/

			result		= insert_case_one_both_sides(
					  workobject, workpointer);
		     }

		  /* Else, we need to rearrange the nodes and not just	*/
		  /* the tree's color scheme.				*/

		  else
		     {
		        /* If the saved difference of the record on the	*/
			/* top of the stack equals less then we need to	*/
			/* change the record that is being worked on.	*/

			if (itsdifferences[itsindex] == MORE)
			   {
			      /* Get the record pointer for the record	*/
			      /* that is on top of the stack and read	*/
			      /* it in as the new object to work with.	*/

			      workpointer	= itspointers[itsindex];
	                      if (read_index_record(workobject,
			         workpointer) == ERROR)
	                         {
	                            /* Could not delete key.		*/

	                            itserrorinfo = "Attempted to "
						   "delete key \"";
	                            itserrorinfo += itsdeletedobject.
						    Get_Record_Key();
	                            itserrorinfo += "\" from index file ";
	                            itserrorinfo += itsfilename;
	                            itserror	 = DataIndexNoDelete;
	                            result	 = ERROR;
	                            Report_Error();
			            break;
	                         }
			   }
			else
			   {
			      /* Transform case 3 into case 2 and fall	*/
			      /* through to the next transformation	*/

			      result	= insert_case_three_left_side(
					  workobject, workpointer);
			   }

			/* Perform a left rotation around the object's	*/
			/* grandparent.					*/

			if (result == OK)
			   {
			      result	= insert_case_two_left_side(
					  workobject, workpointer);
			   }

			/* Break out of the loop.			*/

		        break;
		     }
	       }
	 }

      /* Make sure that there were no errors before going on.		*/

      if (result == OK)
         {
	    /* The red/black tree is rebalanced at this point. If the	*/
	    /* color of the root node was changed to RED by the loop	*/
	    /* then it needs to be changed back to BLACK. If the root	*/
	    /* node equals the pointer of the first record on the stack	*/
	    /* then the root node is already in memor.			*/

	    if (itsheader.Get_Root_Node() == itspointers[0])
	       {
	          /* If the root node's color has been changed to red...*/

		  if (itsobjects[0].Get_Color() == RBColorRed)
		     {
		        /* Set the root node's color to BLACK.		*/

	                itsobjects[0].Set_Color(RBColorBlack);

			/* Write it back to the index file to record	*/
			/* the change.					*/

	                if (write_index_record(itsobjects[0],
			   itspointers[0]) == ERROR)
	                   {
	                      /* Could not delete key.			*/

	                      itserrorinfo	= "Attempted to delete "
						  "key \"";
	                      itserrorinfo	+= itsdeletedobject.
					           Get_Record_Key();
	                      itserrorinfo	+= "\" from index file ";
	                      itserrorinfo	+= itsfilename;
	                      itserror		= DataIndexNoDelete;
	                      result		= ERROR;
	                      Report_Error();
	                   }
		     }
	       }

	    /* Else, we need to read in the root node from the index	*/
	    /* file so that we can make sure that it's color is BLACK.	*/

	    else
	       {
	          /* Read in the root node from the file.		*/

	          if (read_index_record(workobject, itsheader.
		     Get_Root_Node()) == ERROR)
	             {
	                /* Could not delete key.			*/

	                itserrorinfo	= "Attempted to delete key \"";
	                itserrorinfo	+= itsdeletedobject.
					   Get_Record_Key();
	                itserrorinfo	+= "\" from index file ";
	                itserrorinfo	+= itsfilename;
	                itserror	= DataIndexNoDelete;
	                result	= ERROR;
	                Report_Error();
	             }

		  /* If the color of the root node equals RED then...	*/

		  else if (workobject.Get_Color() == RBColorRed)
		     {
		        /* Set the root node's color to BLACK.		*/

		        workobject.Set_Color(RBColorBlack);

			/* Write the root node back to the index file	*/
			/* to record the change.			*/
			
	                if (write_index_record(workobject, itsheader.
			   Get_Root_Node()) == ERROR)
	                   {
	                      /* Could not delete key.			*/

	                      itserrorinfo	= "Attempted to delete "
						  "key \"";
	                      itserrorinfo	+= itsdeletedobject.
						   Get_Record_Key();
	                      itserrorinfo	+= "\" from index file ";
	                      itserrorinfo	+= itsfilename;
	                      itserror		= DataIndexNoDelete;
	                      result		= ERROR;
	                      Report_Error();
	                   }
		     }
	       }
	 }

      return(result);
   }
