/************************************************************************/
/* File		ftp.cpp							*/
/*									*/
/* Purpose	This C++ program file contains the implementation for	*/
/*		the FTP class. The C++ interface provides File Transfer	*/
/*		Protocol (FTP) functions.				*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Monday, May 20, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <stdlib.h>			// atoi.
#include <stdio.h>			// fgets.
#include <string.h>			// strlen.
#include <time.h>			// struct tm.
#include "ftp.h"			// FTP class.

/* Static	The following variable contains the name of each month.	*/
/*		This list of month names is used by the function that	*/
/*		returns a modification time from a directory listing.	*/

static String	months[] =
   {
      "Jan", "Feb", "Mar", "Apr", "May", "Jun",
      "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
   };

/* Static	The following variable is the error message array that	*/
/*		is specific to the Dir class. The default		*/
/*		UtilityModule error values are overwritten by the Dir	*/
/*		object constructor.					*/

static char *errormessages[] =
   {
      "No error",				// FTPNoError
      "Already connected to a host",		// FTPConnected
      "Not connected to a host",		// FTPNotConnected
      "Could not establish connection",		// FTPBadConnect
      "Could not log onto server",		// FTPBadLogon
      "Could not get server's CWD",		// FTPWorkDir
      "Could not get server directory",		// FTPGetDir
      "Directory does not exist",		// FTPNotDir
      "Directory exist on server",		// FTPDirExist
      "Could not create directory",		// FTPMakeDir
      "Could not remove directory",		// FTPDelDir
      "Could not get system type",		// FTPNoType
      "Could not change to parent",		// FTPParentDir
      "Could not get size of file",		// FTPBadSize
      "Could not get modification date",	// FTPBadMode
      "Could not get file",			// FTPBadGet
      "Could not put file",			// FTPBadPut
      "Could not rename file",			// FTPBadRename
      "Could not remove file",			// FTPBadDel
      "File path buffer overflow",		// FTPPathBuff
   };

/************************************************************************/
/* Function	FTP()							*/
/*									*/
/* Purpose	This is the default constructor for an FTP object. This	*/
/*		constructor will prepare the object to report errors if	*/
/*		it needs to.						*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function alters inherited UtilityModule variables.	*/
/************************************************************************/

FTP::FTP()
   {
      itsmodulename	= "FTP";
      itsmaxerror	= FTPErrors;
      itserrormessages	= errormessages;
      itschangedirflag	= true;
      itsnetbuf		= (netbuf*)0;
      FtpInit();
   }

/************************************************************************/
/* Function	~FTP()							*/
/*									*/
/* Purpose	This is the default destructor for an FTP object. This	*/
/*		destructor makes sure that any open connection is	*/
/*		terminated.						*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	Any connection that may have been open will be closed.	*/
/************************************************************************/

FTP::~FTP()
   {
      if (itsnetbuf != (netbuf*)0)
         {
	    FtpQuit(itsnetbuf);
	 }
   }

/************************************************************************/
/* Function	status Connect_To_Host(const char* hostname)		*/
/*									*/
/* Purpose	This function is responsible for connecting to a host	*/
/*		server. The host server is the server that you will be	*/
/*		communicating with (e.g. ftp.server.com). You must	*/
/*		establish a connection with a host server before you	*/
/*		can use the FTP functions.				*/
/*									*/
/* Input	This function expects the variable 'hostname' to	*/
/*		contain the name of the host that you want to establish	*/
/*		a connection with. (e.g. ftp.server.com).		*/
/*									*/
/* Output	This function will return OK if it was able to		*/
/*		establish a connection with the host that was specified	*/
/*		in the variable 'hostname'. If this function was not	*/
/*		able to establish a connection with the host or if	*/
/*		there was an error then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status FTP::Connect_To_Host(const char* hostname)
   {
      status		result		= OK;

      if (itsnetbuf != (netbuf*)0)
         {
	    /* Already connected to a host.				*/

	    itserrorinfo	= "Attempted to connect to ";
	    itserrorinfo	+= hostname;
	    itserror		= FTPConnected;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (FtpConnect(hostname, &itsnetbuf) == 0)
         {
	    /* Could not make the connection.				*/

	    itserrorinfo	= "Attempted to connect to ";
	    itserrorinfo	+= hostname;
	    itserror		= FTPBadConnect;
	    Report_Error();
	 }
      else
         {
	    itshostname		= hostname;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Connect_To_Host(const String& hostname)		*/
/*									*/
/* Purpose	This function is responsible for connecting to a host	*/
/*		server. The host server is the server that you will be	*/
/*		communicating with (e.g. ftp.server.com). You must	*/
/*		establish a connection with a host server before you	*/
/*		can use the FTP functions.				*/
/*									*/
/* Input	This function expects the variable 'hostname' to	*/
/*		contain the name of the host that you want to establish	*/
/*		a connection with. (e.g. ftp.server.com).		*/
/*									*/
/* Output	This function will return OK if it was able to		*/
/*		establish a connection with the host that was specified	*/
/*		in the variable 'hostname'. If this function was not	*/
/*		able to establish a connection with the host or if	*/
/*		there was an error then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status FTP::Connect_To_Host(const String& hostname)
   {
      return(Connect_To_Host(hostname.Data()));
   }

/************************************************************************/
/* Function	status Logon_To_Host(const char* user,			*/
/*		   const char* password)				*/
/*									*/
/* Purpose	This function is responsible for logging a user into	*/
/*		a remote server. A connection should already have been	*/
/*		established with the remote server using the		*/
/*		Connect_To_Host() function.				*/
/*									*/
/* Input	This function expects the variable 'user' to contain	*/
/*		the name to use when logging a user into the server.	*/
/*		The variable 'password' should contain the password to	*/
/*		use when logging a user into the server.		*/
/*									*/
/* Output	This function will return OK if it was able to log the	*/
/*		user into the server. If this function was not able to	*/
/*		log the user into the server or if there was an error	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status FTP::Logon_To_Host(const char* user, const char* password)
   {
      status		result		= OK;
      char		buffer[1024];

      if (itsnetbuf == (netbuf*)0)
         {
	    /* Not connected to a host.					*/

	    itserrorinfo	= "Attempted to log onto ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPNotConnected;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (FtpLogin(user, password, itsnetbuf) == 0)
         {
	    /* Could not loggon to host.				*/

	    itserrorinfo	= "Attempted to log onto ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPBadLogon;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (FtpPwd(buffer, 1024, itsnetbuf) == 0)
         {
	    /* Could not get server's working directory.		*/

	    itserrorinfo	= "Attempted to log onto ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPBadWorkDir;
	    Report_Error();
	 }
      else
         {
	    itsbasecwd		= buffer;
	    if (itsbasecwd.Data()[itsbasecwd.Length() - 1] != '/')
	       {
	          itsbasecwd	+= "/";
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Logon_To_Host(const String& user,		*/
/*		   const String& password)				*/
/*									*/
/* Purpose	This function is responsible for loging a user into	*/
/*		a remote server. A connection should already have been	*/
/*		established with the remote server using the		*/
/*		Connect_To_Host() function.				*/
/*									*/
/* Input	This function expects the variable 'user' to contain	*/
/*		the name to use when logging a user into the server.	*/
/*		The variable 'password' should contain the password to	*/
/*		use when logging a user into the server.		*/
/*									*/
/* Output	This function will return OK if it was able to log the	*/
/*		user into the server. If this function was not able to	*/
/*		log the user into the server or if there was an error	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status FTP::Logon_To_Host(const String& user, const String& password)
   {
      return(Logon_To_Host(user.Data(), password.Data()));
   }

/************************************************************************/
/* Function	status Close_Connection_To_Host(void)			*/
/*									*/
/* Purpose	This function is responsible for terminating a		*/
/*		connection to a server. A connection should already	*/
/*		have been established with the remote server using the	*/
/*		Connect_To_Host() function.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return OK if it was able to		*/
/*		terminate the connection to the server. If this		*/
/*		function was not able to terminate the connection to	*/
/*		the server or if there was an error then this function	*/
/*		will return ERROR. All errors by this function are	*/
/*		reported to stderr.					*/
/************************************************************************/

status FTP::Close_Connection_To_Host(void)
   {
      status		result		= OK;

      if (itsnetbuf == (netbuf*)0)
         {
	    /* Not connected to a host.					*/

	    itserrorinfo	= "Attempted to close connection to ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPNotConnected;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    FtpQuit(itsnetbuf);
	    itsnetbuf	= (netbuf*)0;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_System_Type(String& type)			*/
/*									*/
/* Purpose	After a connection to a server has been established,	*/
/*		this function can be used to return the type of system	*/
/*		that the remote server is operating on (e.g. unix).	*/
/*									*/
/*		NOTE: This function is not supported by all systems.	*/
/*		For the systems that do not support this function, the	*/
/*		variable 'type' will be returned as an empty string.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to return the remote system's	*/
/*		type then this function will return OK and the variable	*/
/*		'type' will contain a string describing the type of	*/
/*		system that the server is operating on (e.g. unix). If	*/
/*		this function was not able to return the system type	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status FTP::Get_System_Type(String& type)
   {
      status		result		= OK;
      char		buffer[1024];

      if (FtpSysType(buffer, 1024, itsnetbuf) == 0)
         {
	    /* Could not get system type.				*/

	    itserrorinfo	= "Attempted to get system type for "
				  "host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPNoType;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    type		= buffer;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Work_Directory(String& directory)		*/
/*									*/
/* Purpose	This function can be used to get the work directory on	*/
/*		the remote server.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to get the work directory on	*/
/*		the server then this function will return OK and the	*/
/*		variable 'directory' will contain the server's working	*/
/*		directory. If this function is not get the server's	*/
/*		working directory then this function will return ERROR.	*/
/*		All errors by this function are reported to stderr.	*/
/************************************************************************/

status FTP::Get_Work_Directory(String& directory)
   {
      status		result		= OK;
      char		buffer[1024];

      if (FtpPwd(buffer, 1024, itsnetbuf) == 0)
         {
	    /* Could not get working directory.				*/

	    itserrorinfo	= "Attempted to get work directory ";
	    itserrorinfo	+= directory;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPBadWorkDir;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    directory		= buffer;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Directory(const String path, int& entries,	*/
/*		   String*& directory)					*/
/*									*/
/* Purpose	This function can be used to return the contents of a	*/
/*		server directory. The directory is returned as an array	*/
/*		of String objects. Each String object in the array will	*/
/*		contain a directory entry in the form:			*/
/*									*/
/*		flags links owner group size date time name		*/
/*									*/
/*		The array will be allocated by this function but you	*/
/*		must be responsible for deleting it. Furthermore, if	*/
/*		the variable 'directory' is already pointing to an	*/
/*		array of objects when it is passed to this function	*/
/*		its value will be written over without deleting the	*/
/*		array first. This could cause to a memory leak.		*/
/*									*/
/* Input	This function expects the variable 'path' to contain	*/
/*		the directory on the server that is to be returned.	*/
/*									*/
/* Output	If this function is able to return the directory that	*/
/*		is specified in the variable 'path' then this function	*/
/*		will return OK and the variable 'entries' will contain	*/
/*		the number of entries in the directory listing. The	*/
/*		variable 'directory' will be a pointer to an array of	*/
/*		String objects. Each String object in the array will	*/
/*		contain a directory entry. If this function is not able	*/
/*		to get the server's directory then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status FTP::Get_Directory(const String& path, int& entries,
   String*& directory)
   {
      status		result		= OK;
      char		buffer[1024];
      FILE*		file;
      String		fullpath	= itsbasecwd;
      register int	index;

      fullpath		+= path;
      if (FtpChdir(fullpath.Data(), itsnetbuf) == 0)
         {
	    /* Directory does not exist.				*/

	    itserrorinfo	= "Attempted to get directory ";
	    itserrorinfo	+= path;
	    itserrorinfo	+= " from host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPNotDir;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (FtpDir(FTPTempFile, fullpath.Data(), itsnetbuf) == 0)
         {
	    /* Could not get directory.					*/

	    itserrorinfo	= "Attempted to get directory ";
	    itserrorinfo	+= path;
	    itserrorinfo	+= " from host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPGetDir;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
            file	= fopen(FTPTempFile, "r");
	    entries	= 0;
	    while (feof(file) == 0 && result == OK)
	       {
	          if (fgets(buffer, 1024, file) == (char*)0)
		     {
		        entries--;
		     }
		  entries++;
	       }
	    directory	= new String[entries];
	    fclose(file);
	    fopen(FTPTempFile, "r");
	    for (index = 0; index < entries; index++)
	       {
	          fgets(buffer, 1024, file);
		  buffer[strlen(buffer) - 1]	= '\0';
	          directory[index]		= buffer;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Create_Directory(const String& directory)	*/
/*									*/
/* Purpose	This function can be used to create a directory on the	*/
/*		remote server. If the directory already exists then	*/
/*		this function will return an error.			*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the full path (relative to the server's base	*/
/*		directory) of the directory that is to be created.	*/
/*									*/
/* Output	If this function is able to create the directory on	*/
/*		the server then this function will return OK. If this	*/
/*		function is not able to create the directory on the	*/
/*		server then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status FTP::Create_Directory(const String& directory)
   {
      status		result		= OK;
      String		fullpath	= itsbasecwd;

      fullpath		+= directory;
      if (FtpChdir(fullpath.Data(), itsnetbuf) == 1)
         {
	    /* Directory exist.						*/

	    itserrorinfo	= "Attempted to create directory ";
	    itserrorinfo	+= directory;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPDirExist;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (FtpMkdir(fullpath.Data(), itsnetbuf) == 0)
         {
	    /* Could not create directory.				*/

	    itserrorinfo	= "Attempted to create directory ";
	    itserrorinfo	+= directory;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPMakeDir;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Create_Recursive_Directory(			*/
/*		   const char* directory)				*/
/*									*/
/* Purpose	This function will create a directory on the server if	*/
/*		it does not already exist. This is true even if several	*/
/*		subdirectories need to be created.			*/
/*									*/
/* Input	This function expects 'directory' to contain the path	*/
/*		of the directory to be created (relative to the base	*/
/*		directory of the server).				*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the named directory or if the named directory already	*/
/*		existed. This function will return ERROR if it could	*/
/*		not create the named directory. All errors by this	*/
/*		function are reported to stderr.			*/
/************************************************************************/

status FTP::Create_Recursive_Directory(const String& directory)
   {
      status		result		= OK;	// Return value.
      register int	index		= 0;	// Array index.
      String		workpath	= itsbasecwd;
      char		fullpath[1024];		// Work dir path.

      workpath		+= directory;

      /* Loop through each character in the directory path. Each time	*/
      /* that a '/' is found, check the full path to see if it is an	*/
      /* existing directory. If it is not, then create it and continue	*/
      /* through the loop until the end of the directory string.	*/

      while (index < workpath.Length())
	 {
	    /* Check that this iteration will not overflow the buffer.	*/

	    if (index == 1024)
	       {
	          /* Directory path buffer overflow.			*/

		  itserrorinfo	= "Attempted to create recursive "
				  "directory ";
		  itserrorinfo	+= directory;
		  itserrorinfo	+= " on host ";
		  itserrorinfo	+= itshostname;
		  itserror	= FTPPathBuff;
		  result	= ERROR;
		  Report_Error();
	          break;
	       }

	    /* Move one character to the path buffer and then check the	*/
	    /* character to see if it terminates a subdirectory.	*/

	    fullpath[index]	= workpath.Data()[index];
	    if (fullpath[index] == '/' || index + 1 == workpath.Length())
	       {
	          /* Terminate the path buffer and see if the directory	*/
		  /* exists on the server.				*/

		  fullpath[index + 1]	= '\0';
		  if (FtpChdir(fullpath, itsnetbuf) == 0)
		     {
		        /* Directory does not exist. This means that	*/
			/* we need to try and create the directory.	*/

			if (FtpMkdir(fullpath, itsnetbuf) == 0)
			   {
			      /* Could not create directory.		*/

			      itserrorinfo	= "Attempted to create "
						  "recursive directory ";
			      itserrorinfo	+= directory;
			      itserrorinfo	+= " on host ";
			      itserrorinfo	+= itshostname;
			      itserror		= FTPMakeDir;
			      result		= ERROR;
			      Report_Error();
			   }
		     }
	       }
	    index++;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Remove_Directory(const String& directory)	*/
/*									*/
/* Purpose	This function can be used to remove a directory on the	*/
/*		remote server. If the directory does not exist or if	*/
/*		the directory contains files and/or subdirectories then	*/
/*		this function will return an error.			*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the full path (relative to the server's base	*/
/*		directory) of the directory that is to be removed.	*/
/*									*/
/* Output	If this function is able to remove the directory on	*/
/*		the server then this function will return OK. If this	*/
/*		function is not able to remove the directory on the	*/
/*		server then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status FTP::Remove_Directory(const String& directory)
   {
      status		result		= OK;
      String		fullpath	= itsbasecwd;

      fullpath		+= directory;
      if (FtpChdir(fullpath.Data(), itsnetbuf) == 0)
         {
	    /* Directory does not exist.				*/

	    itserrorinfo	= "Attempted to remove directory ";
	    itserrorinfo	+= directory;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPNotDir;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (FtpCDUp(itsnetbuf) == 0)
         {
	    /* Could not move work directory to parent.			*/

	    itserrorinfo	= "Attempted to remove directory ";
	    itserrorinfo	+= directory;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPParentDir;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (FtpRmdir(fullpath.Data(), itsnetbuf) == 0)
         {
	    /* Could not remove directory.				*/

	    itserrorinfo	= "Attempted to remove directory ";
	    itserrorinfo	+= directory;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPDelDir;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    itschangedirflag	= true;
	 }
      return(result);
   }

/************************************************************************/
/* Function	Purge_Directory(const String& directory)		*/
/*									*/
/* Purpose	This function can be used to purge a server directory.	*/
/*		purging a directory will remove the directory even if	*/
/*		the directory contains files and subdirectories. The	*/
/*		files and subdirectories will be removed first.		*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the full path of the directory that is to be	*/
/*		purged from the server.					*/
/*									*/
/* Output	If this function is able to purge the directory from	*/
/*		the server then this function will return OK. If this	*/
/*		function is not able to purge the directory from the	*/
/*		server then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status FTP::Purge_Directory(const String& directory)
   {
      status		result		= OK;
      register int	index;
      register int	index2;
      String		filename;
      String		nextpath;
      int		items;
      String*		list		= (String*)0;

      result		= Get_Directory(directory, items, list);
      for (index = 0; index < items && result == OK; index++)
         {
	    index2	= list[index].Length();
	    while (list[index].Data()[--index2] != ' ' && index2 > 0);
	    filename	= list[index].Data() + index2 + 1;
	    if (list[index].Data()[0] == 'd' || list[index].Data()[0] == 'D')
	       {
		  nextpath	= directory;
		  nextpath	+= "/";
		  nextpath	+= filename;
	          result	= Purge_Directory(nextpath);
	       }
	    else
	       {
	          result	= Change_Work_Directory(directory);
		  if (result == OK)
		     {
	                result	= Remove_File(filename);
		     }
	       }
	 }
      if (result == OK)
         {
	    result	= Remove_Directory(directory);
	 }
      if (list != (String*)0)
         {
	    delete [] list;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Change_Work_Directory(const String& directory)	*/
/*									*/
/* Purpose	This function can be used to change to a directory on	*/
/*		the remote server. If the directory does not exist then	*/
/*		this function will return an error.			*/
/*									*/
/* Input	This function expects the variable 'directory' to	*/
/*		contain the full path (relative to the server's base	*/
/*		directory) of the directory to change to.		*/
/*									*/
/* Output	If this function is able to change the directory on	*/
/*		the server then this function will return OK. If this	*/
/*		function is not able to change the directory on the	*/
/*		server then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status FTP::Change_Work_Directory(const String& directory)
   {
      status		result		= OK;
      String		fullpath	= itsbasecwd;

      fullpath		+= directory;
      if (FtpChdir(fullpath.Data(), itsnetbuf) == 0)
         {
	    /* Directory does not exist.				*/

	    itserrorinfo	= "Attempted to change to directory ";
	    itserrorinfo	+= directory;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPNotDir;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Change_To_Parent_Directory(void)			*/
/*									*/
/* Purpose	This function can be used to change the working		*/
/*		directory on the remote server to its parent directory.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to change the working		*/
/*		directory on the server to its parent directory then	*/
/*		this function will return OK. If this function is not	*/
/*		able to change the directory on the server then this	*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status FTP::Change_To_Parent_Directory(void)
   {
      status		result		= OK;

      if (FtpCDUp(itsnetbuf) == 0)
         {
	    /* Could not change to parent directory.			*/

	    itserrorinfo	= "Attempted to change to parent "
				  "directory on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPParentDir;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	condition Does_Directory_Exist(const String& path)	*/
/*									*/
/* Purpose	This function can be used to determine if a directory	*/
/*		exists on the server. If this function can change the	*/
/*		working directory to the directory that is specified in	*/
/*		the variable 'path' then this function will return	*/
/*		true. This function will return false, otherwise.	*/
/*									*/
/* Input	This function expects the variable 'path' to contain	*/
/*		the path of the directory on the server to test. This	*/
/*		should include the path information (relative to the	*/
/*		website's home directory).				*/
/*									*/
/* Output	If this function is able to change the server's working	*/
/*		directory to the specified directory then this function	*/
/*		will return true. Otherwise, this function will return	*/
/*		false.							*/
/************************************************************************/

condition FTP::Does_Directory_Exist(const String& filepath)
   {
      condition		result;
      String		fullpath	= itsbasecwd;

      fullpath		+= filepath;
      if (FtpChdir(fullpath.Data(), itsnetbuf) == 1)
         {
	    result	= true;
	 }
      else
         {
	    result	= false;
	 }
      return(result);
   }

/************************************************************************/
/* Function	condition Does_File_Exist(const String& filepath)	*/
/*									*/
/* Purpose	This function can be used to determine if a file exists	*/
/*		on the server. If this function can get the file's	*/
/*		modification date then this function will report that	*/
/*		the file exists on the server.				*/
/*									*/
/* Input	This function expects the variable 'filepath' to	*/
/*		contain the name of the file to test. This should	*/
/*		include the path information (relative to the website's	*/
/*		home directory).					*/
/*									*/
/* Output	If this function is able to get the file's modification	*/
/*		date then this file will return true. Otherwise, this	*/
/*		function will return false.				*/
/************************************************************************/

condition FTP::Does_File_Exist(const String& filepath)
   {
      condition		result;
      String		tempstring;

      if (Get_Modification_Time(filepath, tempstring) == OK)
         {
	    result	= true;
	 }
      else
         {
	    result	= false;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_File_Size(const String& filepath, int& size,	*/
/*		   const FTPMode mode)					*/
/*									*/
/* Purpose	This function can be used to get the size of a file on	*/
/*		the remote server. The size is returned in the variable	*/
/*		'size'.							*/
/*									*/
/* Input	This function expects the variable 'filepath' to	*/
/*		contain the name to the file that is to have its size	*/
/*		returned. The file should be in the server's current	*/
/*		working directory (see Change_Work_Directory). The	*/
/*		variable 'mode' must contain either FTPBinary or	*/
/*		FTPText.						*/
/*									*/
/* Output	If this function is able to get the size of a file on	*/
/*		the remote server then this function will return OK and	*/
/*		the variable 'size' will contain the size of the file.	*/
/*		If this function is not able to return the size of the	*/
/*		file on the remote server then this function will	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status FTP::Get_File_Size(const String& filepath, int& size,
   const FTPMode mode)
   {
      status		result		= OK;
      String		fullpath	= itsbasecwd;

      fullpath		+= filepath;
      if (FtpSize(fullpath.Data(), &size, mode, itsnetbuf) == 0)
         {
	    /* Could not get size of file.				*/

	    itserrorinfo	= "Attempted to get size of file ";
	    itserrorinfo	+= filepath;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPBadSize;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Modification_Time(const String& filename,	*/
/*		   String& date)					*/
/*									*/
/* Purpose	This function can be used to get the modification time	*/
/*		of a file on the remote server. The modification time	*/
/*		is returned as a string. The format of the string is	*/
/*		YYYYMMDDHHMMSS. YYYY is the year (e.g. 2002). MM is the	*/
/*		month (e.g. 07). DD is the day of the month (e.g. 08).	*/
/*		HH is the hour in the day (e.g. 13). MM is the minute	*/
/*		in the hour (e.g. 04). Finally, SS is the seconds in	*/
/*		the minute (e.g. 50). An example of a returned string	*/
/*		is:							*/
/*									*/
/*		20020708132408						*/
/*									*/
/*		The example above is the date and time for July the 8,	*/
/*		2002 at 13:24:08.					*/
/*									*/
/*		NOTE: This function will attempt to get the file's	*/
/*		modification time from the server using an FTP command.	*/
/*		The command that is used to get the modification time	*/
/*		is not supported by all servers. Therefore, if the FTP	*/
/*		command is not able to return the file's modification	*/
/*		time, a private FTP function will be called that will	*/
/*		get the file's modification time from the server's	*/
/*		directory listing.					*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the name to the file that is to have its	*/
/*		modification date returned. The file should be in the	*/
/*		server's current working directory (see			*/
/*		Change_Work_Directory).					*/
/*									*/
/* Output	If this function is able to get the modification date	*/
/*		of a file on the remote server then this function will	*/
/*		return OK and the variable 'date' will contain the	*/
/*		modification date of the file. If this function is not	*/
/*		able to return the modification date of the file on the	*/
/*		remote server then this function will ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status FTP::Get_Modification_Time(const String& filepath, String& time)
   {
      status		result		= OK;
      char		buffer[1024];
      String		fullpath	= itsbasecwd;

      fullpath		+= filepath;
      if (FtpModDate(fullpath.Data(), buffer, 1024, itsnetbuf) == 0)
         {
	    if (get_modification_time(filepath, time) == ERROR)
	       {
	          /* Could not get modification date.			*/

	          itserrorinfo	= "Attempted to get modification time "
				  "of file ";
	          itserrorinfo	+= filepath;
	          itserrorinfo	+= " on host ";
	          itserrorinfo	+= itshostname;
	          itserror	= FTPBadMod;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      else
         {
	    buffer[14]		= '\0';
	    time		= buffer;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_File(const String& outfile,			*/
/*		   const String& infile, const FTPMode mode)		*/
/*									*/
/* Purpose	This function can be used to get a file from a server.	*/
/*		Specify the file to get in the variable 'infile' and	*/
/*		specify the filename and path to copy the file to in	*/
/*		the variable 'outfile'.					*/
/*									*/
/* Input	This function expects the variable 'outfile' to		*/
/*		contain the filename to use when storing the file. The	*/
/*		filename can be a complete file path or it can be a	*/
/*		a file in the current working directory. The variable	*/
/*		'infile' must contain the name of the file to get from	*/
/*		the remote server. This file must be in the server's	*/
/*		current working directory. Finally, the variable 'mode'	*/
/*		must contain either FTPBinary or FTPText.		*/
/*									*/
/* Output	If this function is able to get 'infile' from the	*/
/*		server and write it to 'outfile' then this function	*/
/*		will return OK. If this function is not able to get the	*/
/*		file from the server then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status FTP::Get_File(const String& outfile, const String& infile,
   const FTPMode mode)
   {
      status		result		= OK;

      if (FtpGet(outfile.Data(), infile.Data(), mode, itsnetbuf) == 0)
         {
	    /* Could not get file.					*/

	    itserrorinfo	= "Attempted to get file ";
	    itserrorinfo	+= outfile;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPBadGet;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Put_File(const String& infile,			*/
/*		   const String& outfile, const FTPMode mode)		*/
/*									*/
/* Purpose	This function can be used to put a file on the server.	*/
/*		Specify the filename and path to put in the variable	*/
/*		'infile' and specify the server file in the variable	*/
/*		'outfile'.						*/
/*									*/
/* Input	This function expects the variable 'outfile' to		*/
/*		contain the filename to use when storing the file on	*/
/*		the server. The file will be placed in the server's	*/
/*		current working directory. The variable 'infile' must	*/
/*		contain the name of the file to put on the server. This	*/
/*		filename can be a complete path or it can be a file in	*/
/*		the current working directory. Finally, the variable	*/
/*		'mode' must contain either FTPBinary or FTPText.	*/
/*									*/
/* Output	If this function is able to put the file specified by	*/
/*		the variable 'infile' on the server then this function	*/
/*		will return OK. If this function is not able to put the	*/
/*		file on the server then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status FTP::Put_File(const String& infile, const String& outfile,
   const FTPMode mode)
   {
      status		result		= OK;

      if (FtpPut(infile.Data(), outfile.Data(), mode, itsnetbuf) == 0)
         {
	    /* Could not put file.					*/

	    itserrorinfo	= "Attempted to put file ";
	    itserrorinfo	+= infile;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPBadPut;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    itschangedirflag	= true;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Rename_File(const String& currentname,		*/
/*		   const String& newname)				*/
/*									*/
/* Purpose	This function can be used to rename a file on a remote	*/
/*		server. The file with the name specified by the		*/
/*		variable 'currentname' will be renamed to the name	*/
/*		specified in the variable 'newname'.			*/
/*									*/
/* Input	This function expects the variable 'currentname' to	*/
/*		specify the name of the file that already exists in the	*/
/*		server's current working directory. The variable	*/
/*		'newname' must contain the new name for the file on the	*/
/*		server.							*/
/*									*/
/* Output	If this function is able to rename the file on the	*/
/*		server then this function will return OK. If this	*/
/*		function was not able to rename the file on the server	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status FTP::Rename_File(const String& currentname, const String& newname)
   {
      status		result		= OK;

      if (FtpRename(currentname.Data(), newname.Data(), itsnetbuf) == 0)
         {
	    /* Could not rename file.					*/

	    itserrorinfo	= "Attempted to rename file ";
	    itserrorinfo	+= currentname;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPBadRename;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    itschangedirflag	= true;
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Remove_File(const String& filename)		*/
/*									*/
/* Purpose	This function can be used to remove a file from the	*/
/*		remote server.						*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the name of the file that is to be removed from	*/
/*		the server. The file should be in the server's current	*/
/*		working directory.					*/
/*									*/
/* Output	If this function is able to remove the file from the	*/
/*		server then this function will return OK. If this	*/
/*		function is not able to remove the file from the server	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status FTP::Remove_File(const String& filename)
   {
      status		result		= OK;

      if (FtpDelete(filename.Data(), itsnetbuf) == 0)
         {
	    /* Could not remove file.					*/

	    itserrorinfo	= "Attempted to remove file ";
	    itserrorinfo	+= filename;
	    itserrorinfo	+= " on host ";
	    itserrorinfo	+= itshostname;
	    itserror		= FTPBadDel;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
	    itschangedirflag	= true;
	 }
      return(result);
   }

/************************************************************************/
/* Function	condition is_item_in_list(const String& filename,	*/
/*		   const String& entry)					*/
/*									*/
/* Purpose	This function can be used to determine if a given	*/
/*		directory entry is for the given file.			*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the name of the filename that is to be searched	*/
/*		for. The variable 'entry' must contain the directory	*/
/*		entry that may or may not contain the filename.		*/
/*									*/
/* Output	If the filename is in the directory entry then this	*/
/*		function will return true. Otherwise, this function	*/
/*		will return false.					*/
/************************************************************************/

condition FTP::is_item_in_list(const String& filename,
   const String& entry)
   {
      condition		result		= false;
      register int	index;

      /* The filename is the last item in the directory entry.		*/
      /* Therefore, move backwards through the entry to isolate the	*/
      /* filename.							*/

      index		= entry.Length();
      while (entry.Data()[--index] != ' ' && index > 0);

      /* Compare the filename to the isolated filename.			*/

      if (filename.Compare(entry.Data() + index + 1) == EQUAL)
	 {
	    result	= true;
	 }
      return(result);
   }

/************************************************************************/
/* Function	void get_time_from_entry(const String& entry,		*/
/*		   String& timestring)					*/
/*									*/
/* Purpose	This function can be used to get a file's modification	*/
/*		time from the server's directory listing. The		*/
/*		modification time is returned as a string. The format	*/
/*		of the string is YYYYMMDDHHMMSS. YYYY is the year (e.g.	*/
/*		2002). MM is the month (e.g. 07). DD is the day of the	*/
/*		month (e.g. 08). HH is the hour in the day (e.g. 13).	*/
/*		MM is the minute in the hour (e.g. 04). Finally, SS is	*/
/*		the seconds in the minute (e.g. 50). An example of a	*/
/*		returned string is:					*/
/*									*/
/*		20020708132408						*/
/*									*/
/*		The example above is the date and time for July the 8,	*/
/*		2002 at 13:24:08.					*/
/*									*/
/* Input	This function expects the variable 'entry' to contain	*/
/*		the server's directory entry that contains the file's	*/
/*		modification time.					*/
/*									*/
/* Output	This function assumes that a valid directory string	*/
/*		has been passed to it in the variable 'entry'. This	*/
/*		function will return the file's modification time in	*/
/*		the variable 'timestring'.				*/
/************************************************************************/

void FTP::get_time_from_entry(const String& entry, String& timestring)
   {
      struct tm		temptime;
      register int	index;
      register int	index2;
      String		tempstring;
      char		buffer[1024];
      String		listitem	= entry;
      time_t		dt;
      struct tm		dc;

      timestring	= "";

      /* Towards the end of this function, a time function will be	*/
      /* called that requires the daylight savings time flag to be set.	*/
      /* The following code gets the system's setting and saves it.	*/

      time(&dt);
      dc		= *gmtime(&dt);
      temptime.tm_isdst	= dc.tm_isdst;

      /* Start searching for the time components by moving backwards	*/
      /* through the directory entry.					*/

      index		= listitem.Length();

      /* This loop will move the pointer past the filename.		*/

      while (listitem.Data()[--index] != ' ' && index > 0);
      listitem.Data()[index]	= '\0';

      /* The next part of the entry is either the hours and minutes or	*/
      /* it is the year.						*/

      while (listitem.Data()[--index] != ' ' && index > 0);
      listitem.Data()[index]	= '\0';
      tempstring		= listitem.Data() + index + 1;

      /* If the isolated substring contains the character ':' then it	*/
      /* represents the hours and minutes (e.g. 00:00). Otherwise, it	*/
      /* represents the year (e.g. 2002).				*/

      if (strchr(tempstring.Data(), ':') == (char*)0)
	 {
	    /* This is a year.				*/

	    temptime.tm_year	= atoi(tempstring.Data()) - 1900;
	    temptime.tm_sec		= 0;
	    temptime.tm_min		= 0;
	    temptime.tm_hour		= 0;
	 }
      else
	 {
	    /* This is an hour and minutes.				*/

	    temptime.tm_sec		= 0;
	    index2			= tempstring.Length();
	    while (tempstring.Data()[--index2] != ':' && index > 0);
	    tempstring.Data()[index2]	= '\0';
	    temptime.tm_min		= atoi(tempstring.Data() +
					  index2 + 1);
	    temptime.tm_hour		= atoi(tempstring.Data());
	    temptime.tm_year		= dc.tm_year;
        }

      /* Move past any spaces.						*/

      while (listitem.Data()[--index] == ' ' && index > 0);
      index++;

      /* Now get the day of the month.					*/

      while (listitem.Data()[--index] != ' ' && index > 0);
      listitem.Data()[index]	= '\0';
      tempstring		= listitem.Data() + index + 1;
      temptime.tm_mday		= atoi(tempstring.Data());

      /* Now get the month.						*/

      while (listitem.Data()[--index] != ' ' && index > 0);
      tempstring		= listitem.Data() + index + 1;
      for (index = 0; index < 12; index++)
         {
	    if (tempstring.Case_Compare(months[index]) == EQUAL)
	       {
	          temptime.tm_mon	= index;
		  break;
	       }
	 }

      /* Now build the time string.					*/

      strftime(buffer, 1024, "%Y%m%d%H%M%S", &temptime);
      timestring		= buffer;
   }

/************************************************************************/
/* Function	status get_modification_time(const String& path,	*/
/*		   String& time)					*/
/*									*/
/* Purpose	This function can be used to get a file's modification	*/
/*		time from the server's directory listing. The		*/
/*		modification time is returned as a string. The format	*/
/*		of the string is YYYYMMDDHHMMSS. YYYY is the year (e.g.	*/
/*		2002). MM is the month (e.g. 07). DD is the day of the	*/
/*		month (e.g. 08). HH is the hour in the day (e.g. 13).	*/
/*		MM is the minute in the hour (e.g. 04). Finally, SS is	*/
/*		the seconds in the minute (e.g. 50). An example of a	*/
/*		returned string is:					*/
/*									*/
/*		20020708132408						*/
/*									*/
/*		The example above is the date and time for July the 8,	*/
/*		2002 at 13:24:08.					*/
/*									*/
/* Input	This function expects the variable 'path' to contain	*/
/*		the file's path (relative to the server's base address).*/
/*									*/
/* Output	If this function is able to get the file's modification	*/
/*		time from the server's directory listing then this	*/
/*		function will return OK and the variable 'time' will	*/
/*		contain the time string. If the file does not exist	*/
/*		then the variable 'time' will contain an empty string.	*/
/*		If this function was not able to get the file's		*/
/*		modification time from the server's directory listing	*/
/*		then this function will return ERROR. All errors by	*/
/*		this function are reported to stderr.			*/
/************************************************************************/

status FTP::get_modification_time(const String& path, String& time)
   {
      status		result		= OK;
      static String	current;
      static int	items;
      static String*	list;
      register int	index;
      String		filename;
      String		dir		= path;

      /* Move backwards through the path to isolate the directory.	*/

      index		= dir.Length();
      while (dir.Data()[--index] != '/' && index > 0);
      if (index == 0)
         {
	    /* The path only contained a filename so use base dir.	*/

	    dir			= "/";
	 }
      else
         {
	    /* We found the directory so remove the filename.		*/

	    dir.Data()[++index]	= '\0';
	 }

      /* If 'itschangedirflag' == true OR the current loaded directory	*/
      /* is not the same as the directory that we need...		*/

      if (itschangedirflag == true || current.Compare(dir) != EQUAL)
         {
	    /* Get the server's directory listing for our directory.	*/

	    result	= Get_Directory(dir, items, list);
	    if (result == OK)
	       {
	          /* Make the loaded directory the current directory.	*/

	          current		= dir;
		  itschangedirflag	= false;
	       }
	 }

      /* Isolate the filename from the path.				*/

      index		= path.Length();
      while (path.Data()[--index] != '/' && index > 0);
      if (index == 0)
         {
	    /* The path consisted of a filename only.			*/

	    filename	= path;
	 }
      else
         {
	    /* Skip the information in the file's path.			*/

	    filename	= path.Data() + index + 1;
	 }

      /* Search through the directory list, looking for the file.	*/

      for (index = 0; index < items && result == OK; index++)
	 {
	    if (is_item_in_list(filename, list[index]) == true)
	       {
	          /* The filename was found in the directory listing so	*/
		  /* get the modification time using this directory	*/
		  /* entry.						*/

	          get_time_from_entry(list[index], time);
		  break;
	       }
	 }
      return(result);
   }
