"======================================================================
|
|   Smalltalk TCP/IP sockets - IPAddress class
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Steve Byrne and Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================
"

Object subclass: #IPAddress
       instanceVariableNames: 'hostName address'
       classVariableNames: 'AnyLocalAddress LocalHostName LoopbackHost UnknownAddress Cache'
       poolDictionaries: ''
       category: 'Sockets-Protocols'
! 

!IPAddress class methodsFor: 'initialization'!

flush
    | localAddrBytes |
    LocalHostName := self primLocalName.

    self
        primAnyLocalAddress: LocalHostName
        in: (localAddrBytes := ByteArray new: 4).
    AnyLocalAddress := IPAddress fromBytes: localAddrBytes.

    LoopbackHost := IPAddress fromBytes: #(127 0 0 1).
    LoopbackHost name: 'localhost'.

    UnknownAddress := IPAddress fromBytes: (ByteArray new: 4).
    UnknownAddress name: '0.0.0.0'.

    Cache := Dictionary new.
!

onStartup
    self flush
! !

!IPAddress class methodsFor: 'constants'!

addressSize
    ^4
!

anyLocalAddress
    ^AnyLocalAddress
!

loopbackHost
    ^LoopbackHost		"127.0.0.1"
!

unknownAddress
    ^UnknownAddress		"0.0.0.0"
!

version
    ^4
! !

!IPAddress class methodsFor: 'host name lookup'!

allByName: aString
    | host |
    (self isDigitAddress: host) ifTrue: [
        ^Cache at: host ifAbsentPut: [ self fromString: host ]
    ].

    host := aString asLowercase.

    ^Cache at: host ifAbsentPut: [
        | result index addresses addr |
        result := self lookupAllHostAddr: host.
        result := CByteType new: result.		"typecast it"
        addresses := WriteStream on: (Array new: 1).

        index := 0.
        [   addr := self
                with: (result at: index)
                with: (result at: index + 1)
                with: (result at: index + 2)
                with: (result at: index + 3).

            index := index + 4.
            addr = UnknownAddress
        ]   whileFalse: [ addresses nextPutAll: addr ].

        addresses := addresses contents.
        addresses isEmpty ifTrue: [ addresses := Array with: UnknownAddress ].
        result free.
        addresses
    ]
!

byName: aString
    ^aString isEmpty
        ifTrue: [ LoopbackHost ]
        ifFalse: [ (self allByName: aString) anyOne ]
! !

!IPAddress class methodsFor: 'instance creation'!

fromBytes: aByteArray
    ^self basicNew
        address: (aByteArray copy makeReadOnly: true)
!

fromSockAddr: aByteArray port: portAdaptor
    | s |
    s := CSockAddrStruct sizeof.
    portAdaptor value:
        (aByteArray at: s - 13) * 256 + (aByteArray at: s - 12).
    
    ^self fromBytes: (aByteArray copyFrom: s - 11 to: s - 8)
!

fromString: aString
    | substrings result |
    substrings := aString substrings: $. .
    result := ByteArray new: 4.
    1 to: result size do: [ :i |
        result at: i put: substrings asInteger
    ].
    ^self fromBytes: result
!

new
    self shouldNotImplement
!

with: b1 with: b2 with: b3 with: b4
    ^self basicNew address:
        ((ByteArray with: b1 with: b2 with: b3 with: b4) makeReadOnly: true)
! !

!IPAddress class methodsFor: 'private'!

isDigitAddress: aString
    | dots |
    dots := 0.
    (aString substrings: $.) do: [ :part |
        dots := dots + 1.
        (part allSatisfy: [ :each | each isDigit ])
            ifFalse: [ ^false ].

        part asInteger > 255 ifTrue: [ ^false ].
    ].
    ^dots = 4
! !


!IPAddress methodsFor: 'accessing'!

= anIPAddress
    "The host name is not checked. See name to understand why!"
    ^(self class == anIPAddress class) and: [
        self asByteArray == anIPAddress asByteArray
    ]
!

asByteArray
    ^address
!

hash
    ^address hash
!

host
    ^address at: 2
!

imp
    ^(address at: 3) * 256 + (address at: 4)
!

impNo
    ^address at: 4
!

isMulticast
    ^self net between: 224 and: 239
!

logicalHost
    ^address at: 3
!

name
    | addresses |
    hostName isNil ifFalse: [ ^hostName ].
    hostName := self class primName: self asByteArray.

    addresses := Cache at: hostName ifAbsentPut: [ Array with: self ].

    addresses do: [ :each |
        (each hostName = hostName) & (each ~= self) ifTrue: [
            "Seems that someone's joking with the DNS server
             and changed this host's IP address even though the
             name stays the same. Don't cache the name and don't
             even give away an alphanumeric name"
            ^hostName := self printString
        ].
    ].
    ^hostName
!

net
    ^self at: 1
!

port: port
    "Return a ByteArray containing a struct sockaddr for the given port
     on the IP address represented by the receiver. Family = AF_INET."

    ^(ByteArray new: CSockAddrStruct sizeof)
        
        "Write sin_addr"
        replaceFrom: CSockAddrStruct sizeof - 11
        to: CSockAddrStruct sizeof - 8
        with: address
        startingAt: 1;

        "Write sin_family = AF_INET in host order"
        shortAt: 1 put: self class afInet;

        "Write sin_port in network order (big endian)"
        at: CSockAddrStruct sizeof - 13 put: port \\ 256;
        at: CSockAddrStruct sizeof - 12 put: (port bitAnd: 255);

        "ouf..."
        yourself
! !

!IPAddress methodsFor: 'printing'!

printOn: aStream

    (address at: 1) printOn: aStream.
    2 to: address size do: [ :i |
        aStream nextPut: $.; print: (address at: i) 
    ].
! !

!IPAddress methodsFor: 'private'!

address: aByteArray
    address := aByteArray
!

name: name
    hostName := name
! !

