"======================================================================
|
|   Smalltalk in Smalltalk compiler
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"


Object subclass: #STFakeCompiler
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: 'VMOtherConstants VMByteCodeNames'
       category: 'System-Compiler'
! 

STFakeCompiler comment:
'I am the Smalltalk equivalent of a wimp. I never do nothing: they tell me
to compile something, and I just return nil...

Actually, I am used when conditionally compiled code has to be skipped.'!

!STFakeCompiler class methodsFor: 'compilation'!

compile: methodDefNode for: aBehavior classified: aString parser: aParser
    ^nil
! !

STFakeCompiler subclass: #STCompiler
       instanceVariableNames: 'node symTable parser bytecodes depth maxDepth'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'System-Compiler'
! 

STCompiler comment:
'Unlike my brother STFakeCompiler, I am a real worker. Give me some nodes, and
I will output a full-fledged CompiledMethod!!

But seriously... STCompiler''s responsibility is much limited. All its #compile
method does is to tell each statement to compile itself, adding a ''pop stack
top'' bytecode at the end of each statement. In addition, STCompiler mantains an
interface between itself, the symbol table and the bytecodes stream.
The need to mantain such an interface and pass the compiler object around
on every compileFor: call is dictated by the need to make the compiler
reentrant. This way, the user can nest fileIns to any level.

The actual compilation policy does not reside in STCompiler, but it is spread
through subclasses of STParseNode. This may look clumsy, but actually it allows
to use polymorphism, which is the fastest dispatching method of all, and produces
very streamlined code. Thus we have elegant and fast code... not bad!

For example, when we send the ''true printOn: stdout'' message, the structure
of the tree is:
    MessageNode, which contains:
        the receiver, an IdentifierNode (in this case a SpecialIdentifierNode)
        the message, a kind of MessageSendNode (a KeywordNode), which contains
             the selector
             the parameters, a Collection which contains a VariableNode

The MessageNode checks if the receiver is super. If so, it tells the message
to compile itself as a send to super. In this case however it tells both the
receiver and the message to compile themselves.
The SpecialIdentifierNode will output a ''push true'' bytecode.
The MessageSendNode, in turn, asks the parameters to compile themselves, asks
the STCompiler object to add the #printOn: literal (this request is delegated
to an instance of STSymTable), then compiles a ''send message'' bytecode.
The VariableNode which refers to stdout, when it is asked to compile itself,
tells the STCompiler object to add a literal (since it refers to a global
variable) and then compiles either a ''push global variable'' or a ''push
indexed'' bytecode. The resulting stream is

        push true
        push literal corresponding to (#stdout -> stdout)
        send message with 0 args, selector = #printOn:

This is possible by providing consistent protocols. Here they are:
- ParseNodes respond to #compileFor: and #compileReturnFor:. The former is
  abstract, while the latter''s default implementation calls #compileFor: and
  appends a ''return stack top from method'' bytecode.

- IdentifierNodes, which might appear on the left of an assignment, under-
  stand #compileAssignmentFor: too.

- MessageSendNode implements the compilation of message sends, including
  optimizations when compiling while loops and boolean expressions. For it
  to work properly, its subclasses (UnaryNode, BinaryNode, KeywordNode)
  implement #argumentCount and #allExpressionsDo:

- MessageNode implements #compileFor:dupReceiver:, which adds if necessary
  a ''duplicate stack top'' bytecode after having compiled the receiver, and
  returns whether the MessageNode identified a send to ''super''. This message
  is used by CascadeNodes.
'!

!STCompiler class methodsFor: 'compilation'!

compile: methodDefNode for: aBehavior classified: aString parser: aParser
    | cm |
    cm := self new
        setNode: methodDefNode class: aBehavior parser: aParser;
        compile.

    cm methodCategory: aString.

    ^aBehavior
        addSelector: (methodDefNode selector) selector asSymbol
        withMethod: cm
! !

!STCompiler methodsFor: 'compilation'!

compile
    | body lastIsReturn method |
    body := node body.

    node selector args do: [ :anArg |
	symTable
	    declareTemporary: anArg
	    isArgument: true
	    for: self
    ].
    body temporaries do: [ :aTemp |
        symTable
            declareTemporary: aTemp
            isArgument: false
            for: self
    ].

    depth := maxDepth := node selector args size + body temporaries size.

    body statements doWithIndex: [ :each :index |
        index = 1 ifFalse: [
	    self depthDecr: 1.
	    self nextPut: PopStackTop
	].
        each compileFor: self.
        lastIsReturn := each isReturn.
    ].

    body statements isEmpty
        ifTrue: [ self nextPut: ReturnSelf ]
        ifFalse: [ lastIsReturn ifFalse: [ self nextPut: self finalReturn ] ].

    method := CompiledMethod
        literals: symTable literals
        numArgs: node selector args size
        numTemps: symTable numTemps - node selector args size
        primitive: body primitive
        bytecodes: bytecodes contents
        depth: depth.

    method getDescriptor setSourceCode: node source.
    ^method
! !

!STCompiler methodsFor: 'private'!

finalReturn
    ^ReturnSelf
!

setNode: methodDefNode class: aBehavior parser: aParser
    node := methodDefNode.
    symTable := STSymbolTable new.
    parser := aParser.
    bytecodes := WriteStream on: (ByteArray new: 240).

    symTable declareEnvironment: aBehavior.
! !


!STCompiler methodsFor: 'accessing'!

addLiteral: literal
    ^symTable addLiteral: literal
!

bytecodeSize
    ^bytecodes position
!

bytecodesFor: aBlockNode
    | saveBytecodes result |
    saveBytecodes := bytecodes.
    bytecodes := WriteStream on: (ByteArray new: 240).
    aBlockNode compileStatementsFor: self.

    result := bytecodes contents.
    bytecodes := saveBytecodes.
    ^result
!

checkStore: aVariableName
    (symTable canStore: aVariableName) ifFalse: [
        self compileError: 'cannot store in argument ', aVariableName
    ]
!

compileBigLiteral: op index: index
    self
        nextPut: BigLiteral;
        nextPut: index // 256 + op;
        nextPut: (index bitAnd: 255)
!

compileError: aString
    parser parseError: aString
!

declareTemporaries: parameters numArguments: numArgs
    | result |
    parameters doWithIndex: [ :each :index |
        result := symTable
            declareTemporary: each
            isArgument: index <= numArgs
            for: self
    ].
    ^result
!

depthDecr: n
    depth := depth - n
!

depthIncr
    depth = maxDepth
        ifTrue: [ depth := depth + 1. maxDepth := maxDepth + 1 ]
        ifFalse: [ depth := depth + 1 ]
!

depthSet: n
    | oldDepth |
    oldDepth := n.
    ^(depth := n) > maxDepth
        ifTrue: [ maxDepth := depth. oldDepth ]
        ifFalse: [ oldDepth ]
!

isReceiver: variable
    ^symTable isReceiver: variable
!

isTemporary: variable
    ^symTable isTemporary: variable
!

lookupName: variable
    | definition |
    definition := symTable lookupName: variable.

    definition isNil ifTrue: [
        "Might want to declare this puppy as a local and go on
         notwithstanding the error"

         self compileError: 'Undefined variable ', 
             variable printString, 'referenced.'
    ].
    ^definition
!

nextPut: aByte
    bytecodes nextPut: aByte.
!

nextPutAll: aByteArray
    bytecodes nextPutAll: aByteArray.
!

outerScopes: variable
    ^symTable outerScopes: variable
!

scopeEnter
    symTable scopeEnter
!

scopeLeave
    symTable scopeLeave
!

undeclareTemporaries: temps
    temps do: [ :each | symTable undeclareTemporary: each ]
! !

STCompiler subclass: #STDoitCompiler
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'System-Compiler'
! 

STDoitCompiler comment:
'Like my father STCompiler, I am a real worker, not a do-nothing compiler.
Unlike it, I always interpret as a return the last statement I cope with.'
!


!STDoitCompiler methodsFor: 'private'!

finalReturn
    ^ReturnMethodStackTop
! !

"--------------------------------------------------------------------"
STBlockNode comment:
'STBlockNode has a variable that contains a string for each parameter,
and one that contains a list of statements. Here is how STBlockNodes
are compiled:

        push active context
        push number of parameters  (if 0 to 2, ''push special'' is used)
        push number of temporaries (if 0 to 2, ''push special'' is used)
        send #blockCopy:
        jump to @@1
        <<statements go here>>
        return from block, result = stack top
    @@1:<<stack top contains BlockContext answered by #blockCopy:>>
    
compileStatementsFor: compiles everything after the jump and before the
return. It is this method that is called by STCompiler>>bytecodesFor:'!

!STBlockNode methodsFor: 'compiling'!

compileStatementsFor: stCompiler

    stCompiler
	declareTemporaries: self parameters
	numArguments: self numArgs.

    self statements doWithIndex: [ :each :index |
	index = 1 ifFalse: [
	    stCompiler
		depthDecr: 1;
		nextPut: PopStackTop
	].
        each compileFor: stCompiler.
    ].
    self statements isEmpty ifTrue: [
        stCompiler
            depthIncr;
            nextPut: PushNil
    ].
    stCompiler undeclareTemporaries: self parameters
!

compileFor: stCompiler
    | bc depth |
    stCompiler nextPut: PushActiveContext.
    self numArgs <= 2
        ifTrue: [
            stCompiler nextPut: PushSpecial + self numArgs + 5
        ]
        ifFalse: [
            self
                compilePushLiteral: (stCompiler addLiteral: self numArgs)
                for: stCompiler
        ].

    self numTemps <= 2
        ifTrue: [
            stCompiler nextPut: PushSpecial + self numTemps + 5
        ]
        ifFalse: [
            self
                compilePushLiteral: (stCompiler addLiteral: self numTemps)
                for: stCompiler
        ].

    depth := stCompiler depthSet: self numArgs + self numTemps.
    stCompiler nextPut: BlockCopyColonSpecial.
    stCompiler scopeEnter.
    bc := stCompiler bytecodesFor: self.
    stCompiler scopeLeave.

    self compileJump: bc size + 1 to: stCompiler if: nil.

    stCompiler
        depthSet: depth;
        nextPutAll: bc;
        nextPut: ReturnBlockStackTop.
! !

!STBlockNode methodsFor: 'private'!

compileParameterPop: location for: stCompiler

    location <= 7
        ifTrue: [
	    stCompiler nextPut: (PopTemporaryVariable + location).
	]
        ifFalse: [   
            stCompiler
                nextPut: PopStoreIndexed;
	        nextPut: (TemporaryLocation + location)
	]
!

compilePushLiteral: location for: stCompiler

    location <= 31
        ifTrue: [
	    stCompiler nextPut: (PushLitConstant + location).
	]
        ifFalse: [   
            stCompiler
                nextPut: PushIndexed;
	        nextPut: (LiteralConstantLocation + location)
	]
! !


"--------------------------------------------------------------------"
STCascadeNode comment:
'STCascadeNode has two variable: the receiver of the cascaded messages
and a collection with one item per message.'!

!STCascadeNode methodsFor: 'compiling'!

compileFor: stCompiler
    | toSuper |

    toSuper := receiver compileFor: stCompiler dupReceiver: true.

    toSuper ifTrue: [
        messages doWithIndex: [ :each :index |
	    stCompiler depthDecr: 1; nextPut: PopStackTop.
            each compileSendToSuperFor: stCompiler.
        ].
        ^self
    ].
    messages doWithIndex: [ :each :index |
        stCompiler nextPut: PopStackTop.
        index = messages size
            ifFalse: [ stCompiler nextPut: DupStackTop ]
            ifTrue: [ stCompiler depthDecr: 1 ].

        each compileFor: stCompiler.
    ].
! !

"--------------------------------------------------------------------"
STConstNode comment:
'STConstNode has one instance variable, the literal it represents.'!

!STConstNode methodsFor: 'compiling'!

compileFor: stCompiler
    | definition |

    stCompiler depthIncr.

    VMSpecialIdentifiers at: self value ifPresent: [ :encodedValue |
        stCompiler nextPut: PushSpecial + encodedValue.
        ^self
    ].

    definition := stCompiler addLiteral: self value.

    definition <= 31 ifTrue: [
        stCompiler nextPut: (PushLitConstant + definition).
	^self
    ].    
    
    definition > 63 ifTrue: [
	stCompiler compileBigLiteral: PushLiteral index: definition.
        ^self
    ].
    stCompiler
        nextPut: PushIndexed;
	nextPut: (LiteralConstantLocation + definition)
! !

"--------------------------------------------------------------------"
STExpressionNode comment:
'STExpressionNode has two instance variables: a) another STParseNode which
 corresponds to the value to be evaluated, and b) a list of variables which
 have to be set to the result of the expression.'!

!STExpressionNode methodsFor: 'compiling'!

compileFor: stCompiler

    self expression compileFor: stCompiler.
    self assignsDo: [ :assign | 	"Iteration order doesn't matter here"
	assign compileAssignmentFor: stCompiler.
    ].
! !

"--------------------------------------------------------------------"
STIdentifierNode comment:
'STIdentifierNode defines a few abstract methods.'!

!STIdentifierNode methodsFor: 'compiling'!

compileAssignmentFor: stCompiler
    self subclassResponsibility
! !

"--------------------------------------------------------------------"
STMessageNode comment:
'STMessageNode contains a message send. Its instance variable are
a receiver, and a STMessageSendNode.'!

!STMessageNode methodsFor: 'compiling'!

compileFor: stCompiler
    self compileFor: stCompiler dupReceiver: false
!

compileFor: stCompiler dupReceiver: dup
    
    self receiver isSuper ifTrue: [
        message compileSendToSuperFor: stCompiler.
        ^true
    ].
    (self receiver isBlock & dup not and: [
        VMBlockSelectors includes: self message selector])
            ifTrue: [ 
                (message compileFor: stCompiler whileLoop: receiver) ifTrue: [^self]
                "If the send could not be optimized, proceed normally"
            ].

    receiver compileFor: stCompiler.
    dup ifTrue: [ stCompiler depthIncr; nextPut: DupStackTop ].
    message compileFor: stCompiler.
    ^false
! !

"--------------------------------------------------------------------"
STMessageSendNode comment:
'STMessageSendNode is at the top of a small hierarchy of
STParseNodes. It contains code to compile message sends (including
optimized selectors) and relies on a few abstract methods defined by
STUnaryNode, STBinaryNode and STKeywordNode use these implementations.'!

!STMessageSendNode methodsFor: 'compiling'!

compileFor: stCompiler
    | args litIndex |
    (VMBoolSelectors includes: self selector) ifTrue: [
        (self compileBooleanFor: stCompiler) ifTrue: [ ^self ]
    ].
    (VMLoopSelectors includes: self selector) ifTrue: [
        (self compileLoopFor: stCompiler) ifTrue: [ ^self ]
    ].

    self allExpressionsDo: [ :each | each compileFor: stCompiler ].

    VMSpecialMethods at: self selector ifPresent: [ :idx |
        stCompiler nextPut: SendSpecial + idx.
        ^self
    ].
    args := self argumentCount.
    litIndex := stCompiler addLiteral: self selector.
    (args <= 2) & (litIndex <= 15)
        ifTrue: [
            stCompiler nextPut: SendSelectorShort + (args * 16) + litIndex
        ]
        ifFalse: [
            self
                emitExtendedSendBytecodesTo: stCompiler
                toSuperFlag: 0
                literalIndex: litIndex
                argCount: args
        ].

    stCompiler depthDecr: self argumentCount.
!

compileFor: stCompiler whileLoop: whileBlock
    "Answer whether the while loop can be optimized (that is,
     whether the only parameter is a STBlockNode)"

    | whileBytecodes argBytecodes totBytecodesSize |

    (whileBlock isOptimizableWithArgs: 0) ifFalse: [ ^false ].
    self allExpressionsDo: [ :onlyArgument |
        onlyArgument isBlock ifFalse: [ ^false ].
        (onlyArgument isOptimizableWithArgs: 0) ifFalse: [ ^false ].
        argBytecodes := stCompiler bytecodesFor: onlyArgument
    ].
    whileBytecodes := stCompiler bytecodesFor: whileBlock.

    totBytecodesSize := whileBytecodes size + argBytecodes size
        + self sizeOfJump + self sizeOfJump.

    stCompiler nextPutAll: whileBytecodes.

    "The if: clause means: if selector is whileFalse:, compile
     a 'pop/jump if true'; else compile a 'pop/jump if false'"

    self
        compileJump: argBytecodes size + self sizeOfJump
        to: stCompiler
        if: (self selector == #whileFalse:).

    stCompiler nextPutAll: argBytecodes.

    self
        compileJump: totBytecodesSize negated
        to: stCompiler
        if: nil.

    "Somebody might want to use the return value of #whileTrue:
     and #whileFalse:"

    stCompiler depthIncr; nextPut: PushNil.
    ^true
!

compileSendToSuperFor: stCompiler

    stCompiler depthIncr; nextPut: PushSpecial.
    self allExpressionsDo: [ :each | each compileFor: stCompiler ].
    self
        emitExtendedSendBytecodesTo: stCompiler
        toSuperFlag: 2
        literalIndex: (stCompiler addLiteral: self selector)
        argCount: self argumentCount.

    stCompiler depthDecr: self argumentCount.
! !

!STMessageSendNode methodsFor: 'private'!

compileLoopFor: stCompiler
    | stop step block |
    self allExpressionsDo: [ :each |
	stop := step.			"to:"
	step := block.			"by:"
	block := each.			"do:"
    ].
    step isNil ifTrue: [ 					"#timesRepeat:"
	(block isOptimizableWithArgs: 0) ifFalse: [ ^false ].
	^false
    ].

    (block isOptimizableWithArgs: 1) ifFalse: [ ^false ].
    stop isNil
	ifTrue: [ stop := step. step := STConstNode one ]	"#to:do:"
	ifFalse: [ step isToByDoStep ifFalse: [ ^false ] ].	"#to:by:do:"

    ^false
!

compileBooleanFor: stCompiler
    | bc1 bc2 |
    self allExpressionsDo: [ :each |
        (each isOptimizableWithArgs: 0) ifFalse: [ ^false ].
        bc1 isNil
            ifTrue: [ bc1 := each ]
            ifFalse: [ bc2 := stCompiler bytecodesFor: each ].
    ].
    bc1 := stCompiler bytecodesFor: bc1.

    self selector == #ifTrue:ifFalse: ifTrue: [
        ^self compileFor: stCompiler ifTrue: bc1 ifFalse: bc2
    ].
    self selector == #ifFalse:ifTrue: ifTrue: [
        ^self compileFor: stCompiler ifTrue: bc2 ifFalse: bc1
    ].
    self selector == #ifTrue: ifTrue: [
        ^self compileFor: stCompiler ifTrue: bc1 ifFalse: #(115) "Push nil"
    ].
    self selector == #ifFalse: ifTrue: [
        ^self compileFor: stCompiler ifTrue: #(115) ifFalse: bc1
    ].
    self selector == #and: ifTrue: [
        ^self compileFor: stCompiler ifTrue: bc1 ifFalse: #(114) "Push false"
    ].
    self selector == #or: ifTrue: [
        ^self compileFor: stCompiler ifTrue: #(113) ifFalse: bc1 "Push true"
    ].
    ^false "What happened?!?"
!

compileFor: stCompiler ifTrue: bcTrue ifFalse: bcFalse

    self
        compileJump: bcTrue size + self sizeOfJump
        to: stCompiler
        if: false.

    stCompiler nextPutAll: bcTrue.
    
    self
        compileJump: bcFalse size
        to: stCompiler
        if: nil.
    
    stCompiler nextPutAll: bcFalse.
    ^true
!

emitExtendedSendBytecodesTo: stCompiler
    toSuperFlag: toSuperFlag
    literalIndex: litIndex
    argCount: args

    (args <= 7) & (litIndex <= 31)
        ifTrue: [
            stCompiler
                nextPut: SendSelector1ExtByte + toSuperFlag;
                nextPut: args * 32 + litIndex
        ]
        ifFalse: [
            stCompiler
                nextPut: SendSelector2ExtByte + toSuperFlag;
                nextPut: args;
                nextPut: litIndex
        ]
! !

"--------------------------------------------------------------------"
STParseNode comment:
'STParseNode defines a few abstract methods and the default policy
 for compiling returns.'!

!STParseNode methodsFor: 'compiling'!

compileFor: stCompiler
    self subclassResponsibility
!

compileReturnFor: stCompiler
    self expression compileFor: stCompiler.
    stCompiler nextPut: ReturnMethodStackTop
! !

!STParseNode methodsFor: 'private'!

sizeOfJump
    "For simplicity, I don't use short jump bytecodes."
    ^2
!

compileJump: displacement to: stCompiler if: jmpCondition

    jmpCondition isNil ifTrue: [
        "Unconditional"
        ^stCompiler
            nextPut: JumpLong + (((displacement + 1024) bitShift: -8) bitAnd: 7);
            nextPut: (displacement bitAnd: 255).
    ].
    displacement < 0 ifTrue: [
         "Should not happen"
         ^self error: 'Cannot compile backwards conditional jumps'.
    ].
    jmpCondition
        ifFalse: [
            stCompiler
                depthDecr: 1;
                nextPut: PopJumpFalse + ((displacement bitShift: -4) bitAnd: 3)
        ]
        ifTrue: [
            stCompiler
                depthDecr: 1;
                nextPut: PopJumpTrue + ((displacement bitShift: -4) bitAnd: 3)
        ].
    stCompiler
        nextPut: (displacement bitAnd: 255)
! !

"--------------------------------------------------------------------"
STReturnNode comment:
'STReturnNode has one instance variable, that is another STParseNode
 which compiles to the value to be returned.'!

!STReturnNode methodsFor: 'compiling'!

compileFor: stCompiler

    self expression compileReturnFor: stCompiler
! !

"--------------------------------------------------------------------"
STSpecialIdentifierNode comment:
'STSpecialIdentifierNode has one instance variable, an id of the
pseudo-variable that it represents.'!

!STSpecialIdentifierNode methodsFor: 'compiling'!

compileFor: stCompiler
    | bytecodeID |
    bytecodeID := self isSuper
        ifTrue: [ 0 ]
        ifFalse: [ id ].

    stCompiler depthIncr; nextPut: bytecodeID + PushSpecial.
!

compileReturnFor: stCompiler
    | bytecodeID |
    bytecodeID := self isSuper
        ifTrue: [ 0 ]
        ifFalse: [ id ].

    bytecodeID <= 3
        ifTrue: [ stCompiler nextPut: bytecodeID + ReturnSpecial ]
        ifFalse: [ super compileReturnFor: stCompiler ].
!

compileAssignmentFor: stCompiler
    stCompiler compileError: 'cannot assign to ',
        (VMSpecialIdentifiers keyAtValue: id)
! !

"--------------------------------------------------------------------"
STVariableNode comment:
'STVariableNode has one instance variable, the name of the variable
that it represents.'!

!STVariableNode methodsFor: 'compiling'!

compileAssignmentFor: stCompiler
    | locationType definition |

    definition := stCompiler lookupName: self id.
    locationType := LiteralVariableLocation.

    (stCompiler isTemporary: self id) ifTrue: [
        stCompiler checkStore: self id.
        ^self
            compileStoreTemporary: definition
            scopes: (stCompiler outerScopes: self id)
            for: stCompiler
    ].    

    (stCompiler isReceiver: self id) ifTrue: [
	locationType := ReceiverLocation.
	definition <= 7 ifTrue: [
	    stCompiler nextPut: DupStackTop.
	    stCompiler nextPut: (PopReceiverVariable + definition).
	    ^self
	]
    ].
    
    definition > 63 ifTrue: [
	stCompiler compileBigLiteral: StoreVariable index: definition.
        ^self
    ].
    stCompiler
        nextPut: StoreIndexed;
	nextPut: (locationType + definition)
!

compileFor: stCompiler
    | locationType definition |

    stCompiler depthIncr.
    definition := stCompiler lookupName: self id.
    locationType := LiteralVariableLocation.

    (stCompiler isTemporary: self id) ifTrue: [
        ^self
            compilePushTemporary: definition
            scopes: (stCompiler outerScopes: self id)
            for: stCompiler
    ].    
    (stCompiler isReceiver: self id) ifTrue: [
	locationType := ReceiverLocation.
	definition <= 15 ifTrue: [
	    stCompiler nextPut: (PushReceiverVariable + definition).
	    ^self
	]
    ].
    definition <= 31 ifTrue: [
	stCompiler nextPut: (PushLitVariable + definition).
        ^self
    ].
    definition > 63 ifTrue: [
	stCompiler compileBigLiteral: PushVariable index: definition.
        ^self
    ].
    
    stCompiler
        nextPut: PushIndexed;
	nextPut: (locationType + definition)
!

compilePushTemporary: number scopes: outerScopes for: stCompiler
    outerScopes = 0 ifFalse: [
        stCompiler
            nextPut: OuterVar;
            nextPut: number + PushVariable;
            nextPut: outerScopes.
        ^self
    ].
    number < 16 ifTrue: [
        stCompiler
            nextPut: PushTemporaryVariable + number.
        ^self
    ].
    stCompiler
        nextPut: PushIndexed;
        nextPut: (TemporaryLocation + number)
!

compileStoreTemporary: number scopes: outerScopes for: stCompiler
    outerScopes = 0 ifFalse: [
        stCompiler
            nextPut: OuterVar;
            nextPut: number + StoreVariable;
            nextPut: outerScopes.
    ].
    stCompiler
        nextPut: StoreIndexed;
        nextPut: (TemporaryLocation + number)
! !

