"======================================================================
|
|   Smalltalk Tk-based GUI building blocks (basic widget classes).
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Steve Byrne.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

"-------------------------- Edit boxes ---------------------------"

BPrimitive subclass: #BEdit
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BEdit comment: '
I am a control showing modifiable text.'!

BEdit
    defineColorProtocolWithActivePrefix: 'select'!

!BEdit class methodsFor: 'instance creation'!

new: parent contents: aString
    ^(self new: parent)
        contents: aString;
        yourself
! !

!BEdit methodsFor: 'accessing'!

contents
    self tclEval: 'return ${var', self connected, '}'.
    ^self tclResult
!

contents: newText
    self tclEval: 'set var', self connected, ' ', newText asTkString
! !

!BEdit methodsFor: 'widget protocol'!

destroyed
    self tclEval: 'unset var', self connected.
    super destroyed.
! !

!BEdit methodsFor: 'private'!

create
    super create.
    self tclEval: ('
        set var%1 {}
        %1 configure -textvariable var%1 -takefocus 1'
        bindWith: self connected).
!

widgetType
    ^'entry '
! !


"---------------------------- Labels -----------------------------"

BPrimitive subclass: #BLabel
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BLabel comment: '
I am a label showing static text.'!

BLabel
    defineColorProtocolWithActivePrefix: 'active';
    defineAccessor: 'label' option: 'text' fromStringCode: '';
    defineMutator: 'label' option: 'text' toStringCode: ''!

!BLabel class methodsFor: 'instance creation'!

new: parent label: label
    ^(self new: parent)
        label: label;
        yourself
! !

!BLabel methodsFor: 'private'!

create
    self create: ' -anchor nw'
!

widgetType
    ^'label '
! !

"---------------------------- Buttons ----------------------------"

BPrimitive subclass: #BButton
       instanceVariableNames: 'callback'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BButton comment: '
I am a button that a user can click. In fact I am at the head
of a small hierarchy of objects which exhibit button-like look
and behavior'!

BButton
    defineColorProtocolWithActivePrefix: 'active';
    defineAccessor: 'label' option: 'text' fromStringCode: '';
    defineMutator: 'label' option: 'text' toStringCode: ''!


!BButton class methodsFor: 'instance creation'!

new: parent label: label
    ^(self new: parent)
        label: label;
        yourself
! !


!BButton methodsFor: 'accessing'!

callback: aReceiver message: aString
    callback := DirectedMessage
        selector: aString asSymbol
        arguments: #()
        receiver: aReceiver
! !

!BButton methodsFor: 'private'!

callback
    ^callback
!

create
    self create: (' -takefocus 1 -command {callback %1 invokeCallback}'
        bindWith: self asOop printString)
!

invokeCallback
    self callback isNil ifFalse: [ self callback send ]
!

widgetType
    ^'button '
! !

"---------------------------- Forms ------------------------------"

BPrimitive subclass: #BForm
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BForm comment: '
I am used to group many controls together. I leave the heavy
task of managing their position to the user.'!

BForm
    defineAccessor: 'backgroundColor' option: 'background' fromStringCode: '';
    defineMutator: 'backgroundColor' option: 'background' toStringCode: ''!

!BForm methodsFor: 'private'!

create
    self create: ' -takefocus 1'
!

init: parentWidget
    super init: parentWidget.
    parentWidget isNil ifFalse: [
        self backgroundColor: parentWidget backgroundColor
    ].
!

widgetType
    ^'frame '
! !

"---------------------------- Containers -------------------------"

BForm subclass: #BContainer
       instanceVariableNames: 'vertical'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BContainer comment: '
I am used to group many controls together. I can perform simple
management by putting controls next to each other, from left to
right or from top to bottom.'!


!BContainer methodsFor: 'accessing'!

setVerticalLayout: aBoolean
    children isEmpty ifFalse: [
        ^self error: 'cannot set layout after the first child is created'
    ].
    vertical := aBoolean
! !

!BContainer methodsFor: 'private'!

addChild: child
    | last |
    child isWindow ifFalse: [
        children isEmpty ifFalse: [
            last := children at: children size.
            vertical
                ifFalse: [ child posHoriz: last ]
                ifTrue:  [ child posVert:  last ]
        ].
        vertical
            ifFalse: [ child height: self height ]
            ifTrue:  [ child width:  self width  ]
    ].
    ^super addChild: child
!

init: parentWidget
    super init: parentWidget.
    vertical := true
! !


"------------------------------ Radio Groups ------------------------------"

BContainer subclass: #BRadioGroup
     instanceVariableNames: 'lastValue lastAssignedValue'
     classVariableNames: ''
     poolDictionaries: ''
     category: 'Graphics-Windows'
!

BRadioGroup comment: '
I am used to group many mutually-exclusive radio buttons together.
In addition, just like every BContainer I can perform simple management
by putting controls next to each other, from left to right or (which is
more useful in this particular case...) from top to bottom.'!


!BRadioGroup methodsFor: 'accessing'!

value
   self tclEval: 'return ${var', self connected, '}'.
   ^self tclResult asNumber
!

value: value
    self tclEval: 'set var', self connected, ' ', value printString
! !

!BRadioGroup methodsFor: 'widget protocol'!

destroyed
    self tclEval: 'unset var', self connected
    super destroyed.
! !


!BRadioGroup methodsFor: 'private'!

init: parentWidget
    super init: parentWidget.
    lastAssignedValue := lastValue := 0.
    self tclEval: 'set ', self variable, ' 1'.
!

lastValue
    ^lastValue
!

lastValue: value
    lastValue := value
!

newButtonValue
    ^lastAssignedValue := lastAssignedValue + 1
!

variable
    ^'var', self connected
! !


"------------------------------ Radio buttons ------------------------------"

BButton subclass: #BRadioButton
       instanceVariableNames: 'variableValue'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BRadioButton comment: '
I am just one in a group of mutually exclusive buttons.'!

BRadioButton
    defineMutator: 'variableValue' option: 'value' toStringCode: 'printString';
    defineMutator: 'variable' option: 'variable' toStringCode: ''!

!BRadioButton methodsFor: 'accessing'!

callback: aReceiver message: aString
    super callback: aReceiver message: aString.
    self callback arguments: (Array with: true)
!

value
    ^self parent value = variableValue
!

value: aBoolean
    aBoolean
        ifTrue: [ self parent value: variableValue ].

    "aBoolean is false - unhighlight everything if we're active"
    self value
        ifTrue: [ self parent value: 0 ]
! !


!BRadioButton methodsFor: 'private'!

init: parentWidget
    super init: parentWidget.
    variableValue := self parent newButtonValue.
    self
        tclEval: self connected, ' configure -anchor nw';
        variableValue: variableValue;
        variable: self parent variable;
        backgroundColor: parentWidget backgroundColor;
        activeBackground: parentWidget backgroundColor.

    variableValue = 1 ifTrue: [self parent value: 1]
!

widgetType
   ^'radiobutton '
! !

"------------------------------ Toggles ------------------------------------"

BButton subclass: #BToggle
       instanceVariableNames: 'value variableReturn'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BToggle comment: '
I represent a button whose choice can be included (by checking
me) or excluded (by leaving me unchecked).'!

BToggle
    defineMutator: 'variable' option: 'variable' toStringCode: ''!


!BToggle methodsFor: 'private'!

callback: aReceiver message: aString
    super callback: aReceiver message: aString.
    self callback arguments: (Array with: nil)
!

init: parentWidget
    | variable |
    super init: parentWidget.
    self tclEval: self connected, ' configure -anchor nw'.
    self tclEval: 'variable var', self connected.
    self variable: 'var', self connected.
    self backgroundColor: parentWidget backgroundColor.
    self activeBackground: parentWidget backgroundColor
!

invokeCallback
    self callback arguments at: 1 put: self value.
    super invokeCallback
!

value
   self tclEval: 'return ${var', self connected, '}'.
   ^self tclResult = '1'
!

widgetType
   ^'checkbutton '
! !

"---------------------------- Images ----------------------------"

BPrimitive subclass: #BImage
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BImage comment: '
I can display colorful images.'!

BImage
    defineColorProtocolWithActivePrefix: 'active';
    defineAccessor: 'gamma' command: 'img%1 cget -gamma'
        key: #gamma fromStringCode: 'asNumber';

    defineMutator: 'gamma' command: 'img%1 configure -gamma %3'
        key: #gamma toStringCode: 'asFloat printString';

    defineAccessor: 'displayWidth' command: 'img%1 cget -width'
        key: #displayWidth fromStringCode: 'asNumber';

    defineMutator: 'displayWidth' command: 'img%1 configure -width %3'
        key: #displayWidth toStringCode: 'asFloat printString';

    defineAccessor: 'displayHeight' command: 'img%1 cget -width'
        key: #displayHeight fromStringCode: 'asNumber';

    defineMutator: 'displayHeight' command: 'img%1 configure -width %3'
        key: #displayHeight toStringCode: 'asFloat printString'!

!BImage class methodsFor: 'instance creation'!

new: parent size: aPoint
    ^(self new: parent)
        imageWidth: aPoint x;
        imageHeight: aPoint y;
        blank;
        yourself
!

new: parent image: aFileStream
    ^(self new: parent)
        image: aFileStream;
        yourself
! !

!BImage methodsFor: 'image management'!

blank
    self tclEval: 'img', self connected, ' blank'
!

dither
    self tclEval: 'img', self connected, ' redither'
!

fillRectangle: rectangle color: color
    self fillFrom: rectangle origin to: rectangle corner color: color
!

fillFrom: origin to: corner color: color
    self tclEval: ('img%1 put { %2 } -to %3 %4'
        bindWith: self connected
        with: color
        with: (origin x printString, ' ', origin y printString)
        with: (corner x printString, ' ', corner y printString))
!

fillFrom: origin extent: extent color: color
    self fillFrom: origin to: origin + extent color: color
!

image: aFileStream
    self tclEval: 'img', self connected, ' read ', aFileStream name asTkString
!

imageHeight
    self tclEval: 'return [image height img', self connected, ']'.
    ^self tclResult asInteger
!

imageWidth
    self tclEval: 'return [image width img', self connected, ']'.
    ^self tclResult asInteger
!

lineFrom: origin extent: extent color: color
    self lineFrom: origin to: origin + extent color: color
!

lineFrom: origin to: corner color: color
    self notYetImplemented
!

lineFrom: origin toX: endX color: color
    self tclEval: ('img%1 put { %2 } -to %3 %4'
        bindWith: self connected
        with: color
        with: (origin x printString, ' ', origin y printString)
        with: (endX printString, ' ', origin y printString))
!

lineFrom: origin toY: endY color: color
    self tclEval: ('img%1 put { %2 } -to %3 %4'
        bindWith: self connected
        with: color
        with: (origin x printString, ' ', origin y printString)
        with: (origin x printString, ' ', endY printString))
!

lineInside: rectangle color: color
    self lineFrom: rectangle origin to: rectangle corner color: color
! !

!BImage methodsFor: 'widget protocol'!

destroyed
    primitive isNil ifFalse: [
        self tclEval: 'image delete img', self connected.
    ].
    super destroyed
! !

!BImage methodsFor: 'private'!

create
    self tclEval: 'image create photo img', self connected.
    self create: ' -anchor nw -image img', self connected
!

widgetType
    ^'label '
! !

"------------------------------ List box -----------------------------------"

BViewport subclass: #BList
       instanceVariableNames: 'labels items callback'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BList comment: '
I represent a list box from which you can choose one or more
elements.'!

BList
    defineColorProtocolWithActivePrefix: 'highlight';

    "Tk calls it 'highlightcolor', not 'highlightforeground', so we redefine it"
    defineAccessor: 'highlightForeground' option: 'highlightcolor' fromStringCode: '';
    defineMutator: 'highlightForeground' option: 'highlightcolor' toStringCode: '';

    "#single, #browse, #extended, #multiple"
    defineAccessor: 'mode' option: 'selectmode' fromStringCode: 'asSymbol';
    defineMutator: 'mode' option: 'selectmode' toStringCode: '';

    defineAccessor: 'index'
        command: 'return [%1 index active] '
        key: #index
        fromStringCode: 'asNumber';

    defineMutator: 'setIndex'
        command: '%1 activate %3 '
        key: #index
        toStringCode: 'printString'!


!BList methodsFor: 'callbacks'!

itemSelected: receiver at: index 
    stdout nextPutAll: 'List item '; print: index;
	nextPutAll: ' selected!'; nl.
    stdout nextPutAll: 'Contents: '; nextPutAll: (items at: index + 1); nl.
! !


!BList methodsFor: 'accessing'!

at: anIndex
    ^items isNil
	ifTrue: [ labels at: anIndex + 1 ]
	ifFalse: [ items at: anIndex + 1 ]
!

add: string afterIndex: index
    labels isNil
        ifTrue: [
            index > 0 ifTrue: [ ^self error: 'index out of bounds' ].
            labels := OrderedCollection with: string.
            items := nil
        ]
        ifFalse: [
            labels add: string afterIndex: index.
            items notNil ifTrue: [ items add: nil afterIndex: index ].
        ].
    self tclEval:
        self connected, ' insert ', index printString, ' ', string asTkString.
    ^string
!

add: string element: element afterIndex: index
    labels isNil
        ifTrue: [
            index > 0 ifTrue: [ ^self error: 'index out of bounds' ].
            labels := OrderedCollection with: string.
            element isNil ifFalse: [
                items := OrderedCollection with: element.
            ].
        ]
        ifFalse: [
            labels add: string afterIndex: index.
            element notNil | items notNil ifTrue: [
                items add: element afterIndex: index
            ].
        ].
    self tclEval:
        self connected, ' insert ', index printString, ' ', string asTkString.
    ^element isNil ifTrue: [ element ] ifFalse: [ string ]
!

addLast: string
    labels isNil
        ifTrue: [
            labels := OrderedCollection with: string.
            items := nil
        ]
        ifFalse: [
            labels addLast: string.
            items notNil ifTrue: [ items addLast: nil ].
        ].
    self tclEval: self connected, ' insert end ', string asTkString.
    ^string
!

addLast: string element: element
    labels isNil
        ifTrue: [
            labels := OrderedCollection with: string.
            element isNil ifFalse: [
                items := OrderedCollection with: element
            ].
        ]
        ifFalse: [
            labels addLast: string.
            element notNil | items notNil ifTrue: [
                items addLast: element
            ].
        ].
    self tclEval: self connected, ' insert end ', string asTkString.
    ^element isNil ifTrue: [ element ] ifFalse: [ string ]
!

contents: stringCollection
    ^self contents: stringCollection elements: nil
!

contents: stringCollection elements: elementList
    | stream | 
    (elementList notNil and: [ elementList size ~= stringCollection size ])
	ifTrue: [ ^self error: 'label collection must have the same size as element collection' ].

    labels := stringCollection asOrderedCollection.
    items := elementList.
    items isNil ifFalse: [ items := items asOrderedCollection ].

    self tclEval: self connected, ' delete 0 end'.

    stream := WriteStream on: (String new: 1000).
    stream nextPutAll: self connected; nextPutAll: ' insert 0'.

    stringCollection do: [ :each |
        stream space.
        stream nextPutAll: each asTkString
    ].

    self tclEval: stream contents
!

isSelected: index
    self tclEval: self connected, ' selection includes ', index printString.
    ^self tclResult = '1'
!

labelAt: anIndex
    ^labels at: anIndex + 1
!

numberOfStrings
    ^labels size
!

removeAtIndex: index
    | result |
    result := labels removeAtIndex: index.
    items isNil ifFalse: [
        result := items removeAtIndex: index.
    ].
    self tclEval: self connected, 'delete ', index printString.
    ^result
!

size
    ^labels size
! !

!BList methodsFor: 'disabled operations'!

rowSpacing: spacing
!

columnSpacing: spacing
!

numberOfColumns: numCols
!

usingVerticalLayout
    ^true
!

useVerticalLayout: verticalBool
! !

!BList methodsFor: 'widget protocol'!

highlight: index
    self mode = #multiple ifFalse: [ self unhighlight ].

    self
        setIndex: index;
        show: index
!

isSelected: index
    self tclEval: 'return [', self connected,
        ' selection includes ', index printString, ']'.

    ^self tclResult = '1'
!

select: index
    | indexString |
    indexString := (index - 1) printString.
    self tclEval: self connected, ' selection clear 0 end'.
    self tclEval: self connected, ' selection set ', indexString.
    self tclEval: self connected, ' see ', indexString.
    self tclEval: self connected, ' selection set ', indexString
!

show: index
    self tclEval: self connected, ' see ', (index - 1) printString
!

unselect: index
    self tclEval: self connected, ' selection clear ', (index - 1) printString
!

unhighlight
    self tclEval: self connected, ' selection clear 0 end'
!

callback: aReceiver message: aString
    callback := DirectedMessage
        selector: aString asSymbol
        arguments: (Array with: self with: nil)
        receiver: aReceiver
! !


!BList methodsFor: 'private'!

callback
    ^callback
!

create
    self
        create: ' -takefocus 1 -bg white -exportselection no'
        horizScrollbar: false
        vertical: true.

    "Tcl hack to get the callback upon activate. See analogous
     trick for text boxes in BText>>#init:."

    self tclEval: ('
      rename %1 .%1
      proc %1 args {
        if [regexp {^a.*} [lindex $args 0]] {
          callback %2 invokeCallback: [%1 index [lindex $args 1]]
        }
        uplevel .%1 $args
      }' bindWith: self connected with: self asOop printString).
!

init: parentWidget
    super init: parentWidget.
    self cache at: #index put: nil.
    labels := OrderedCollection new.
!

invokeCallback: indexString
    | index |
    index := indexString asNumber.

    "Tk allows a single/browse-selection list box to have no element
     selected by clicking on the selected item. Thwart its intentions by 
     reselecting that item. This behavior (which is e.g. the one adopted
     by Windows) seemed more reasonable to me when using the listbox in a
     browser"

    (self isSelected: index) ifFalse: [
        (self mode = #single) | (self mode = #browse) ifTrue: [
            (self cache at: #index) isNil ifFalse: [
                self tclEval: self connected, ' selection set ',
                    (self cache at: #index) printString.
                ^self
            ]
        ]
    ].
    self cache at: #index put: index.
    self callback notNil ifTrue: [
        self callback arguments at: 2 put: index.
        self callback send
    ]
!

widgetType
    ^'listbox '
! !


"------------------------------ Window ------------------------------"

BForm subclass: #BWindow		"Top level container"
      instanceVariableNames: 'x y width height callback'
      classVariableNames: 'TopLevel'
      poolDictionaries: ''
      category: 'Graphics-Windows'
! 

BWindow comment: '
I am the boss. Nothing else could be viewed or interacted with if
it wasn''t for me... )):->'!

BWindow
    defineMutator: 'menu' option: 'menu' toStringCode: 'container';

    defineAccessor: 'label' command: 'wm title %1'
        key: #label fromStringCode: '';

    defineMutator: 'label' command: 'wm title %1 %3'
        key: #label toStringCode: '';

    defineAccessor: 'resizable' command: 'wm resizable %1'
        key: #resizable fromStringCode: ' = ''{1 1}'' ';

    defineMutator: 'resizable' command: 'wm resizable %1 %3 %3'
        key: #resizable toStringCode: ' asCBooleanValue printString'!

!BWindow class methodsFor: 'initialization'!

initializeOnStartup
    self tclEval: 'wm withdraw .'.
    TopLevel := OrderedCollection new
!

!BWindow class methodsFor: 'instance creation'!

new
    ^TopLevel add: (super new: nil)
!

new: label
    ^self new
        label: label
! !


!BWindow methodsFor: 'accessing'!

callback: aReceiver message: aString
    callback := DirectedMessage
        selector: aString asSymbol
        arguments: #()
        receiver: aReceiver
! !

!BWindow methodsFor: 'widget protocol'!

centerIn: view
    self
        x: view x + (view width  // 2) - (self parent width  // 2)
        y: view x + (view height // 2) - (self parent height // 2)
!

center
    | scrWidth scrHeight |
    self tclEval: 'return [winfo screenwidth .]'.
    scrWidth := self tclResult asNumber.

    self tclEval: 'return [winfo screenheight .]'.
    scrHeight := self tclResult asNumber.

    self
        x: (scrWidth  // 2) - (self width  // 2)
        y: (scrHeight // 2) - (self height // 2)
!

createToplevelWindow: dummy
    "for backward compatibility"
!

height
    ^height
!

height: anInteger
    height := anInteger.
    self resetGeometry
!

isWindow
    ^true
!

map
    self tclEval: 'wm deiconify ', self connected.
    Blox idle.
    self bringToTop.
!

modalMap
    | previousGrab |
    previousGrab := Grab.
    Grab := self connected.
    self map.
    Blox idle.
    self
        tclEval: 'grab set ', Grab;
        bringToTop;
        tclEval: 'tkwait window ', self connected.

    previousGrab isNil
        ifTrue: [
            self tclEval: 'grab release ', Grab.
            Grab := previousGrab
        ]
        ifFalse: [
            Grab := previousGrab.
            self tclEval: 'grab set ', Grab
        ]
!

state
    ^#normal
!

state: aSymbol
    "Not used for BWindows."
!

unmap
    self tclEval: 'wm withdraw ', self connected
!

width
    ^width
!

width: anInteger
    width := anInteger.
    self resetGeometry
!

width: xSize height: ySize
    width := xSize.
    height := ySize.
    self resetGeometry
!

window
    ^self
!

x
    ^x
!

x: anInteger
    x := anInteger.
    self resetGeometry
!

x: xPos y: yPos
    x := xPos.
    y := yPos.
    self resetGeometry
!

x: xPos y: yPos width: xSize height: ySize
    x := xPos.
    y := yPos.
    width := xSize.
    height := ySize.
    self resetGeometry
!

y
    ^y
!

y: anInteger
    y := anInteger.
    self resetGeometry
! !

!BWindow methodsFor: 'disabled operations'!

heightOffset: value
    self shouldNotImplement
!

widthOffset: value
    self shouldNotImplement
!

xOffset: value
    self shouldNotImplement
!

yOffset: value
    self shouldNotImplement
! !

!BWindow methodsFor: 'private'!

callback
    ^callback
!

destroyed
    super destroyed.
    TopLevel remove: self ifAbsent: [ ]
!

invokeCallback
    ^self callback isNil
        ifTrue: [ true ]
        ifFalse: [ self callback send ]
!

init: parentWidget
    super init: parentWidget.

    self tclEval: ('
        wm protocol %1 WM_DELETE_WINDOW {
          if [callback %2 invokeCallback] {destroy %1}
        }'

        bindWith: self connected
        with: self asOop printString)
!

resetGeometry
    | s |
    s := WriteStream on: (String new: 50).
    s
        nextPutAll: 'wm withdraw ';
        nextPutAll: self connected;
        nl;
        nextPutAll: 'wm geometry ';
        nextPutAll: self connected;
        nextPutAll: ' =';
        print:      width;
        nextPut:    $x;
        print:      height;
        nextPut:    $+;
        print:      x;
        nextPut:    $+;
        print:      y;
        nl;
        nextPutAll: 'wm deiconify ';
        nextPutAll: self connected.

    self tclEval: s contents
!

setInitialSize
    self tclEval: 'winfo x ', self connected. x := self tclResult asNumber.
    self tclEval: 'winfo y ', self connected. y := self tclResult asNumber.
    self width: 100 height: 100
!

setWindowAttributes
    "Do nothing"
!

widgetType
    ^'toplevel '
! !

"Blox extensions "
"Written by Brad Diller June 3, 1995"

"------------------------------ Transient windows ----------------------"

BWindow subclass: #BTransientWindow	
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BTransientWindow comment: '
I am almost a boss. I represent a window which is logically linked
to another which sits higher in the widget hierarchy, e.g. a dialog
box'!

!BTransientWindow class methodsFor: 'instance creation'!

new: label in: parentWindow
    ^self basicNew
        init: parentWindow;
        label: label;
        yourself
! !

!BTransientWindow methodsFor: 'widget protocol'!

map
    super map.
    self tclEval: 'wm transient ', self connected, ' ', self parent connected
! !

!BTransientWindow methodsFor: 'private'!

setWidgetName: parentWidget

    | unique |
    unique := self class uniqueName.
    parentWidget isNil 
        ifTrue: [ ^unique ].

    ^parentWidget parent isNil
        ifTrue: [ unique ]
        ifFalse: [ parentWidget parent container copy, unique ].
! !


"------------------------------ Auto dialog widgets --------------------"

BForm subclass: #BDialog
       instanceVariableNames: 'callbacks initInfo'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BDialog comment: '
I am a facility for implementing dialogs with many possible choices
and requests. In addition I provide support for a few platform native
common dialog boxes, such as choose-a-file and choose-a-color.'!


!BDialog class methodsFor: 'prompters'!

chooseColor: parent label: aLabel default: color
    parent map.
    self tclEval: ('tk_chooseColor -parent %1 -title %2 -initialcolor %3'
        bindWith: parent container
        with: aLabel asTkString
        with: color asTkString).
    ^self tclResult
!

chooseFileToOpen: parent label: aLabel default: name defaultExtension: ext types: typeList
    "e.g.
        fileName := BDialog
            chooseFileToOpen: aWindow
            label: 'Open file'
            default: nil
            defaultExtension: 'gif'
            types: #(
               ('Text Files'       '.txt' '.diz')
               ('Smalltalk files'  '.st')
               ('C source files'   '.c')
               ('GIF files'        '.gif'))           "

    ^self
        chooseFile: 'Open'
        parent: parent
        label: aLabel
        default: name
        defaultExtension: ext
        types: typeList
!

chooseFileToSave: parent label: aLabel default: name defaultExtension: ext types: typeList
    "example: see chooseFileToOpen..."

    ^self
        chooseFile: 'Save'
        parent: parent
        label: aLabel
        default: name
        defaultExtension: ext
        types: typeList
! !

!BDialog class methodsFor: 'private'!

chooseFile: operation parent: parent label: aLabel default: name
    defaultExtension: ext types: typeList

    | stream strictMotif file |
    stream := WriteStream on: String new.

    stream
       nextPutAll: 'return [tk_get';
       nextPutAll: operation;
       nextPutAll: 'File -parent ';
       nextPutAll: parent container;
       nextPutAll: ' -title ';
       nextPutAll: aLabel asTkString;
       nextPutAll: ' -defaultextension ';
       nextPutAll: ext asTkString;
       nextPutAll: ' -filetypes {'.

    typeList do: [ :each |
        stream
            nextPut: ${;
            nextPutAll: (each at: 1) asTkString;
            nextPutAll: ' {'.

        each size > 1
            ifTrue: [
                each from: 2 to: each size do: [ :type |
                    stream nextPutAll: type; space.
                ]
            ].
        stream nextPutAll: '}} '.
    ].
    stream nextPutAll: '{"All files" * }}'.

    (name notNil and: [ name notEmpty ]) ifTrue: [
       stream
           nextPutAll: ' -initialfile ';
           nextPutAll: name asTkString
    ].
    stream nextPutAll: ' ]'.

    strictMotif := BText emacsLike.
    BText emacsLike: (Blox platform ~= 'unix').

    parent map.
    self tclEval: stream contents.
    file := self tclResult.

    BText emacsLike: strictMotif.
    ^file
! !


!BDialog class methodsFor: 'instance creation'!

new: parent
    ^self basicNew initInfo: '' -> nil; init: parent
!

new: parent label: aLabel
    ^self basicNew initInfo: aLabel -> nil; init: parent
!

new: parent label: aLabel prompt: aString
    ^self basicNew initInfo: aLabel -> aString; init: parent
! !

!BDialog methodsFor: 'accessing'!

addButton: aLabel receiver: anObject index: anInt
    callbacks addLast: (DirectedMessage
        selector: #dispatch:
        arguments: (Array with: anInt)
        receiver: anObject).
    self addButton: aLabel.
!

addButton: aLabel receiver: anObject message: aSymbol
    callbacks addLast: (DirectedMessage
        selector: aSymbol
        arguments: #()
        receiver: anObject).
    self addButton: aLabel.
!

contents
    self tclEval: 'return ${var', self connected, '}'.
    ^self tclResult
!

contents: newText
    self tclEval: 'set var', self connected, ' ', newText asTkString
! !


!BDialog methodsFor: 'widget protocol'!

center
    self parent center
!

centerIn: view
    self parent centerIn: view
!

destroyed
    self tclEval: 'unset var', self connected
    super destroyed.
!

loop
    self parent width: (self parent width min: 200).
    self parent modalMap.
! !

!BDialog methodsFor: 'private'!

addButton: aLabel
    self tclEval: ('button %1.buttons.b%2 -text %3 -takefocus 1 -command {
        callback %4 "invokeCallback:" %2
        destroy %1
    }'
        bindWith: self container
        with: callbacks size printString
        with: aLabel asTkString
        with: self asOop printString).

    self tclEval: ('pack %1.buttons.b%2 -side left -expand 1'
        bindWith: self container
        with: callbacks size printString).
!

create
    super create.
    self tclEval: ('
        label %1.msg -padx 5 -pady 5 -anchor nw -text ', initInfo key asTkString, '
        place %1.msg -x 0.0 -y 0.0 -relwidth 1.0
        %1 configure -background [ %1.msg cget -background ]
        frame %1.buttons -takefocus 1
        place %1.buttons -anchor sw -x 0.0 -rely 1.0 -relwidth 1.0 -height 14m
        lower %1.buttons
        lower %1.msg' bindWith: self connected).

    initInfo value isNil ifTrue: [ ^self ].

    self tclEval: ('
        set var%1 %2
        entry %1.text -textvariable var%1 -takefocus 1
        place %1.text -in %1.msg -x 5 -y 5 -width -10 -rely 1.0 -relwidth 1.0
        raise %1.text'
        bindWith: self connected with: initInfo value asTkString).
!

init: parentWidget
    super init: parentWidget.
    callbacks := OrderedCollection new.
!

initInfo: assoc
    initInfo := assoc
!

invokeCallback: index
    (callbacks at: index asNumber) send.
    self parent destroy
! !

"setInitialSize
    self restoreWidgetSize
! !"

"------------------------------ Menu Bar ------------------------------"

BMenuObject subclass: #BMenuBar
     instanceVariableNames: ''
     classVariableNames: ''
     poolDictionaries: ''
     category: 'Graphics-Windows'
! 

BMenuBar comment: '
I am the Menu Bar, the top widget in a full menu structure.'!

!BMenuBar methodsFor: 'accessing'!

add: aMenu
    aMenu create.
    ^self addChild: aMenu
!

remove: aMenu
    self tclEval: 'delete ', aMenu connected
! !

!BMenuBar methodsFor: 'private'!

container
    ^primitive
!

connected
    ^primitive
!

init: parentWidget
    super init: parentWidget.
    primitive := self parent isNil
        ifTrue: [ self class uniqueName ]
        ifFalse: [ self parent container, '.menu' ].

    "BMenuBar is NOT a BElement, so it has to explicitly create itself"
    self tclEval: 'menu ', self connected, ' -tearoff 0'.
    self parent isNil ifFalse: [ self parent menu: self ]
! !


"------------------------------ Menu ------------------------------"

BMenuObject subclass: #BMenu
     instanceVariableNames: 'label exists'
     classVariableNames: ''
     poolDictionaries: ''
     category: 'Graphics-Windows'
! 

BMenu comment: '
I am the Menu, which groups together commands in a menu structure.'!

!BMenu class methodsFor: 'instance creation'!

new: parent label: label
    ^self basicNew init: parent; label: label; yourself
! !

!BMenu methodsFor: 'accessing'!

exists
    ^exists
!

label: value
    label := value.
    exists ifTrue: [
        self tclEval: self connected, ' configure -title ', (value asTkString)
    ].
!

label
    ^label
! !

!BMenu methodsFor: 'callback registration'!

addLine
    ^self addMenuItemFor: #() notifying: self    "self is dummy"
!

addMenuItemFor: anArray notifying: receiver
    "Receiver will be sent the callback messages.  anArray
     is something that responds to at: and size.  Possible types are:
     #()                insert a seperator line
     #(name)            create a menu item with name, but no callback
     #(name symbol)     create a menu item with the given name and
                        no parameter callback.
     #(name symbol arg) create a menu item with the given name and
                        one parameter callback."

    | item |
    item := self newMenuItemFor: anArray notifying: receiver.
!

callback: receiver using: selectorPairs
    "Receiver will be sent the callback messages.  Selector pairs
     is a collection of Arrays (or something that responds to at:
     and size). Possible types are:
     ()             insert a seperator line
     #(name)            create a menu item with name, but no callback
     #(name symbol)     create a menu item with the given name and
                        no parameter callback.
     #(name symbol arg) create a menu item with the given name and
                        one parameter callback."

    | item |
    selectorPairs do: [ :pair |
        item := self newMenuItemFor: pair notifying: receiver.
        exists ifTrue: [ item create ].
    ].
!

destroy
    self parent remove: self.
! !

!BMenu methodsFor: 'private'!

addChild: menuItem
    menuItem menuIndex: self childrenCount.
    ^super addChild: menuItem
!

container
    ^primitive
!

connected
    ^primitive
!

create
    | s |
    s := WriteStream on: (String new: 80).
    s
        nextPutAll: 'menu ';
        nextPutAll: self connected;
        nextPutAll: ' -tearoff 0';
        nl;
        nextPutAll: self parent container;
        nextPutAll: ' add cascade -label ';
        nextPutAll: self label asTkString;
        nextPutAll: ' -underline 0 -menu ';
        nextPutAll: self connected.

    self tclEval: s contents.
    
    "Set the title for torn-off menus"
    self label: self label.
    self childrenDo: [ :each | each create ].
    exists := true.
!

init: parentWidget
    super init: parentWidget.
    label := ''.
    primitive := self parent container copy, self class uniqueName.
    exists := false.
!

newMenuItemFor: pair notifying: receiver
    | item |
    pair size = 0 ifTrue: [ ^BMenuItem new: self ].

    item := BMenuItem new: self label: (pair at: 1).

    pair size = 1
        ifTrue: [ ^item ].

    pair size = 2
	ifTrue: [ ^item callback: receiver message: (pair at: 2) ].
    
    ^item callback: receiver message: (pair at: 2) argument: (pair at: 3)
! !

"------------------------------ Popup Menu ------------------------------"

BMenu subclass: #BPopupMenu
       instanceVariableNames: ''
       classVariableNames: 'PopupMenuBar'
       poolDictionaries: ''
       category: 'Graphics-Windows'
! 

BMenu comment: '
I am a Menu that pops up on another window. A handy shortcut, surely'!

!BPopupMenu class methodsFor: 'accessing'!

initializeOnStartup
    PopupMenuBar := nil
!

popupMenuBar
    PopupMenuBar isNil ifTrue: [ PopupMenuBar := BMenuBar new: nil ].
    ^PopupMenuBar
! !

!BPopupMenu methodsFor: 'widget protocol'!

popup
    self tclEval: ('event generate %1 <Shift-F10>'
        bindWith: self parent connected)
! !

!BPopupMenu methodsFor: 'private'!

init: parentWindow
    super init: self class popupMenuBar.
    self parent add: self.
    parentWindow
        bind: '<Button-3>'
        to: #popup:y:
        of: self
        parameters: '%X %Y'.

    parentWindow
        bind: '<Shift-F10>'
        to: #popup:y:
        of: self
        parameters: '[expr 2+[winfo rootx %W]] [expr 2+[winfo rooty %W]]'.
!

popup: x y: y
    "Note that x and y are strings!"
    self tclEval: 'tk_popup ', self connected, ' ', x, ' ', y
! !


"------------------------------ Menu Item ------------------------------"

BMenuObject subclass: #BMenuItem
       instanceVariableNames: 'index callback createCode'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'
!

BMenuItem comment: '
I am the tiny and humble Menu Item, a single command choice in the
menu structure. But if it wasn''t for me, nothing could be done...
eh eh eh!!'!

!BMenuItem class methodsFor: 'instance creation'!

new: parent label: label
    ^self basicNew init: parent label: label
!

new: parent
    ^self basicNew init: parent
! !


!BMenuItem methodsFor: 'accessing'!

callback: aReceiver message: aSymbol
    callback := DirectedMessage
        selector: aSymbol asSymbol
        arguments: #()
        receiver: aReceiver
!

callback: aReceiver message: aSymbol argument: anObject
    callback := DirectedMessage
        selector: aSymbol asSymbol
        arguments: (Array with: anObject)
        receiver: aReceiver
! !


!BMenuItem methodsFor: 'accessing'!

label: value
    (self cache at: #label) isNil
        ifTrue: [ ^self error: 'no label for separator lines' ].

    self parent exists ifTrue: [
        self tclEval:
            self parent connected, ' entryconfigure ', index printString,
            ' -label ', value asTkString
    ].
    self cache at: #label put: value
!

label
    ^self cache at: #label
! !


!BMenuItem methodsFor: 'private'!

callback
    ^callback
!

create
    | bind |
    self label isNil
        ifFalse: [ createCode := createCode bindWith: self label asTkString ].

    self tclEval: createCode.
    createCode := ''       "free some memory"
!

init: parentWidget label: label
    | s |
    super init: parentWidget.

    s := WriteStream on: (String new: 80).
    s
        nextPutAll: self parent container;
        nextPutAll: ' add command -label %1 -underline 0 -command { callback ';
        print:      self asOop;
        nextPutAll: ' invokeCallback }'.

    createCode := s contents.
    self cache at: #label put: label.
    parent addChild: self.
    parent exists ifTrue: [ self create ].
!

init: parentWidget
    super init: parentWidget.
    createCode := self parent container, ' add separator'.
    self cache at: #label put: nil.
    parent addChild: self.
    parent exists ifTrue: [ self create ].
!

invokeCallback
    self callback isNil ifFalse: [ self callback send ]
!

menuIndex
    ^index
!

menuIndex: anIndex
    index := anIndex
! !
