/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*
 * Name          : sumThreshold.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 *
 *  This file contains two functions:
 *
 *     pipsJobManager(), which is executed on the local host and
 *     pipsProcPar(),    which is executed on all hosts.
 *
 *  Both have their own description appended below.
 */

#include "pips.h"   /* PIPS system header file  */
#include "pvm3.h"   /* PVM system header file   */
#include <math.h>   /* Used: sqrt()             */

/*
 * +++ Set the minimum number of variables and parameters, 
 * the program expects on the command line
 */
int parameterNumber = 2;

/*
 * +++ Description of program for display on stderr by pipsPrintError()
 */
char *programName = "sumThreshold";           /* Used in program description */
char *description[] = {
  "  sumThreshold [-h] ncName variable1 variable2 [winSize [threshold]]",
  " ",
  "Where:",
  "  -h         Prints this help text.",
  "  ncName     Name of netCDF file to process, must be first parameter.",
  "  variable1  Variable to operate on.",
  "  variable2  Variable to store the result; must not be variable 1.",
  "  winSize    The window size to use",
  "  threshold  The threshold to use for assuming pixels similar.",
  " ",
  "Description:",
  "  sumThreshold opens 'ncName', spawns tasks on all processors in the",
  "  virtual parallel machine and ",
  "  Expects byte data as input.",
  "  Counts, how many pixels in a window of size winSize x winSize differ",
  "  less than 'threshold' from the center pixel.",
  ""
}; /* end description */




/******************************************************************************
 *  P A R E N T   P R O C E S S  (on local host)
 ******************************************************************************
 *
 *   void pipsJobManager (ntask, tid, ncId, params)
 *
 *   int ntask        IS   Number of tasks available for parallel
 *                         data processing
 *   int tid []       IS   An array with task Id's of the tasks
 *   int ncId         IS   Id of opened netcdf file, returned by ncopen()
 *   char **params    IS   Tail of argv containing command line options
 *
 * Description   : 
 *
 *   Here, the data SEGMENTATION is done.
 *   Segments the data into pieces that can be processed in parallel,
 *   sends them to the tasks (that run pipsProcPar) and recollects it.
 *   If the number of data fragments (left) is less than the number of
 *   tasks, a competition is started that also guaranties that all lost
 *   data (crashed hosts, etc) is reprocessed.   
 */

#ifdef __STDC__

void pipsJobManager (int ntask, int tid[], int ncId, int parac, char **parav)

#else

pipsJobManager (ntask, tid, ncId, parac, parav)
    int ntask;
    int tid[];
    int ncId;
    int parac;
    char **parav;

#endif
{
/*
 * Variables for the job distribution
 */
  int done = 0;         /* Total number of processed jobs                    */
  int *jobId;           /* Array with currently processed jobs (1 per task)  */
  int *perform;         /* how many jobs each task has allready accomplished */
  int index;            /* Index to tid[], jobId[] and perform[]             */
  long i;               /* Index to tid[], jobId[] and perform[]             */
  int job = 0;          /* Current job Id (0 .. njobs-1)                     */
  int finishedJob;      /* Id of job that has just been finished             */
  int newJobs = 1;      /* Indicates that new jobs are still available       */
  int njobs;            /* Number of data slabs to process parallel          */

/*
 * +++ Netcdf variables and variables for data segmentation
 * start, edge: recordDim, rowDim, columnDim
 */
  long start[3];        /* Start of each datapacket                          */
  long edge [3];        /* Shape (size) of each datapacket                   */
  int fileType;         /* Value to use for file type attribute              */
  nc_type dataType;     /* Data type used in the result variable             */
  int var0Id;           /* Id of the first variable                          */
  int var1Id;           /* Id of the second variable                         */
  unsigned char *data;  /* holds data that has to be processed               */
  unsigned long *result;/* holds data that has been processed                */
  long rows;            /* Number of rows of data in one data fragment       */
  long images;          /* Number of images in current variable              */
  long height;          /* Height of image in variable 'varName' (dataId)    */
  long width;           /* Width of image in variable 'varName' (dataId)     */
  nc_type type;         /* Data type used in 'dataId'                        */
  nc_type resType;      /* Type of an existing variable for the results      */
  long winSize;         /* Size of window to examine arround center pixel    */
  float threshold;      /* Threshold to determine similarity                 */
  int memSize;          /* Size of data buffer in byte                       */

/*
 * +++ Set the granularity to adjust the programs performance (see document.)
 */
  int granularity = 3;  /* The higher it is, the more jobs are generated     */


/******************************************************************************
 * +++  P R O C E S S   P A R A M E T E R S   A N D   P R E P A R A T I O N S 
 ******************************************************************************
 */

/*
 * +++ Check the input variables (existence and properties)
 * Get their dimensions and calculate minimum of each dimension
 */
  var0Id = ncvarid (ncId, parav[0]);
  if (! pipsVarInq (ncId, &var0Id, parav[0], &type,&images,&height,&width)) {
    pipsPrintError (ncId, PipsInvalidVar, parav[0], PipsDontExit);
    return;
  }

/*
 * +++ Check type of incoming data
 * Complain, if the variable doesn't have the expected data type
 */
  if (nctypelen (type) != 1) {
    pipsPrintError (ncId, PipsDataIncompat, parav[0], PipsDontExit);
    return;
  } /* end if */

/*
 * +++ Select type of outgoing data and fileType to create from it (later on)
 */
  dataType = NC_LONG;
  fileType = PipsPpmRaw;

/*
 * +++ Create the output variable (exit on failure)
 */
  resType = dataType;
  if ((var1Id = pipsVarDef (ncId, parav[1], &resType, &fileType,
                             &i, &height, &width)) < 0) {
    pipsPrintError (ncId, PipsInvalidVar, parav[1], PipsDontExit);
    return;
  } /* end if */

/*
 * Complain and exit, if the result variable exists and is unsuited for result
 */
  if (dataType != resType) {
    pipsPrintError (ncId, PipsVariableType, parav[1], PipsDontExit);
    return;
  } /* end if */

/*
 * +++ Define start and shape of data fragments (constant parts)
 */
  start [0] = 0;    edge  [0] = 1;
  start [2] = 0;    edge  [2] = width;

/*
 * +++ Evaluate the rest of command line parameters
 */
  index = 1;

  index++;
  winSize = 3;                    /* Default value */
  if (parac > index)
    winSize = atol (parav[index]);

  index++;
  threshold = 1;                  /* Default value */
  if (parac > index)
    threshold = atol (parav[index]);

/*
 * +++ Make window dimension odd to allow for a central element.  
 */
  if (winSize % 2 == 0)
    winSize++;

  fprintf (stderr, "Using window %d, threshold %f\n", winSize, threshold);
/*
 * +++ Determine the total size of data per variable
 */
  rows = (height - winSize +1) * images;        /* number of rows to process */

/*
 * Calculate the number of jobs (data fragments) and rows per job
 */
  rows = pipsDataPerJob (ntask,granularity,images,sqrt(rows),rows,&njobs);

/*
 * Reset the processed job array - currently no task is processing a job
 */
  jobId = pipsNewFlagArray (ntask, PipsNoJob);

/*
 * Reset the number of jobs accomplished by each task
 */
  perform = pipsNewFlagArray (ntask, 0);

/*
 * +++ Allocate memory, one buffer for both, loading and saving of data
 * Make sure the buffer size is a multiple of the larger data type
 */
  memSize = PipsMax ((rows + winSize) * sizeof (char), rows * sizeof (long));
  memSize -= (memSize % sizeof (long));
  data   = (unsigned char *) calloc (width, memSize);
  result = (unsigned long *) data;                /* uses same memory */

/******************************************************************************
 *  D A T A   S E G M E N T A T I O N
 ******************************************************************************
 * Distribute jobs until all have been processed
 */
  while (done < njobs) {
    pipsAwaitChildData (tid,jobId,perform,&done, ntask, &finishedJob, &index);
/******************************************************************************
 * +++ R E C E I V E   A N D   S T O R E   P R O C E S S E D   D A T A 
 ******************************************************************************
 */
/*
 * Unpack and save data, if task was busy with a job
 */
    if (finishedJob != PipsNoJob) {
/*
 * +++ Set the image number, the data fragment belongs to
 */
      start [0] = (finishedJob * rows) / height;
/*
 * +++ Set the row number, the data fragment starts at
 */
      start [1] = (finishedJob * rows) % height + winSize / 2;
      edge  [1] = rows;
      pvm_upkulong (result, edge[0]*edge[1]*edge[2], 1);
      ncvarput (ncId, var1Id, start, edge, (void *) result);
    } /* end if finishedJob */

/*
 * Keep sending, while there is still an unfinished job left
 */
    if (pipsJobAvailable (jobId, &job, ntask, index, newJobs)) {
/******************************************************************************
 * +++  L O A D   A N D   S E N D   P R O C E S S   D A T A 
 ******************************************************************************
 */
/*
 * +++ Set the image number, the data fragment belongs to
 */
      start [0] = (job * rows) / height;
/*
 * +++ Set the row number, the data fragment starts at
 */
      start [1] = (job * rows) % height;
      edge  [1] = rows + winSize - 1;
      ncvarget (ncId, var0Id, start, edge, (void *) data);
/*
 * +++ Pack in the processing parameters
 */
      pvm_pklong  (&width, 1, 1);
      pvm_pklong  (&rows, 1, 1);
      pvm_pklong  (&winSize,   1, 1);
      pvm_pkfloat (&threshold, 1, 1);
      pvm_pkubyte (data, edge[0]*edge[1]*edge[2], 1);

/*
 * Send data packet to the idle child and tell it to commence work on it
 */
      pvm_send (tid[index], PipsCommence);
    } /* end if pipsJobAvailable */

/*
 * Select a job that hasn't been processed yet, if available
 * (for the next idle task)
 */
    pipsSelectNewJob (&newJobs, njobs, &job);
  } /* end while done < njobs */


/******************************************************************************
 *  E V A L U A T I O N   A N D   C L E A N U P
 ******************************************************************************
 */
/*
 * Report performance evaluation
 */
  pipsPrintPerformance (tid, perform, ntask, njobs);

/*
 * Free allocated memory
 */
  free (data);
  free (jobId);
  free (perform);
} /* pipsJobManager */







/******************************************************************************
 *  C H I L D   P R O C E S S  (on all hosts)
 ******************************************************************************
 *
 *   void pipsProcPar (void)
 *
 * Description   : 
 *
 *   Here the data PROCESSING is performed.
 *   Unpacks data received from the pipsJobManager (), processes it
 *   and packs the result data in again to be send back to the
 *   pipsJobManager. 
 */

#ifdef __STDC__

void pipsProcPar (void)

#else

pipsProcPar ()

#endif
{
/*
 * +++ Variables sent by the jobManager
 * NOTE: Declaring all variables 'static' increases speed dramatically
 */
  static long width;
  static long rows;
  static long  winSize;
  static float threshold;

/*
 * +++ 'Backup' of above variables to keep track of changes in variables
 */
  static long currWidth = 0;
  static long currRows  = 0;
  static long  currWinSize = 0;
  static float currThreshold = 0;

/*
 * +++ Variables for processing the data
 */
  static long x, y, wx, wy;
  static long middle;
  static long elements;
  static long sum;
  static unsigned char *data = NULL;
  static unsigned long *result = NULL;

/*
 * +++ Define macros for better readability
 */
#define DATA(X,Y)   data   [(Y)*width + (X)]
#define RESULT(X,Y) result [(Y)*width + (X)]

/******************************************************************************
 * R E C E I V E   P R O C E S S   D A T A 
 ******************************************************************************
 * +++ Unpack data that has arrived from the jobmanager
 */
  pvm_upklong (&width, 1, 1);
  pvm_upklong (&rows, 1, 1);
  pvm_upklong (&winSize, 1, 1);
  pvm_upkfloat (&threshold, 1, 1);

/*
 * +++ Preparations after change of parameters (data from different image)
 * Memory has to be reallocated and variables re-initialized
 */
  if ((width     != currWidth)    ||
      (rows      != currRows)     ||
      (winSize   != currWinSize)  ||
      (threshold != currThreshold)) {
    currWidth = width;
    currRows = rows;
    currWinSize = winSize;
    currThreshold = threshold;

/*
 * +++ Allocate memory for variables
 */
    if (data)               /* Free allocated memory first */
      free (data);
    data = (unsigned char *) malloc (width * (rows+winSize-1) *sizeof(char));

    if (result)             /* Free allocated memory first */
      free (result);
    result = (unsigned long *) calloc (width *  rows, sizeof (long));

/*
 *  Code simplifying varibles
 */
    middle = winSize / 2;
    elements = winSize * winSize;
  } /* end if */

/*
 * +++ Receive the image data
 */
  pvm_upkubyte (data, width * (rows + winSize - 1), 1);
      
/******************************************************************************
 * +++ C A L C U L A T I O N S   O N   D A T A 
 ******************************************************************************
 */
/*   
 *  Loop over image   
 */
  for ( y = middle; y < rows + middle; y++ )
    for ( x = middle; x < width - middle; x++ ) {
/* 
 *  Loop over window; count pixel with difference < threshold
 */
      sum = 0;
      for ( wy = y-middle; wy < ( y + middle + 1); wy++ )
        for ( wx = x-middle; wx < ( x + middle + 1); wx++ )
          sum += (abs (DATA (wx, wy) - DATA (x, y)) <= threshold);
      RESULT (x, y-middle) = sum;
    } /* end for x */

  pvm_initsend (PvmDataDefault);
/******************************************************************************
 * S E N D   R E S U L T   D A T A 
 ******************************************************************************
 * +++ Pack result data and send to jobmanager (sending done by main() )
 */
  pvm_pkulong (result, width * rows, 1);

} /* end pipsProcPar */


/* end sumThreshold.c */
