/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*
 * Name          : histogram.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : Thu Oct 13 21:36:52 WET DST 1994
 * Modified on   : 
 *
 *  This file contains two functions:
 *
 *     pipsJobManager(), which is executed on the local host and
 *     pipsProcPar(),    which is executed on all hosts.
 *
 *  Both have their own description appended below.
 */

#include "pips.h"   /* PIPS system header file  */
#include "pvm3.h"   /* PVM system header file   */
#include <math.h>   /* Used: sqrt()             */

/*
 * +++ Set the minimum number of variables and parameters, 
 * the program expects on the command line
 */
int parameterNumber = 2;

/*
 * +++ Description of program for display on stderr by pipsPrintError()
 */
char *programName = "histogram";           /* Used in program description */
char *description[] = {
  "  histogram [-h] ncName variable1 variable2 [histMin [histMax]]",
  " ",
  "Where:",
  "  -h         Prints this help text.",
  "  ncName     Name of netCDF file to process, must be first parameter.",
  "  variable1  Variable to operate on; must contain long values.",
  "  variable2  Variable to store the result; must not be variable 1.",
  "  histMin    Values from 0 to histMin are set to 0 (ignored), default is -1",
  "  histMax    Values from histMax to 255 are set to 0, default is 256",
  " ",
  "Description:",
  "  histogram opens 'ncName', spawns tasks on all processors in the",
  "  virtual parallel machine and ",
  "  Creates a 256x256 image containing the histogram of variable1",
  "  that is stored in variable2",
  ""
}; /* end description */




/******************************************************************************
 *  P A R E N T   P R O C E S S  (on local host)
 ******************************************************************************
 *
 *   void pipsJobManager (ntask, tid, ncId, parac, parav)
 *
 *   int ntask        IS   Number of tasks available for parallel
 *                         data processing
 *   int tid []       IS   An array with task Id's of the tasks
 *   int ncId         IS   Id of opened netcdf file, returned by ncopen()
 *   int parac        IS   Number of valid arguments in parav[]
 *   char **parav     IS   Tail of argv containing command line options
 *
 * Description   : 
 *
 *   Here, the data SEGMENTATION is done.
 *   Segments the data into pieces that can be processed in parallel,
 *   sends them to the tasks (that run pipsProcPar) and recollects it.
 *   If the number of data fragments (left) is less than the number of
 *   tasks, a competition is started that also guaranties that all lost
 *   data (crashed hosts, etc) is reprocessed.   
 */

#ifdef __STDC__

void pipsJobManager (int ntask, int tid[], int ncId, int parac, char **parav)

#else

pipsJobManager (ntask, tid, ncId, parac, parav)
    int ntask;
    int tid[];
    int ncId;
    int parac;
    char **parav;

#endif
{
/*
 * Variables for the job distribution
 */
  int done = 0;         /* Total number of processed jobs                    */
  int *jobId;           /* Array with currently processed jobs (1 per task)  */
  int *perform;         /* how many jobs each task has allready accomplished */
  int index;            /* Index to tid[], jobId[] and perform[]             */
  long i;               /* Index to tid[], jobId[] and perform[]             */
  int job = 0;          /* Current job Id (0 .. njobs-1)                     */
  int finishedJob;      /* Id of job that has just been finished             */
  int newJobs = 1;      /* Indicates that new jobs are still available       */
  int njobs;            /* Number of data slabs to process parallel          */

/*
 * +++ Netcdf variables and variables for data segmentation
 * start, edge: recordDim, rowDim, columnDim
 */
  long start[3];           /* Start of each datapacket                       */
  long edge [3];           /* Shape (size) of each datapacket                */
  int fileType;            /* Value to use for file type attribute           */
  nc_type dataType;        /* Data type used in the result variable          */
  int var0Id;              /* Id of the first variable                       */
  int var1Id;              /* Id of the second variable                      */
  unsigned char *data;     /* holds data that has to be processed            */
  long rows;               /* Number of rows of data in one data fragment    */
  long images;             /* Number of images in current variable           */
  long height;             /* Height of image in variable 'varName' (var0Id) */
  long width;              /* Width of image in variable 'varName' (var0Id)  */
  nc_type type;            /* Data type used in 'dataId'                     */
  nc_type resType;         /* Type of an existing variable for the results   */
  long histTotal [256];    /* Total number of pixels                         */
  long hist [256];         /* Number of pixels in one datafragment           */
  int point;               /* Point in histogram image, where a bar starts   */
  int x, y;                /* Indices for creating the histogram image       */
  long max;                /* Maximum number of pixels in the histogram      */
  char img[256][256];      /* Bitplane, where the histogram is drawn first   */
  long histHeight;         /* Height of histogram in pixels                  */
  long histWidth;          /* Width of histogram in pixels                   */
  long histMin;            /* Lower bound of the histogram                   */
  long histMax;            /* Higher bound of the histogram                  */
  long elements;           /* Number of histogram values                     */
  float sumX,sumX2,sigma;  /* Var's to calculate standard deviation          */

/*
 * +++ Set the granularity to adjust the programs performance (see document.)
 */
  int granularity = 3;  /* The higher it is, the more jobs are generated     */


/******************************************************************************
 * +++  P R O C E S S   P A R A M E T E R S   A N D   P R E P A R A T I O N S 
 ******************************************************************************
 */

/*
 * +++ Check the input variables (existence and properties)
 * Get their dimensions and calculate minimum of each dimension
 */
  var0Id = ncvarid (ncId, parav[0]);
  if (! pipsVarInq (ncId, &var0Id, parav[0], &type,&images,&height,&width)) {
    pipsPrintError (ncId, PipsInvalidVar, parav[0], PipsDontExit);
    return;
  }

/*
 * +++ Check type of incoming data
 * Complain, if the variable doesn't have the expected data type
 */
  if (nctypelen (type) != 1) {
    pipsPrintError (ncId, PipsDataIncompat, parav[0], PipsDontExit);
    return;
  } /* end if */

/*
 * +++ Select type of outgoing data and fileType to create from it (later on)
 */
  dataType = NC_BYTE;
  fileType = PipsPgmRaw;
  histHeight = 256;
  histWidth = 256;

/*
 * +++ Create the output variable (exit on failure)
 */
  resType = dataType;
  if ((var1Id = pipsVarDef (ncId, parav[1], &resType, &fileType,
                             &i, &histHeight, &histWidth)) < 0) {
    pipsPrintError (ncId, PipsInvalidVar, parav[1], PipsDontExit);
    return;
  } /* end if */

/*
 * Complain and exit, if the result variable exists and is unsuited for result
 */
  if (dataType != resType) {
    pipsPrintError (ncId, PipsVariableType, parav[1], PipsDontExit);
    return;
  } /* end if */

/*
 * +++ Evaluate the rest of command line parameters
 */
  index = 1;

  index++;
  histMin = -1;                    /* Default value */
  if (parac > index)
    histMin = atol (parav[index]);

  index++;
  histMax = 256;                  /* Default value */
  if (parac > index)
    histMax = atol (parav[index]);

/*
 * +++ Determine the total size of data per variable
 */
  rows = height * images;                       /* number of rows to process */

/*
 * Calculate the number of jobs (data fragments) and rows per job
 */
  rows = pipsDataPerJob (ntask,granularity,images,sqrt(rows),rows,&njobs);

/*
 * Reset the processed job array - currently no task is processing a job
 */
  jobId = pipsNewFlagArray (ntask, PipsNoJob);

/*
 * Reset the number of jobs accomplished by each task
 */
  perform = pipsNewFlagArray (ntask, 0);

/*
 * +++ Allocate memory, one buffer for both, loading and saving of data
 */
  data = (unsigned char *) malloc (width * rows * sizeof (char));

/*
 * +++ Define start and shape of data fragments (constant parts)
 */
  start [0] = 0;    edge  [0] = 1;
                    edge  [1] = rows;
  start [2] = 0;    edge  [2] = width;

/*
 * Initialize the total histogram
 */
  for (i=0; i<256; i++)
    histTotal [i] = 0;

/******************************************************************************
 *  D A T A   S E G M E N T A T I O N
 ******************************************************************************
 * Distribute jobs until all have been processed
 */
  while (done < njobs) {
    pipsAwaitChildData (tid,jobId,perform,&done, ntask, &finishedJob, &index);
/******************************************************************************
 * +++ R E C E I V E   A N D   S T O R E   P R O C E S S E D   D A T A 
 ******************************************************************************
 */
/*
 * Unpack and save data, if task was busy with a job
 */
    if (finishedJob != PipsNoJob) {
      pvm_upklong (hist, 256, 1);
      for (i=0; i<256; i++)
        histTotal [i] += hist [i];
    } /* end if finishedJob */

/*
 * Keep sending, while there is still an unfinished job left
 */
    if (pipsJobAvailable (jobId, &job, ntask, index, newJobs)) {
/******************************************************************************
 * +++  L O A D   A N D   S E N D   P R O C E S S   D A T A 
 ******************************************************************************
 */
/*
 * +++ Set the image number, the data fragment belongs to
 */
      start [0] = (job * rows) / height;
/*
 * +++ Set the row number, the data fragment starts at
 */
      start [1] = (job * rows) % height;
      ncvarget (ncId, var0Id, start, edge, (void *) data);
/*
 * +++ Pack in the processing parameters
 */
      pvm_pklong  (&width, 1, 1);
      pvm_pklong  (&rows, 1, 1);
      pvm_pkubyte (data, edge[0]*edge[1]*edge[2], 1);

/*
 * Send data packet to the idle child and tell it to commence work on it
 */
      pvm_send (tid[index], PipsCommence);
    } /* end if pipsJobAvailable */

/*
 * Select a job that hasn't been processed yet, if available
 * (for the next idle task)
 */
    pipsSelectNewJob (&newJobs, njobs, &job);
  } /* end while done < njobs */

/*
 * Get the maximum number in histTotal[] and set all values that are to be
 * ignored, to 0. Calculate the standard deviation of the histogram
 */
  sumX = 0.0;
  sumX2 = 0.0;
  max = 0;
  for (i=0; i<256; i++) {
    if ((i <= histMin) || (i >= histMax))
      histTotal[i] = 0;
    if (histTotal[i] > max)
      max = histTotal[i];
    sumX += histTotal[i];
    sumX2 += (histTotal[i] * histTotal[i]);
  } /* end if */
  if (histMax > 256)
    histMax = 256;
  if (histMin < -1)
    histMin = -1;
  elements = histMax-histMin-1;
  sigma = sqrt ((sumX2 - sumX*sumX / elements) / (elements-1));

/*
 * Draw the histogram in the bitplane
 */
  for (x=0; x<256; x++) {
    if (max == 0)
      point = 256;
    else
      point = histTotal[x] * 256 / max;
    for (y=0; y<point; y++)
      img[255-y][x] = 0;
    for (y=point; y<256; y++)
      img[255-y][x] = 255;
  } /* end for y */

/*
 * Write values in histTotal[] as bars to var1Id
 */
  edge [0] = 1;
  edge [1] = 256;
  edge [2] = 256;
  start[1] = 0;
  start[2] = 0;
  for (i=0; i<images; i++) {
    start [0] = i;
    ncvarput (ncId, var1Id, start, edge, (void *) img);
  }	/* for (i) */

/*
 * Go to define mode and add the standard deviation & average as attribute
 */
  ncredef (ncId);
  ncattput (ncId, var1Id, "StandardDeviation", NC_FLOAT, 1, (void*) &sigma);


/******************************************************************************
 *  E V A L U A T I O N   A N D   C L E A N U P
 ******************************************************************************
 */
/*
 * Report performance evaluation
 */
  pipsPrintPerformance (tid, perform, ntask, njobs);

/*
 * Free allocated memory
 */
  free (data);
  free (jobId);
  free (perform);
} /* pipsJobManager */







/******************************************************************************
 *  C H I L D   P R O C E S S  (on all hosts)
 ******************************************************************************
 *
 *   void pipsProcPar (void)
 *
 * Description   : 
 *
 *   Here the data PROCESSING is performed.
 *   Unpacks data received from the pipsJobManager (), processes it
 *   and packs the result data in again to be send back to the
 *   pipsJobManager. 
 */

#ifdef __STDC__

void pipsProcPar (void)

#else

pipsProcPar ()

#endif
{
/*
 * +++ Variables sent by the jobManager
 * NOTE: Declaring all variables 'static' increases speed dramatically
 */
  static long width;
  static long rows;

/*
 * +++ 'Backup' of above variables to keep track of changes in variables
 */
  static long currWidth = 0;
  static long currRows  = 0;

/*
 * +++ Variables for processing the data
 */
  static long x, y;
  static long hist[256];
  static unsigned char *data = NULL;

/*
 * +++ Define macros for better readability
 */
#define DATA(X,Y)   data   [(Y)*width + (X)]

/******************************************************************************
 * R E C E I V E   P R O C E S S   D A T A 
 ******************************************************************************
 * +++ Unpack data that has arrived from the jobmanager
 */
  pvm_upklong (&width, 1, 1);
  pvm_upklong (&rows, 1, 1);

/*
 * +++ Preparations after change of parameters (data from different image)
 * Memory has to be reallocated and variables re-initialized
 */
  if ((width != currWidth) ||
      (rows  != currRows)) {
    currWidth = width;
    currRows = rows;

/*
 * +++ Allocate memory for variables
 */
    if (data)               /* Free allocated memory first */
      free (data);
    data = (unsigned char *) malloc (width * rows * sizeof (char));
  } /* end if */

/*
 * +++ Receive the image data
 */
  pvm_upkubyte (data, width * rows, 1);
      
/******************************************************************************
 * +++ C A L C U L A T I O N S   O N   D A T A 
 ******************************************************************************
 */
/*
 * Initialize the histogram
 */
  for (x=0; x<256; x++)
    hist [x] = 0;

/*   
 * +++ Count the number of pixels for each colour
 */
  for ( y = 0; y < rows; y++ )
    for ( x = 0; x < width; x++ ) {
      (hist [DATA (x, y)])++;
    }

  pvm_initsend (PvmDataDefault);
/******************************************************************************
 * S E N D   R E S U L T   D A T A 
 ******************************************************************************
 * +++ Pack result data and send to jobmanager (sending done by main() )
 */
  pvm_pklong (hist, 256, 1);

} /* end pipsProcPar */


/* end histogram.c */
