/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*
 * Name          : ncprocseq.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : 7 Sep 1994
 * Modified on   : 
 * Synopsis      :
 *
 * Usage:
 *  ncprocseq [-h] ncName [-v varName] [n] [-r resName]
 *
 * Where:
 *   -h         Prints this help text.
 *   ncName     Name of netCDF file to process, must be first parameter.
 *   varName    Variable to get the data from.
 *   n          Number of image to process from 'varName'
 *   resName    Variable to put the result data in (create if non-existent)
 *
 * Description:
 *   ncprocseq opens 'ncName', and processes image no. 'n' of variable
 *   'varName'. Stores the result in variable 'resName'
 *   If varName is omitted, all variables are processed (if possible).
 *   If n is omitted, all images of a variable are processed.
 *   If resName is omitted, then 'varName' + ".res" is used as name.
 */

#include "pips.h"   /* PIPS system header file  */

/*
 * Global definitions
 */
/*
 * Description of program for display on stderr by pipsPrintError()
 */
char *programName = "ncprocseq";
char *description[] = {
  "  ncprocseq [-h] ncName [-v varName] [n] [-r resName] ",
  " ",
  "Where:",
  "  -h         Prints this help text.",
  "  ncName     Name of netCDF file to process, must be first parameter.",
  "  varName    Variable to get the data from.",
  "  n          Number of image to process from 'varName'",
  "  resName    Variable to put the result data in (create if non-existent)",
  " ",
  "Description:",
  "  ncprocseq opens 'ncName' and processes image no. 'n' of variable",
  "  'varName'. Stores the result in variable 'resName'",
  "  If varName is omitted, all variables are processed (if possible).",
  "  If n is omitted, all images of a variable are processed.",
  "  If resName is omitted, then 'varName' + '.res' is used as name.",
  ""
}; /* end description */


/******************************************************************************
 *   M A I N
 ******************************************************************************
 */
#ifdef __STDC__

int main (int argc, char *argv[])

#else

main (argc, argv)
    int argc;
    char *argv[];

#endif
{
/*
 * Variables for netcdf file
 */
  int  ncId      = -1;          /* netcdf file Id                            */
  char *ncName   = NULL;        /* Name of netcdf file                       */
  char varName [MAX_NC_NAME];   /* Name of variable to get the data from     */
  char *resName    = NULL;      /* Name of variable to put results in        */
  long imageNumber = -1;        /* Number of image to process (-1 means ALL) */
  long start[3];        /* Start of each datapacket               */
  long edge [3];        /* Shape (size) of each datapacket        */
  int dataId;           /* Id of variable that supplies the data to process  */
  int resId;            /* Id of variable that will get the results          */
  long width;           /* Width of image in variable 'varName' (dataId)     */
  long height;          /* Height of image in variable 'varName' (dataId)    */
  long image;           /* Number of images in an existing variable          */
  nc_type type;         /* Data type used in 'dataId' and 'resId'            */
  int fileType;         /* Value to use for file type attribute of 'resId'   */
  int resNameSet = 0;   /* Indicates, if user has given result variable name */
  char *name = NULL;    /* Buffer for constructing a result variable name    */
/*
 * Variables for the dataprocessing
 */

#define DATA(X,Y)   data   [(Y)*width + (X)]
#define RESULT(X,Y) result [(Y)*width + (X)]
  long x, y, wx, wy;
  long winSize;
  long sum;
  long middle;
  float normalizer;
  char *data;
  char *result;

/*
 * other variables
 */
  int i;                        /* index for argv[] */

/*
 * Init variables
 */
  start[0] = 0;  start[1] = 0;  start[2] = 0;
  edge [0] = 1;  edge [1] = 1;  edge [2] = 1;

/******************************************************************************
 *   A N A L Y Z E   C O M M A N D   L I N E   P A R A M E T E R S
 ******************************************************************************
 */

/*
 * If no parameters given, complain about missing netcdf filename
 */
    if (argc < 2)
      pipsPrintError (-1, PipsNetcdfMissing, NULL, PipsExit);

/*
 * If the -h option is given, print explanation and quit
 */
    if (strncmp ("-h", argv[1], 2) == 0)
      pipsPrintError (-1, PipsOk, NULL, PipsExit);

/*
 * First argument is expected to be the name of the netcdf file
 */
    ncName = argv[1];

/*
 * Examine argv[i]: Store a number in 'imageNumber',
 * and look for parameters '-v', '-r'
 */
  varName [0] = 0;
  varName [MAX_NC_NAME-1] = 0;
  for (i=2; i < argc; i++)
    if (! sscanf (argv[i], "%lu", &imageNumber))
      if (argv[i][0]) {
        if (i+1 >= argc) 
          pipsPrintError (-1, PipsSinglePara, argv[i], PipsExit);
        else
          switch (argv[i][1]) {
/*
 * '-v' option : store next parameter in 'varName'
 */
          case 'v' : strncpy (varName, argv[++i], MAX_NC_NAME-1);
                     break;
/*
 * '-r' option : store next parameter in 'resName'
 */
          case 'r' : resName = argv[++i];
                     break;
          default  : pipsPrintError (-1,PipsBadParameter,argv[i],PipsExit);
    } /* end switch */
  }  /* end if ! */


/******************************************************************************
 *  O P E N   N E T C D F   F I L E
 ******************************************************************************
 * Prevent error messages (error codes are evaluated for further calls)
 */
    ncopts = NC_NOERR;

/*
 * Open an EXISTING netCDF file 
 */
    ncId = ncopen (ncName, NC_WRITE);
    if (ncId == -1)
      pipsPrintError (-1, PipsErrorFileRead, ncName, PipsExit);

/*
 * If a result variable name is given, extract only one variable
 * (as writing all variables in the same result isn't very usefull)
 */
  resNameSet = (resName != NULL);
  if (resNameSet) {
    if (strlen (varName) == 0)
      ncvarinq (ncId, dataId, varName, 0, 0, 0, 0);
    pipsStrApp (&name, resName);
  } /* end if */

/*
 * As long as varInq() says there is still an image, do process it
 */
  while (pipsVarInq (ncId, &dataId, varName, &type, &imageNumber, &height, &width)) {

    printf ("Processing '%s', dataId %d, imageNumber %d\n", varName, dataId, imageNumber);
/*
 * If no result variable name was given, take the variable name
 */
    if (! resNameSet) {
      pipsStrApp (&name, varName);
      pipsStrApp (&name, ".res");
    } /* end if */

/*
 * Create a variable that can hold an image sequence of width x height
 * Set it's fileType attribute according to type of data used in 'dataId'
 */
    if (nctypelen(type) == nctypelen (NC_LONG))
      fileType = PipsPpmRaw;
    else
      fileType = PipsPgmRaw;
    resId = pipsVarDef (ncId, name, &type, &fileType, &image,&height,&width);

/******************************************************************************
 *  D A T A   P R O C E S S I N G   P R E P A R A T I O N S
 ******************************************************************************
 * Ask User about window size
 */
    printf ("Which window size to use ?:");
    scanf ("%ld", &winSize);
    printf ("\n");

/*
 *  Make window dimension odd to allow for a central element.  
 */
    if (winSize % 2 == 0)
      winSize++;

/*
 * Allocate memory for loading and saving of data
 */
    data = malloc (width * height * sizeof (char));
    result = malloc (width * height * sizeof (char));

/*
 *  Code simplifying varibles
 */
    middle = winSize / 2;
    normalizer = 255.0 / winSize / winSize;

/******************************************************************************
 *  L O A D   P R O C E S S   D A T A 
 ******************************************************************************
 */
    start [0] = imageNumber;
    edge  [1] = height;
    edge  [2] = width;
    ncvarget (ncId, dataId, start, edge, (void *) data);
    memcpy (result, data, width * height * sizeof (char));

/******************************************************************************
 *  C A L C U L A T I O N S   O N   D A T A 
 ******************************************************************************
 * Counts the number of adjacent pixels with same colourvalue
 */
/*   
 *  Loop over image   
 */
  for ( y = middle; y < height + middle; y++ )
    for ( x = middle; x < width - middle; x++ ) {
      sum = 0;
/* 
 *  Loop over window
 */  
      for ( wy = y-middle; wy < ( y + middle ); wy++ )
        for ( wx = x-middle; wx < ( x + middle ); wx++ )

/*
 *  Summing up the elements in the windows if they are in range   
 */  
          sum += (DATA (wx, wy) == DATA (x, y));

/*   
 *  Placing the mean of the window in the middle element   
 */
      RESULT (x, y-middle) = (char) (normalizer * sum);
    } /* end for x */                                                                                 

/******************************************************************************
 *  S T O R E   R E S U L T   D A T A 
 ******************************************************************************
 */
    ncvarput (ncId, resId, start, edge, (void *) result);
  } /* while pipsVarInq() */

/*
 * Commit changes and close netCDF-file
 */
  if (ncclose (ncId) == -1)
    pipsPrintError (-1, PipsErrorFileWrite, ncName, PipsExit);
  return (PipsOk);
} /* end main */


/* end ncprocseq.c */
