/* PIPS version 1.01: Parallel Information Processing System 
   Copyright (C) 1994, 95, 96 Free Software Foundation, Inc.

This file is part of GNU PIPS.

GNU PIPS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU PIPS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/*
 * Name          : ncattdef.c
 * Author        : Frank Heimes
 * Institution   : Imperial College of Science, Technology, and Medicine
 * Written on    : 7 Sep 1994
 * Modified on   : 
 * Synopsis      :
 *
 * Usage:
 *  ncattdef [-h] ncName [defName]
 *
 * Where:
 *   -h         Prints this help text.
 *   ncName     Name of netCDF file to process, must be first parameter.
 *   defName    Name of file containing attribute definitions in CDL syntax.
 *              If defName is omitted, stdin is read.
 *
 * Description:
 *   ncattdef parses the contents of 'defName' (or stdin) and adds global and
 *   variable attributes to 'ncName' if possible. This allows you to transfer
 *   attributes between netcdf files:
 *
 *        ncdump ncName1 | grep attributeName | ncattdef ncName2
 *
 *   Only valid CDL attribute definitions are accepted. Syntax errors are re-
 *   ported on stderr and parsing is continued.
 */

#include "pips.h"   /* PIPS system header file  */

/*
 * Global definitions
 */
/*
 * Description of program for display on stderr by pipsPrintError()
 */
char *programName = "ncattdef";
char *description[] = {
  "  ncattdef [-h] ncName [defName] ",
  " ",
  "Where:",
  "  -h         Prints this help text.",
  "  ncName     Name of netCDF file to process, must be first parameter.",
  "  defName    Name of file containing attribute definitions in CDL syntax.",
  "             If defName is omitted, stdin is read.",
  " ",
  "Description:",
  "  ncattdef parses defName and adds global and variable attributes to ncName",
  "  if possible. This allows you to transfer attributes between netcdf files:",
  "    ncdump ncName1 | grep attributeName | ncattdef ncName2 ",
  ""
}; /* end description */


/******************************************************************************
 *   M A I N
 ******************************************************************************
 */
#ifdef __STDC__

int main (int argc, char *argv[])

#else

main (argc, argv)
    int argc;
    char *argv[];

#endif
{
  char *ncName;                 /* User supplied name of netcdf file         */
  char *defName = NULL;         /* User supplied name of definitions file    */
  int ncId;                     /* Identifier of netcdf file                 */
  char *history  = NULL;        /* Saved command line parameters             */
  int histIndex;               /* Variable for constructing history string   */

/******************************************************************************
 *   A N A L Y Z E   C O M M A N D   L I N E   P A R A M E T E R S
 ******************************************************************************
 */
/*
 * If no parameters given, complain about missing netcdf filename
 */
  if (argc < 2)
	pipsPrintError (-1, PipsNetcdfMissing, NULL, PipsExit);

/*
 * If the -h option is given, print explanation and quit
 */
  if (strncmp ("-h", argv[1], 2) == 0)
    pipsPrintError (-1, PipsOk, NULL, PipsExit);

/*
 * First argument is expected to be the name of the netcdf file
 */
  ncName = argv[1];

/*
 * Extract definitions name
 */
  if (argv[2])
    defName = argv[2];

/******************************************************************************
 *  O P E N   N E T C D F   F I L E ,  C H A N G E   A T T R I B U T E S
 ******************************************************************************
 */
/*
 * Prevent error messages (i.e. don't abort for nonexisting variables)
 */
  ncopts = NC_NOERR;

/*
 * Open an EXISTING netCDF file 
 */
  ncId = ncopen (ncName, NC_WRITE);
  if (ncId == -1)
    pipsPrintError (-1, PipsErrorFileRead, ncName, PipsExit);

/*
 * Save the command line parameters in the history attribute (concatenated)
 */
    for (histIndex = 0; histIndex < argc; histIndex++) {
      pipsStrApp (&history, argv[histIndex]);
      pipsStrApp (&history, " ");
    } /* end for */
    pipsHistory (ncId, history);
    free (history);

/*
 * Prevent error messages (error codes are evaluated for further calls)
 */
  ncopts = NC_NOERR;

/*
 * Change attributes according to definitions in 'defName'
 */
   pipsChangeAtts (ncId, defName);

/*
 * Commit changes and close netCDF-file
 */
  if (ncclose (ncId) == -1)
    pipsPrintError (-1, PipsErrorFileWrite, ncName, PipsExit);
  return (PipsOk);
} /* end main */


/* end ncattdef.c */
