/*
* (C) 2000 David O'Toole $Date: 2001/04/10 18:27:29 $
* 
* machine loading, instantiation, communication
* and memory allocation 
* 
* $Revision: 1.9 $
*
* This software is distributed under the terms of the
* GNU General Public License (GPL). Read the included file
* COPYING for more information. 
*
*/
static const char rcsid[]="$Id: machine.c,v 1.9 2001/04/10 18:27:29 dto Exp $";
#include <dlfcn.h>  /* for UNIX shared-object support */
#include <stdio.h>
#include <stdlib.h> 
#include <dirent.h> /* directory-handling stuff */
#include <sys/types.h>
#include <string.h>
#include <assert.h>
#include "machine.h"
#include "package.h"
#include "pattern.h"

package global_oxapi_pkg; 
static machine_type type_registry[OX_MAX_TYPES];  
static int num_types = 0; 

//
// OPTION STRINGS TO MATCH UP WITH THE ENUMS IN MACHINE.H
//

char *mc_type_strings[] = {
		"note",
		"velocity",
		"generic",
		"trigger",
		"wave",
		NULL
};	

char *mc_widget_strings[] = {
		"slider",
		"option",
		"button",
		NULL
};                    


// 
// CREATING THE PACKAGE AND REGISTRY
// 


void mc_init_global_pkg(void) 
{
	static int is_initialized = 0; 
	package* p = &global_oxapi_pkg;
  
	if (is_initialized == 1) /* lazy evaluation */ 
		return; 
	
	is_initialized = 1; 
  
	p->alloc = pkg_alloc; 
	p->free = ox_free; 
	p->text2note = p_text2note;
	p->note2text = p_note2text; 
	p->text2freq = p_text2freq;
	p->note2freq = p_note2freq; 
	p->get_sr = p_get_sr;
}


// CREATE REGISTRY FROM DIRECTORY FULL OF MACHINE LIBS

int mc_make_registry(const char* machine_dir) {
   
	char *fn;
	char temp[1024];  /* for the concatenated filename */
	int ix = 0;     
	DIR *dp;
	struct dirent *ep;

	dp = opendir(machine_dir);
  
	if (dp) {
		while ((ep = readdir(dp))) {
			fn = ep->d_name;
			if ( strcmp(fn + (strlen(fn)-3), ".so") != 0 ) 
				continue; 
      
			// now build the full pathname in temp[] and load the machine 
			strcpy(temp, machine_dir);
			strcat(temp, "/");          // separator 
			strcat(temp, ep->d_name); 

			if (mc_load_type(&type_registry[ix], temp)) ix++; 
		}
		num_types = ix; 
	}
  
	else ox_die("%%OCTAL: Couldn't open machine_dir during call to load_type_registry() \n");

	if (ix == 0)
		ox_die("%%OCTAL: No machines found.");

	// PRINT BRIEF REPORT ABOUT MACHINES LOADED

	printf("\nSuccessfully loaded type registry from %s with %d machine types. \n", 
		machine_dir, ix);

	return num_types;
}


// 
// INVESTIGATING THE TYPE REGISTRY 
//

machine_type* mc_get_type(const char* name) 
{
	int i;

	if (!num_types) ox_die("%%OCTAL: No machine types loaded. \n");
  
	for (i=0; i<num_types; i++) {
		if (strcmp(name, type_registry[i].short_name) == 0)
			return(&type_registry[i]);
		else continue;  
	}
	return(0);  /* wasn't found */ 
}


machine_type* mc_get_type_by_index(int x)
{
	assert(x < num_types);
	return (type_registry + x);	
}

int mc_get_num_types() 
{
	return num_types;
}


//
// LOADING TYPES FROM SHARED OBJECTS
//

// attempts to match the input string against the array of strings,
// returning index within array if matched or -1 otherwise

static int match_name(char *needle, char **p)
{
	int i;
	
	for (i = 0; *p; p++, i++)
		if (strcmp(needle, *p) == 0) 
			return i;

	printf("Option <%s> not matched!\n", needle);
	return -1;
}


// parses a string into the param_spec structure specified
// returns TRUE on success

static int parse_param_info(char *s, param_spec *p)
{
	int k, r;
	char type[100], widget[100];

	assert(s);

	// remove leading/trailing delimiters 

	k = strlen(s) - 1;
	if (*s == '|') s++; 
	if (s[k] == '|')  s[k] = 0;  	

	r = sscanf(s, "%100[^|]|%100[^|]|%100[^|]|%100[^|]|%f|%f|%f",
			p->name, p->description, type, widget,
			&p->min, &p->max, &p->start); 	

	// get integer values from string key

	p->type = match_name (type, mc_type_strings);
	p->widget = match_name (widget, mc_widget_strings);

	printf ("<%s> (%d)\n", s, r);
	return (r == 7);	// make sure all were converted
}


// accepts a string filename, fills in the structure pointed to by t
// returns 0 on failure, otherwise returns value returned by machine's
// init() 

int mc_load_type(machine_type *t, const char *fn) 
{
	int ret, i;	
	char **cp;
	const char *error;

	t->so_handle = dlopen(fn, RTLD_NOW); 
	if (!t->so_handle) {
		 
		fprintf(stderr, "octal: %s: dlopen() returned NULL.\n", fn);
		error = dlerror();
		if (error) fprintf(stderr, "octal: dlerror() gives <%s>\n", error);
		return(0); 
	}
			
	t->ox_init    = dlsym(t->so_handle, "ox_init");
	t->ox_create  = dlsym(t->so_handle, "ox_create");
	t->ox_destroy = dlsym(t->so_handle, "ox_destroy");
	t->ox_event   = dlsym(t->so_handle, "ox_event");
	t->ox_work    = dlsym(t->so_handle, "ox_work");
	t->ox_desc    = dlsym(t->so_handle, "ox_desc");
	t->ox_channel = dlsym(t->so_handle, "ox_channel");
  	
	if (!t->ox_init || !t->ox_create || !t->ox_destroy 
			|| !t->ox_work || !t->ox_desc || !t->ox_event || !t->ox_channel) {
		fprintf(stderr, "%%OCTAL: %s is not a valid OCTAL machine.\n", fn);
		dlclose(t->so_handle);
		return(0);
	}
	ret = t->ox_init(t);			

	printf("\n\nLoading machine type %s (%s)\n", t->short_name, t->long_name);

	// now we parse the param_info stuff into param_spec structures

	for (i=0, cp=t->param_info; *cp; cp++, i++) {
		if (parse_param_info (*cp, t->param_specs + i)) continue;
		else {
			fprintf(stderr, "Error parsing param_info[%d] while loading machine %s !\n", i, t->short_name);
			return 0;
		}
	}
	t->num_params = i;

	return ret;
}


int mc_unload_type(machine_type *t) 
{
	dlclose(t->so_handle);
	
	t->so_handle  = 0;
  
	t->ox_init    = 0;
	t->ox_work    = 0;
	t->ox_create  = 0;
	t->ox_destroy = 0;
	t->ox_desc    = 0;
	t->ox_event   = 0;
	t->ox_channel = 0;

	return(1);	
}

//
// MACHINE INSTANCE-RELATED FUNCTIONS
//

// 
// CONSTRUCTOR AND DESTRUCTOR
// 


machine* mc_create(machine_type *t) 
{
	machine* m;
	int i;

	m = malloc(sizeof(machine));  
  
	// install oxapi package
 
	mc_init_global_pkg(); 
	m->pkg = &global_oxapi_pkg;

	// call create on machine

	t->ox_create(m);
	m->type = t;

	// initialize some fields to sane values

	m->frame = 0;
	m->num_channels = 0; 	
	m->param_state = NULL;
	strncpy(m->instance_name, m->type->short_name, MAX_NAME - 1);  // default 
	m->x = 300; m->y = 300; 

	for (i=0; i<MAX_CHANNELS; i++)
		m->channel_state[i] = 0;

	mc_channels_configure(m, 1);
	mc_allocate_outbuf(m, &m->lout, &m->rout); 

	if (!m->rout || !m->lout)
		ox_die("%%OCTAL: Memory allocation failed during create_machine()\n");

	return m; 
}


void mc_destroy(machine *m) 
{
	mc_channels_configure(m, 0);
	m->type->ox_destroy(m);
	mc_free_buf(&m->lout, &m->rout);
	m->type = NULL; 
	free(m);
}


// 
// CHANNEL-RELATED FUNCTIONS
//


static int add_channel(machine *m) 
{
	int i; 
	machine_type *t = m->type;
	param_spec *ps;
	int cn; 	// number of new channel

	if (m->num_channels == t->max_channels) return 0; 

	// tell machine to create new channel at proper index
	// then set default params in that track 
	// we don't want to send trigger or note events, though

	cn = m->num_channels++;
	t->ox_channel (m, 1, cn);
	for (i=0; i<t->num_params; i++) {
		ps = t->param_specs + i;
		if (ps->type == OX_TYPE_NOTE || ps->type == OX_TYPE_TRIGGER) continue;
		else mc_event (m, cn, i, ps->start);
	}

	return 0;
}


static int remove_channel(machine *m)
{
	int cn = m->num_channels--;
	m->type->ox_channel (m, 0, cn);

	return 1;
}


void mc_channels_configure(machine *m, int channels)
{
	param *temp;
	int i; 
	int size = sizeof(param) * m->type->num_params * channels;
	
	if (channels == m->num_channels) return; 
	assert (size);

	// allocate or deallocate storage for channels	

	temp = realloc (m->param_state, size);		// ok to pass null
	assert (temp);
	m->param_state = temp; 

	// now notify machine and initialize channel states to sane values

	for (i = m->num_channels; i < channels; i++)	// add channels if needed		
		add_channel (m);
	
	for (i = m->num_channels; i > channels; i--)	// remove channels if needed
		remove_channel (m);

	m->num_channels = channels;
}


// wrapper function that maintains current parameters

void mc_event(machine *m, int chan, int which_param, param value) 
{
	int index = chan * m->type->num_params + which_param;
	
	m->type->ox_event (m, chan, which_param, value);
	m->param_state[index] = value;	
}


//
// AUDIO BUFFER ALLOCATION 
//


void mc_allocate_outbuf(machine* m, samp** left, samp** right) 
{
	if (mc_is_stereo_out(m)) {
		*left = (samp*) ox_alloc(OX_OUTPUT_BUFFER_SIZE);
		*right = (samp*) ox_alloc(OX_OUTPUT_BUFFER_SIZE);
	}
  
	else { 
		*left = (samp*) ox_alloc(OX_OUTPUT_BUFFER_SIZE);
		*right = *left;
	}
}

     
void mc_free_buf(samp** left, samp** right) 
{
	if (*right == *left) 
		free(*right); 
  	else {
		free(*right); 
		free(*left);
  	}
    
	*right = 0;
	*left = 0; 
}


// 
// SOME CONVENIENCE FUNCTIONS
//

int mc_is_stereo_out(machine* m) {
	return (m->type->output_channels == OX_STEREO);
}


int mc_is_stereo_in(machine* m) {
	return (m->type->input_channels == OX_STEREO);
}


int wants_input(machine* m) {
	return (m->type->input_channels > 0);
}


// 
// MEMORY ALLOCATION WRAPPERS
// these might do something in the future

void* ox_alloc(size_t s)
{
	void* t; 

	t = malloc(s);
	// lock the memory if it worked? 
	return t; 
}

void* pkg_alloc(size_t s)
{
	void* t;

	t = ox_alloc(s);
	// destroy machine if this fails 
	return t; 
}

void ox_free(void* t)
{
	// munlock(t);  
	free(t);
}





















