/*
* (C) 2000, 2001 David O'Toole $Date: 2001/06/25 09:08:39 $
* 
* engine_view.c 
* this is the gtk+ engine viewer for OCTAL. 
*
* This software is distributed under the terms of the
* GNU General Public License (GPL). Read the included file
* COPYING for more information. 
*
*/

#include <gtk/gtk.h>
#include "engine_view.h"
#include <assert.h>
#include <stdio.h>
#include <math.h>
#include <string.h>

#define PI (3.14159)

enum {ENGINE_VIEW_SIZE = 600, ENGINE_MACHINE_SIZE = 45};

GdkCursor *drag_cursor, *normal_cursor, *connect_cursor;

// 
// CONTEXT MENU DATA 
// null values cause separators in the menu

char *mc_menu_text[] = {"control box", "rename", 0, "delete"};
void *mc_menu_cb[] = {control_box_cb, rename_cb, 0, remove_cb};


//
// USER MESSAGES
// collected here for later internationalization

static char connecting_message[] = "Move pointer to destination machine and release button to make connection.";
static char dragging_message[] = "Release button to drop machine in new position.";
static char default_message[] = "[Move: click + drag] [Connect: shift-click + drag] [Menu: right-click]";
static char delete_error_message[] = "%%% You can't delete that machine.";

// 
// INITIALIZATION AND DESTRUCTION
//

engine_view *ev_create(engine *subject) 
{	
	engine_view *self = malloc(sizeof(engine_view));
	GtkWidget *hpane, *vbox, *frame;
	GtkWidget *tm;
	int i, j, size;
	machine_type *type;  
	char buffer[1024];

	// let's know who we're looking at 

	self->subject = subject; 

	// set up those global cursors

	drag_cursor = gdk_cursor_new(GDK_FLEUR);
	normal_cursor = gdk_cursor_new(GDK_LEFT_PTR);
	connect_cursor = gdk_cursor_new(GDK_CROSS_REVERSE);

	// create main widgets 
 
	self->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	self->drawing = gtk_drawing_area_new(); 
	self->entry = gtk_entry_new_with_max_length(64);
	self->label = gtk_label_new(default_message);
	self->cbarbox = gtk_vbox_new(FALSE, 2);

	self->pixmap = 0; 	// can't create until a configure_event

	vbox = gtk_vbox_new(FALSE, 2); 
	hpane = gtk_hpaned_new();
	frame = gtk_frame_new(NULL);

	// set some properties
	
	gtk_window_set_title(GTK_WINDOW(self->window), "mixing & machine control");
	gtk_drawing_area_size(GTK_DRAWING_AREA(self->drawing), 
				ENGINE_VIEW_SIZE, ENGINE_VIEW_SIZE);  
	gtk_label_set_justify(GTK_LABEL(self->label), GTK_JUSTIFY_LEFT);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_OUT);
	
	// stick em together

	gtk_container_add(GTK_CONTAINER(self->window), vbox);
	gtk_container_add(GTK_CONTAINER(frame), self->label);

	gtk_box_pack_start(GTK_BOX(vbox), hpane, TRUE, TRUE, 0);
	gtk_box_pack_end(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	
	gtk_paned_pack1(GTK_PANED(hpane), self->drawing, TRUE, TRUE);
	gtk_paned_pack2(GTK_PANED(hpane), self->cbarbox, FALSE, FALSE);
	gtk_paned_set_gutter_size(GTK_PANED(hpane), 6);	

	// connect callbacks

	gtk_signal_connect(GTK_OBJECT(self->window),
		 "delete_event",
		 GTK_SIGNAL_FUNC(delete_event_cb),
		 self); 
	gtk_signal_connect(GTK_OBJECT(self->drawing),
		 "configure_event",
		 GTK_SIGNAL_FUNC(configure_event_cb),
		 self);
	gtk_signal_connect(GTK_OBJECT(self->drawing),
		 "expose_event",
		 GTK_SIGNAL_FUNC(expose_event_cb),
		 self);
	gtk_signal_connect(GTK_OBJECT(self->drawing),
		 "button_press_event",
		 GTK_SIGNAL_FUNC(button_press_event_cb),
		 self);
	gtk_signal_connect(GTK_OBJECT(self->drawing),
		 "button_release_event",
		 GTK_SIGNAL_FUNC(button_release_event_cb),
		 self);

	gtk_widget_add_events(GTK_WIDGET(self->drawing), GDK_BUTTON_PRESS_MASK);
	gtk_widget_add_events(GTK_WIDGET(self->drawing), GDK_BUTTON_RELEASE_MASK);

	// Right-click menu. There's a menu option for each machine type.  

	self->menu = gtk_menu_new();
	gtk_menu_set_title(GTK_MENU(self->menu), "Create Machine");
	size = mc_get_num_types();	

	for (j = 0; j < size; j++) {
		type = mc_get_type_by_index (j);
		sprintf (buffer, "%s (%s)", type->short_name, type->long_name);	
		tm = gtk_menu_item_new_with_label (buffer);
		gtk_menu_append (GTK_MENU(self->menu), GTK_WIDGET(tm)); 
		gtk_signal_connect(GTK_OBJECT(tm),
			"activate",
			new_machine_menu_cb,
			(gpointer)type);
		gtk_object_set_data(GTK_OBJECT(tm), "self", (gpointer)self);
	}

	// context menu for machines 

	self->mc_menu = gtk_menu_new();
	size = sizeof(mc_menu_text) / sizeof(mc_menu_text[0]);
	
	for (i=0; i<size; i++) {
		if (mc_menu_text[i]) tm = gtk_menu_item_new_with_label(mc_menu_text[i]);
		else tm = gtk_menu_item_new();
		gtk_menu_append(GTK_MENU(self->mc_menu), GTK_WIDGET(tm));
		if (mc_menu_cb[i])
			gtk_signal_connect(GTK_OBJECT(tm), "activate", mc_menu_cb[i], self);				
	}

	// clear flags
	
	self->dragging = 0;
	self->mc_menu_id = 0;
	self->connecting = 0;

	for (i=0; i<max_machines; i++)
		self->cb_homes[i] = 0;

	// if you don't do this, the menus turn really teensy!	
	
	gtk_widget_show_all(GTK_WIDGET(self->window));
	gtk_widget_show_all(GTK_WIDGET(self->menu));
	gtk_widget_show_all(GTK_WIDGET(self->mc_menu));

	return self;
}

void destroy_engine_view() 
{

}


// 
// INTERFACE RENDERING UTILITIES
// Locking is the responsibility of the caller for these functions
//

static void update_status(engine_view *self, char *message) 
{
	gtk_label_set_text(GTK_LABEL(self->label), message);
}


static void draw_machine(engine_view *self, machine *m, int nid) 
{
	int x = m->x, y = m->y;
	int dy = ENGINE_MACHINE_SIZE;
	int dx = dy + 20;

	char id[10];

	GtkWidget *widget = self->drawing;
	GdkDrawable *drawable = self->pixmap; 

	gtk_paint_box (widget->style,
		drawable,
		widget->state,
		GTK_SHADOW_OUT,
		NULL,
		widget,
		NULL,
		x, y, dx, dy);
		
	sprintf(id, "%d", nid);

	gtk_draw_string(widget->style,
		drawable,
		widget->state,
		m->x + 4,
		m->y + (dy/2),
		m->instance_name);

	gtk_draw_string(widget->style,
		drawable,
		widget->state,
		m->x + 8,
		m->y + (dy/2) + 9,
		id);

	// master machine must not get past this point.
	// this is a temporary fix until master is a real machine type

	if (strcmp(m->instance_name, "master") == 0) return;

	// mark "pure generators" with colored decal

	if (m->type->input_channels == 0) {
		gtk_draw_box(widget->style, 
			drawable,
			widget->state,
			GTK_SHADOW_ETCHED_OUT,
			x + 3,
			y + dy - 15, 
			10, 10);
	}
}


static void draw_connection(engine_view* self, int source, int sink)
{
	int sx, sy, kx, ky; // Source and sinK line locations  
	int dx, dy; 		// slope calc
	int mx, my;			// midpoints
	GtkWidget *widget = self->drawing;
	GdkDrawable *drawable = self->pixmap; 
	engine *en = self->subject;
	GdkPoint poly[3];
	int i; 
	double theta;		// angle arrow makes with the horizontal

 	sx = en->machines[source]->x  + (ENGINE_MACHINE_SIZE/2+10);
	sy = en->machines[source]->y  + (ENGINE_MACHINE_SIZE/2);
	kx = en->machines[sink]->x    + (ENGINE_MACHINE_SIZE/2+10);
	ky = en->machines[sink]->y    + (ENGINE_MACHINE_SIZE/2);	      

	mx = ((sx+kx) / 2);
	my = ((sy+ky) / 2);
 
	gdk_draw_line(drawable,
		widget->style->fg_gc[widget->state], sx, sy, kx, ky);

	// make a copy of the normal triangle
	for (i=0; i<3; i++)
		poly[i] = triangle[i];

	// determine angle and rotate polygon
	dx = kx-sx;
	dy = ky-sy;
	
	// thanks to richard todd for fixing angle bug that
	// happened when dy==0 and source is to left of sink

	if (dy == 0) theta = (dx<=0) ? 0 : PI;
		else {
			theta = atan(((double)dx)/((double)dy));
			theta = (PI/2.0) - theta; 	// we actually want the complementary angle
			if (sy < ky)
				theta += PI;
		}
	
	if (theta)
		poly_rotate(poly, 3, theta);

	// translate to midpoint of connection line and plot
	for (i=0; i<3; i++) {
		poly[i].x += mx;
		poly[i].y += my;
		}	

	gdk_draw_polygon(drawable, widget->style->fg_gc[GTK_STATE_PRELIGHT],
		TRUE, poly, 3);

}


static void draw_engine (engine_view *self)
{
	int i,j;  
	machine *m;
	engine *e = self->subject;
	int last; 

	assert(e);
	assert(self->pixmap);

	// first clear the backing store

	gtk_paint_flat_box (self->drawing->style,
		self->pixmap,
		GTK_STATE_ACTIVE,
		GTK_SHADOW_OUT,
		NULL,
		self->drawing,
		NULL,
		0, 0, 
		self->drawing->allocation.width,
		self->drawing->allocation.height);	

	// now lock engine and start drawing 

	pthread_mutex_lock (&self->subject->mutex);
	
		last = en_get_last_machine (e);	

		// draw connections
		for (i=0; i <= last; i++)
			for (j=0; j <= last; j++)
				if (en_is_connected (e, i, j) )
					draw_connection (self, i, j);

		// and then the machines  
		for(i=0; i <= last; i++) {
			m = en_get_machine (e, i); 
			if (m) 
				draw_machine (self, m, i);
		}

	pthread_mutex_unlock (&self->subject->mutex);

	gtk_widget_queue_draw (self->drawing);
}



//
// HIT TESTING
//

static int hit_test (engine_view *self, int x, int y) 
{
	int i, q;
	machine *m;
	engine *e = self->subject;

	q = en_get_last_machine (e); 

	for(i=q; i>=0; i--) {
		m = en_get_machine (e, i); 
		if (m) 
 			if ( x > m->x && x < m->x+ENGINE_MACHINE_SIZE+20 && y > m->y && y < m->y+ENGINE_MACHINE_SIZE  ) 
				return i;  	
	
  	}      
	return -1;
}

static int connection_hit_test (engine_view *self, int x, int y)
{
	return 0;
}


//
// GEOMETRY
//
// I'm pretty glad I took linear algebra now!

const GdkPoint triangle[] = {{10, 10}, {-7, 0}, {10, -10}};


void poly_rotate (GdkPoint *p, int num_points, double theta) 
{
	int j;
	double x, y;

	for (j = 0; j < num_points; j++) {
		x = p[j].x; y = p[j].y;
		p[j].x = ((float)x * cos(theta)) - ((float)y * sin(theta));
		p[j].y = ((float)y * cos(theta)) + ((float)x * sin(theta));
	}	
}


//
// GTK+ EVENT CALLBACKS :: MAIN WINDOW EVENTS
//

gint delete_event_cb(GtkWidget* w, GdkEventAny* e, gpointer data) 
{
  gtk_main_quit();  /* REMOVE THIS LATER! */ 
  return(FALSE); 
}


// 
// GTK+ EVENT CALLBACKS :: DRAWING AREA EVENTS

// copy relevant part of existing pixmap to the screen

gint expose_event_cb (GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	engine_view *self = data;
	GdkPixmap *pixmap = self->pixmap; 

	assert(self && pixmap); 

	gdk_draw_pixmap (widget->window,
		widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
		pixmap,
		event->area.x, event->area.y,
		event->area.x, event->area.y, 
		event->area.width, event->area.height);  
	
	return FALSE;
}


// create a new backing pixmap of the appropriate size

gint configure_event_cb(GtkWidget *widget, GdkEventConfigure *event, gpointer data) 
{
	engine_view *self = data;	
	
	assert(self);	
	if (!self->subject) return TRUE;

	if (self->pixmap)
		gdk_pixmap_unref (self->pixmap);

	self->pixmap = gdk_pixmap_new (widget->window, widget->allocation.width,	
			widget->allocation.height, -1);

	draw_engine (self); 

	return TRUE; 
}


// start dragging on left click, or pop up menu on right click
// also, we must take care of connection attempts

gint button_press_event_cb(GtkWidget* w, GdkEventButton* event, gpointer data) 
{
	int i; 
	engine_view *self = data;	
	GtkWidget *menu;
	GdkCursor *cursor;

	assert(self);
	assert(self->subject);

	switch (event->button) {
	
		case 1:
		i = hit_test(self, event->x, event->y);	// touching a machine?
		if (i > -1) {
			self->drag_id = i;
			if ((event->state & GDK_SHIFT_MASK) == GDK_SHIFT_MASK) {
				self->connecting = 1;
				update_status(self, connecting_message);
				cursor = connect_cursor;
			}
			else {
				self->dragging = 1;
				update_status(self, dragging_message);
				cursor = drag_cursor;
			}	
			gdk_window_set_cursor(self->window->window, cursor);
		}
		break;

		case 2: // nothing for middle button yet
		break;

		case 3:	// either new machine menu or context menu will come up
		i = hit_test(self, event->x, event->y);
		menu = (i == -1) ? self->menu : self->mc_menu;
		self->mc_menu_id = i; 			// in case machine menu comes up
		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, event->button, event->time);
		break;		

		// reserved for mouse wheel operations	
		case 4:
		case 5:
		default:
	
	} 
	return(TRUE); 
}



// handle completion of connections and reposition draggings
// also, filter out stupid things like routing input to generators etc

gint button_release_event_cb(GtkWidget* w, GdkEventButton* event, gpointer data)
{
	double x, y;
	int source, sink; 
	engine_view *self = data;	
	int sig; 
	int did_something = 0;
	engine *en = self->subject;

	assert(self);
	// we only care about button releases in the case of drags and connections
	
	if (self->dragging) {
		x = event->x - (ENGINE_MACHINE_SIZE/2);
		y = event->y - (ENGINE_MACHINE_SIZE/2);	

		pthread_mutex_lock(&self->subject->mutex);
			en_position(self->subject, self->drag_id, x, y);
		pthread_mutex_unlock(&self->subject->mutex);

  		self->dragging = 0; 
  		self->drag_id = -1;
		did_something = 1;	

		// update the interface with new machine info

		// FIXME: WE NEED A LOCKING GET_MACHINE_NAME FUNCTION!	
	}
	else if (self->connecting) {

		source = self->drag_id;
		sink = hit_test(self, event->x, event->y);

		if (sink == -1) return TRUE;
		if (source == sink) return TRUE;

		pthread_mutex_lock(&en->mutex);
			sig = en_is_connected(en, source, sink) ? 0 : 1; 
			en_connect(en, sig, source, sink);
		pthread_mutex_unlock(&en->mutex);

		self->connecting = 0;
		self->drag_id = -1;
		did_something = 1;
	}
	gdk_window_set_cursor(self->window->window, normal_cursor);
	update_status(self, NULL);	

	if (did_something)
			draw_engine (self);

	return TRUE;
}


// 
// ETC
// I really hope I find a better way of doing this

int is_master(machine *m)
{
	return (strcmp(m->instance_name, "master") == 0);
}



//
// GTK+ EVENT CALLBACKS: MENUS AND DIALOG BOXES
//

void rename_cb(GtkMenuItem *menuitem, gpointer data)
{
	engine_view *self = data;	
	engine *en = self->subject;
	char *newname = gtk_entry_get_text(GTK_ENTRY(self->entry));

	if (self->mc_menu_id <= 0) return;

	pthread_mutex_lock(&en->mutex);
		en_name(en, self->mc_menu_id, newname);
	pthread_mutex_unlock(&en->mutex);

	// repaint

	draw_engine (self);
}


void remove_cb(GtkMenuItem *menuitem, gpointer data)
{
	engine_view *self = data;
	engine *en = self->subject;
	int ix = self->mc_menu_id;
	control_box **cbp = &self->cb_homes[ix];

	// not allowed to delete master machine

	if (ix <= 0) {
		update_status(self, delete_error_message);
		return;
	}

	// get rid of control box if one is open for this machine
	
	if (*cbp) cb_destroy(cbp);

	pthread_mutex_lock(&en->mutex);
		en_remove(en, ix);
	pthread_mutex_unlock(&en->mutex);

	draw_engine (self);
}


void control_box_cb(GtkMenuItem *menuitem, gpointer data) 
{
	engine_view *self = data;
	int x = self->mc_menu_id;
	control_box **cbp = &self->cb_homes[x];
	
	if (x <= 0) return;	// can't do this with master yet

	// nothing will happen if control box already exists

	cb_create(cbp, self->subject, x, self->cbarbox);
}


void new_machine_menu_cb(GtkMenuItem *menuitem, gpointer data)
{
	machine_type *type = data;
	machine *m; 
	engine_view *self = gtk_object_get_data(GTK_OBJECT(menuitem), "self");
	engine *en = self->subject;

	printf("new_machine_menu_cb with %p\n", type);
	assert(type);
	assert(self);

	m = mc_create (type);
	
	pthread_mutex_lock (&en->mutex);
		en_add (self->subject, m);
	pthread_mutex_unlock (&en->mutex);

	draw_engine (self);
}

