
/*
   XGWindow.m

   NSWindow for GNUstep GUI X/GPS Backend.

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: February 1996
   Author:  Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: May 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998

   This file is part of the GNUstep GUI X/GPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <stdlib.h>

#define BOOL XWINDOWSBOOL							// prevent X windows BOOL
#include <X11/Xmd.h>								// warning
#undef BOOL

#include <gnustep/xgps/XGContext.h>
#include <gnustep/xgps/XGGState.h>
#include <AppKit/GSWraps.h>
#include <Foundation/NSDebug.h>

#define CUR_CONTEXT	((XGContext*)[XGContext currentContext])

@interface XGWindowView : NSView
{
}
@end

@implementation XGWindowView
@end

//
// WindowMaker window manager interaction
//
typedef struct {
    CARD32 flags;
    CARD32 window_style;
    CARD32 window_level;
    CARD32 reserved;
    Pixmap miniaturize_pixmap;		// pixmap for miniaturize button
    Pixmap close_pixmap;		// pixmap for close button
    Pixmap miniaturize_mask;		// miniaturize pixmap mask
    Pixmap close_mask;			// close pixmap mask
    CARD32 extra_flags;
} GNUstepWMAttributes;

#define GSWindowStyleAttr	(1<<0)
#define GSWindowLevelAttr	(1<<1)
#define GSMiniaturizePixmapAttr (1<<3)
#define GSClosePixmapAttr	(1<<4)
#define GSMiniaturizeMaskAttr	(1<<5)
#define GSCloseMaskAttr		(1<<6)
#define GSExtraFlagsAttr	(1<<7)


//
// Backend structure for XRWindow
//
typedef struct _XGWindow_struct
{
  int			window_number;
  Window		xWindow;
  Display		*xDisplay;
  GC			context;
  BOOL			is_exposed;
  NSMutableArray	*exposedRects;
  Drawable		xPixmap;	// The pixmap used as backing store
  int			x;
  int			y;
  int			width;
  int			height;
  int			depth;
  Region		region;		// Used between several expose events
  XWMHints		gen_hints;	// Give window manager hints.
  XSizeHints		siz_hints;
  GNUstepWMAttributes	win_attrs;

} XGWindow_struct;

#ifndef LIB_FOUNDATION_LIBRARY
static void __NSRetainNothing(void *table, const void *anObject)	{}
static void __NSReleaseNothing(void *table, void *anObject)			{}

static NSString* __NSDescribeObjects(void *table, const void *anObject)
{
  return [(NSObject*)anObject description];
}

static const NSMapTableValueCallBacks NSNonRetainedObjectMapValueCallBacks = {
		(void (*)(NSMapTable *, const void *))__NSRetainNothing,
		(void (*)(NSMapTable *, void *))__NSReleaseNothing,
		(NSString *(*)(NSMapTable *, const void *))__NSDescribeObjects
};
#endif /* LIB_FOUNDATION_LIBRARY */


#define XWINDOW (((XGWindow_struct *)be_wind_reserved)->xWindow)
#define XGC (((XGWindow_struct *)be_wind_reserved)->context)
#define exposedRects (((XGWindow_struct *)be_wind_reserved)->exposedRects)
#define xPixmap (((XGWindow_struct *)be_wind_reserved)->xPixmap)
#define X (((XGWindow_struct *)be_wind_reserved)->x)
#define Y (((XGWindow_struct *)be_wind_reserved)->y)
#define WIDTH (((XGWindow_struct *)be_wind_reserved)->width)
#define HEIGHT (((XGWindow_struct *)be_wind_reserved)->height)
#define DEPTH (((XGWindow_struct *)be_wind_reserved)->depth)
#define region (((XGWindow_struct *)be_wind_reserved)->region)
#define XDISPLAY (((XGWindow_struct *)be_wind_reserved)->xDisplay)
#define GEN_HINTS (((XGWindow_struct *)be_wind_reserved)->gen_hints)
#define SIZ_HINTS (((XGWindow_struct *)be_wind_reserved)->siz_hints)
#define WIN_ATTRS (((XGWindow_struct *)be_wind_reserved)->win_attrs)



static	Atom win_decor_atom = None;

@implementation XGWindow

//
// Class variables
//
static NSMapTable* windowsToXWindows = NULL;
static NSMapTable* windowsToTags = NULL;

//
// Class Initialization
//
+ (void) initialize
{
  if (self == [XGWindow class])
    {
      windowsToXWindows = NSCreateMapTable(NSNonOwnedPointerMapKeyCallBacks,
			    NSNonRetainedObjectMapValueCallBacks, 20);
      windowsToTags = NSCreateMapTable(NSIntMapKeyCallBacks,
			NSNonRetainedObjectMapValueCallBacks, 20);
    }
}

+ (XGWindow*)_windowForXWindow: (Window)xWindow
{
  return NSMapGet(windowsToXWindows, (void*)xWindow);
}

+ (NSWindow*)_windowWithTag: (int)windowNumber
{
  return NSMapGet(windowsToTags, (void*)windowNumber);
}

+ (NSView *)_windowViewWithFrame: (NSRect)frameRect
{
  return [[XGWindowView alloc] initWithFrame: frameRect];
}

//
// Instance methods
//
- (id) initWithContentRect: (NSRect)contentRect
		 styleMask: (unsigned int)aStyle
		   backing: (NSBackingStoreType)bufferingType
		     defer: (BOOL)flag
		    screen: (NSScreen*)aScreen;
{
  static int	windowTag = 0;
  int		xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];
  XGContext	*context = (XGContext *)[XGContext currentContext];
  NSPoint	origin;
  NSSize	size;
  Window	xAppRootWindow = [context xAppRootWindow];
  XSetWindowAttributes winattrs;
  unsigned long valuemask2;
  XGCValues values;
  unsigned long valuemask = (GCForeground | GCBackground);
  XClassHint chint;

  be_wind_reserved = calloc(1, sizeof(XGWindow_struct));

  XDISPLAY = [context xDisplay];
  exposedRects = [NSMutableArray new];
  region = XCreateRegion();
  xPixmap = None;

  // Provide some reasonable values for the contentRect size so the
  // X Window server doesn't complain
  if (!contentRect.size.width)
    contentRect.size.width = 200;
  if (!contentRect.size.height)
    contentRect.size.height = 100;

  if ([context->windowList indexOfObjectIdenticalTo: self] == NSNotFound)
    [context->windowList addObject: self];
  [super initWithContentRect: contentRect
		   styleMask: aStyle
		     backing: bufferingType
		       defer: flag
		      screen: aScreen];

  NSDebugLog(@"XGWindow designated initializer\n");

  // get a unique integer
  window_num = ++windowTag;
  // translate NSWindow's coords to X coords

  size.width = contentRect.size.width;
  size.height = contentRect.size.height;
  origin.x = contentRect.origin.x;
  origin.y = contentRect.origin.y + contentRect.size.height;
  origin.y = DisplayHeight(XDISPLAY, xScreen) - origin.y;

  // set default colors
  valuemask2 = (CWBackPixel|CWBorderPixel);
  winattrs.border_pixel = [(XGColor *)[NSColor blackColor] xColor].pixel;
  winattrs.background_pixel = [(XGColor *)background_color xColor].pixel;
  // create an X window
  XWINDOW = XCreateWindow(XDISPLAY, RootWindow(XDISPLAY, xScreen),
	      (int)origin.x, (int)origin.y,
	      (int)size.width, (int)size.height,
	      0,
	      CopyFromParent,
	      CopyFromParent,
	      CopyFromParent,
	      valuemask2,
	      &winattrs);
  // Create an X GC for the content view set it's colors per chosen NSColor
  values.foreground = winattrs.background_pixel;
  values.background = winattrs.background_pixel;
  values.function = GXcopy;
  valuemask = (GCForeground | GCBackground | GCFunction);
  XGC = XCreateGC(XDISPLAY, RootWindow(XDISPLAY,xScreen), valuemask,&values);

  // Set the X event mask
  XSelectInput(XDISPLAY, XWINDOW, ExposureMask | KeyPressMask |
				KeyReleaseMask | ButtonPressMask |
			     ButtonReleaseMask | ButtonMotionMask |
			   StructureNotifyMask | PointerMotionMask |
			       EnterWindowMask | LeaveWindowMask |
			       FocusChangeMask | PropertyChangeMask |
			    ColormapChangeMask | KeymapStateMask |
			    VisibilityChangeMask);

  GEN_HINTS.window_group = xAppRootWindow;
  GEN_HINTS.initial_state = NormalState;
  GEN_HINTS.input = True;
  GEN_HINTS.flags = StateHint | InputHint | WindowGroupHint;
  XSetWMHints(XDISPLAY, XWINDOW, &GEN_HINTS);
  // WindowMaker ignores the frame origin
  // unless it's also specified as a hint
  SIZ_HINTS.flags = PPosition | USPosition;
  SIZ_HINTS.x = origin.x;
  SIZ_HINTS.y = origin.y;
  XSetNormalHints(XDISPLAY, XWINDOW, &SIZ_HINTS);

#if 0
  // set class hints so WM creates an app icon
  chint.res_class = "XGWindow";
  chint.res_name = (char*)[[[NSProcessInfo processInfo] processName] cString];
  XSetClassHint(XDISPLAY, xAppRootWindow, &chint);
#endif

//  if (!flag)
    [self _updateWindowGeometry];

  // push the arrow as the default cursor
  [[NSCursor arrowCursor] push];

  // if window has close button inform WM that we can handle WM_DELETE_WINDOW
  if (aStyle & NSClosableWindowMask)
    {
      Atom protocols_atom;
      Atom delete_win_atom;

      protocols_atom = XInternAtom(XDISPLAY, "WM_PROTOCOLS", False);
      delete_win_atom = XInternAtom(XDISPLAY, "WM_DELETE_WINDOW", False);

      if ((delete_win_atom != (Atom)None) && (protocols_atom != (Atom)None))
	XSetWMProtocols(XDISPLAY, XWINDOW, &delete_win_atom, 1);
    }

  WIN_ATTRS.flags = GSWindowStyleAttr | GSWindowLevelAttr;
  WIN_ATTRS.window_style = aStyle;
  WIN_ATTRS.window_level = [self level];
  if ((win_decor_atom == (Atom)None))
    win_decor_atom = XInternAtom(XDISPLAY,"_GNUSTEP_WM_ATTR", False);

  // send WindowMaker WM window style hints
  if ((win_decor_atom != (Atom)None))
    XChangeProperty(XDISPLAY, XWINDOW, win_decor_atom, win_decor_atom,
		    32, PropModeReplace, (unsigned char *)&WIN_ATTRS,
		    sizeof(GNUstepWMAttributes)/sizeof(CARD32));

  // Insert window into the mapping from X windows to NS windows
  NSMapInsert (windowsToXWindows, (void*)XWINDOW, self);
  NSMapInsert (windowsToTags, (void*)window_num, self);
  NSDebugLog(@"XGWindow end of designated initializer\n");

  // Create a gstate
  DPSgsave(context);
  DPSsetgcdrawable(context, [self xGC], [self xDrawable], 0, size.height);
  DPSinitmatrix(context);
  DPSinitclip(context);
  DPSgstate(context);
  gstate = GSWDefineAsUserObj(context);
  DPSgrestore(context);

  return self;
}

- (void) close
{
  [super close];
  if (is_released_when_closed)
    {
      NSApplication *theApp = [NSApplication sharedApplication];
      XGContext *ctxt = (XGContext*)[XGContext currentContext];

      [ctxt->windowList removeObjectIdenticalTo: self];
    }
}

- (void) dealloc
{
  NSMapRemove(windowsToXWindows, (void*)XWINDOW);
  NSMapRemove(windowsToTags, (void*)[self windowNumber]);

  if (gstate)
    {
      XGContext* context = (XGContext *)GSCurrentContext();
      if (context)
        DPSundefineuserobject(context, gstate);
    }

  if (XWINDOW)
    {
      XDestroyWindow(XDISPLAY, XWINDOW);		// Destroy the X Window
      XWINDOW = 0;
      XFreeGC (XDISPLAY, XGC);				// Free X window's GC
    }

  [exposedRects release];

  if (xPixmap)
    {
      XFreePixmap (XDISPLAY, xPixmap);
      xPixmap = None;
    }

  XDestroyRegion (region);

  free(be_wind_reserved);
  [super dealloc];
}

#if 0
- (void) setLevel: (int)newLevel
{
  [super setLevel: newLevel];
  WIN_ATTRS.window_level = [self level];
  if ((win_decor_atom != (Atom)None))
    XChangeProperty(XDISPLAY, XWINDOW, win_decor_atom, win_decor_atom,
		    32, PropModeReplace, (unsigned char *)&WIN_ATTRS,
		    sizeof(GNUstepWMAttributes)/sizeof(CARD32));
}
#endif

- (void) setTitle: (NSString*)aString
{
  XTextProperty windowName;

  [super setTitle: aString];

  if (window_title && XWINDOW)
    {
      const char* newTitle = [window_title cString];

      XStringListToTextProperty((char**)&newTitle, 1, &windowName);
      XSetWMName(XDISPLAY, XWINDOW, &windowName);
      XSetWMIconName(XDISPLAY, XWINDOW, &windowName);
    }
}

- (void) performMiniaturize: (id)sender
{
  int xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];

  NSLog(@"XGWindow performMiniaturize\n");
  XIconifyWindow(XDISPLAY, XWINDOW, xScreen);
  [super performMiniaturize: sender];
}

- (void) setContentViewSize: (NSSize)aSize
{
  NSRect r;

  r.origin = NSZeroPoint;
  r.size = aSize;
  if (content_view)
    [content_view setFrame: r];
}

- (void) flushWindow
{
  int x, y, width, height;
  XGCValues values;
  unsigned long valuemask;
  XGContext	*ctxt;
  unsigned	i;

  // do nothing if backing is not buffered
  if (backing_type == NSBackingStoreNonretained)
    return;

  if (disable_flush_window)		// if flushWindow is called
    {					// while flush is disabled
      needs_flush = YES;		// mark self as needing a
      return;				// flush, then return
    }

  /* Check for special case of flushing while we are lock focused.
     For instance, when we are highlighting a button. */
  if (NSIsEmptyRect(rectNeedingFlush))
    {
      if ([rectsBeingDrawn count] == 0)
	{
	  needs_flush = NO;
	  return;
	}
    }

  /*
   * Accumulate the rectangles from all nested focus locks.
   */
  i = [rectsBeingDrawn count];
  while (i-- > 0)
    {
      rectNeedingFlush = NSUnionRect(rectNeedingFlush, 
       [[rectsBeingDrawn objectAtIndex: i] rectValue]);
    }
  
  ctxt = CUR_CONTEXT;
  /* FIXME: Doesn't take into account any offset added to the window
     (from PSsetXgcdrawable) or possible scaling (unlikely in X-windows,
     but what about other devices?) */
  rectNeedingFlush.origin.y = NSHeight(frame) - NSMaxY(rectNeedingFlush);

  values.function = GXcopy;
  values.plane_mask = AllPlanes;
  values.clip_mask = None;
  valuemask = (GCFunction | GCPlaneMask | GCClipMask);
  XChangeGC(XDISPLAY, XGC, valuemask, &values);

  x = rectNeedingFlush.origin.x;		// width/height seems
  y = rectNeedingFlush.origin.y;		// to require +1 pixel
  width = rectNeedingFlush.size.width + 1;	// to copy out
  height = rectNeedingFlush.size.height + 1;

  NSDebugLLog (@"NSWindow", 
	       @"copy X rect ((%d, %d), (%d, %d))", x, y, width, height);

  if (width > 0 || height > 0)
    XCopyArea (XDISPLAY, xPixmap, XWINDOW, XGC, x, y, width, height, x, y);

  needs_flush = NO;
  rectNeedingFlush = NSZeroRect;
  XFlush(XDISPLAY);
}

- (void) update
{
  [super update];
  XFlush(XDISPLAY);
}

//*****************************************************************************
//
//		Non OPENSTEP spec Instance methods
//
//*****************************************************************************

- (void)_captureMouse: sender
{
  int ret = XGrabPointer([XGContext currentXDisplay], [self xWindow], False,
		     PointerMotionMask | ButtonReleaseMask | ButtonPressMask,
		     GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

  if (ret != GrabSuccess)
    NSLog(@"Failed to grab pointer\n");
  else
    {
      NSApplication *theApp = [NSApplication sharedApplication];
      NSDebugLLog(@"NSWindow", @"Grabbed pointer\n");
      [(XGApplication *)theApp setGrabXWindow: [self xWindow]];
    }
}

- (void)_releaseMouse: sender
{
  XUngrabPointer([XGContext currentXDisplay], CurrentTime);
  [(XGApplication *)[NSApplication sharedApplication] setGrabXWindow: 0];
}

- (void)_updateWindowGeometry
{
  int xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];
  unsigned int borderWidth;
  Window rootWindow;
  int oldWidth = WIDTH;
  int oldHeight = HEIGHT;
  int x, y;


  // XGetGeometry returns coordinates relative to it's parent which is the	
  // window manager's title bar if it has one
  if (XWINDOW) 
    {
      int	screenHeight = DisplayHeight(XDISPLAY, xScreen);
      Window	qRoot;
      Window	qParent;
      Window	*qChildren;
      unsigned	qNum;

      XFlush (XDISPLAY);
      if (!XGetGeometry (XDISPLAY, XWINDOW, &rootWindow, &x, &y, &WIDTH, 
			&HEIGHT, &borderWidth, &DEPTH))
	return;
      if (!XQueryTree (XDISPLAY, XWINDOW, &qRoot, &qParent, &qChildren, &qNum))
	return;
      XFree(qChildren);

      // set the NSWindow frame per the actual X Window frame.	if the
      // window either has no title or is a menu just use the value
      // returned by XGetGeometry().
//	if (!(style_mask & NSTitledWindowMask))
      if (qParent == qRoot)
	{		
	  frame.origin = NSMakePoint(x, screenHeight - (y + HEIGHT));
	  X = x;
	  Y = y;
	}
      else
	{
	  // window has a title bar so have X server
	  // translate to display coordinates first
	  XTranslateCoordinates(XDISPLAY, XWINDOW, 
				RootWindow(XDISPLAY, xScreen), 
				x, y, &X, &Y, &rootWindow);
	  frame.origin = NSMakePoint(X, y + screenHeight - (Y + HEIGHT));
	}

      SIZ_HINTS.x = frame.origin.x;
      SIZ_HINTS.y = screenHeight - (frame.origin.y + frame.size.height);
      XSetNormalHints(XDISPLAY, XWINDOW, &SIZ_HINTS);

      frame.size = NSMakeSize(WIDTH, HEIGHT);
      NSDebugLLog (@"NSWindow", @"window geometry user ((%f, %f), (%f, %f)), "
			      @"device ((%d, %d), (%d, %d))",
			      frame.origin.x, frame.origin.y, 
			      frame.size.width, frame.size.height,
			      X, Y, WIDTH, HEIGHT);

      if (oldWidth != WIDTH || oldHeight != HEIGHT) 
	{
	  [self _createBackingStore];
	}
    }
}

- (void) xSetFrameFromXFrame: (NSRect)rect
{
  BOOL		onlyMoved = NO;
  NSRect	oldFrame;

  // Determine if the window was only moved
  if (rect.size.width == WIDTH && rect.size.height == HEIGHT &&
      (rect.origin.x != X || rect.origin.y != Y))
    onlyMoved = YES;

  oldFrame = frame;
  [self _updateWindowGeometry];
  frame = oldFrame;

  if (onlyMoved == NO)
    {
      /*
       * if no backing  store we need to redisplay window
       */
      [self setContentViewSize: rect.size];
      if (backing_type != NSBackingStoreNonretained)
	[[content_view superview] setNeedsDisplay: YES];
      [self update];
    }
}

- (Drawable) xDrawable
{
  if (backing_type != NSBackingStoreNonretained)
    return xPixmap;// backing store
  else
    return XWINDOW;// X window
}

- (void) _updateGState
{
  if (gstate)
    {
      XGContext* context = (XGContext *)GSCurrentContext();
      DPSgsave(context);
      DPSsetgstate(context, gstate);
      DPSsetgcdrawable(context, XGC, [self xDrawable], 0, HEIGHT);
      DPSinitmatrix(context);
      DPSinitclip(context);
      DPSgrestore(context);
    }
}

- (void)_createBackingStore
{
  int xScreen;

  if (backing_type == NSBackingStoreNonretained)
    {
      [self _updateGState];
      return;
    }

  NSDebugLLog (@"NSWindow", @"_createBackingStore");

  if (xPixmap)
    XFreePixmap (XDISPLAY, xPixmap);

  NSDebugLLog (@"NSWindow", @"create backing store: width = %d, height = %d, depth = %d",
	      WIDTH, HEIGHT, DEPTH);

  xScreen = [(XGScreen *)[NSScreen mainScreen] xScreen];
  xPixmap = XCreatePixmap (XDISPLAY, RootWindow(XDISPLAY, xScreen),
			  WIDTH, HEIGHT, DEPTH);
  if (!xPixmap)
    {
      NSLog (@"cannot create the pixmap for backing store!");
      return;
    }

  // Clear the pixmap to avoid artifacts
  XFillRectangle (XDISPLAY, xPixmap, XGC, 0, 0, WIDTH, HEIGHT);

  // Update our gstate
  [self _updateGState];
}

// process expose event
- (void)_addExposedRectangle: (XRectangle)rectangle
{
  if (backing_type != NSBackingStoreNonretained)
    {
      XGCValues values;
      unsigned long valuemask;

      // window has a backing store so just copy the exposed rect from the
      // pixmap to the X window

      NSDebugLLog (@"NSWindow", @"copy exposed area ((%d, %d), (%d, %d))",
		  rectangle.x, rectangle.y, rectangle.width, rectangle.height);

      values.function = GXcopy;
      values.plane_mask = AllPlanes;
      values.clip_mask = None;
      values.foreground = [(XGColor*)background_color xColor].pixel;
      valuemask = (GCFunction | GCPlaneMask | GCClipMask | GCForeground);
      XChangeGC(XDISPLAY, XGC, valuemask, &values);

      XCopyArea (XDISPLAY, xPixmap, XWINDOW, XGC,
		 rectangle.x, rectangle.y, rectangle.width, rectangle.height,
		 rectangle.x, rectangle.y);
    }
  else
    {
      NSRect	rect;

      // no backing store, so keep a list of exposed rects to be
      // processed in the _processExposedRectangles method
      // Add the rectangle to the region used in -_processExposedRectangles
      // to set the clipping path.
      XUnionRectWithRegion (&rectangle, region, region);

      // Transform the rectangle's coordinates to PS coordinates and add
      // this new rectangle to the list of exposed rectangles.
      rect.origin = NSMakePoint((float)rectangle.x, rectangle.y);
      rect.size = NSMakeSize(rectangle.width, rectangle.height);
      [exposedRects addObject: [NSValue valueWithRect: rect]];
    }
}

// handle X expose events
- (void)_processExposedRectangles
{
  if (backing_type != NSBackingStoreNonretained)
    return;

  // Set the clipping path to the exposed rectangles
  // so that further drawing will not affect the non-exposed region
  XSetRegion (XDISPLAY, XGC, region);

  // We should determine the views that need to be redisplayed. Until we
  // fully support scalation and rotation of views redisplay everything.
  [self display];

  // Restore the exposed rectangles and the region
  [exposedRects removeAllObjects];
  XDestroyRegion (region);
  region = XCreateRegion();
  XSetClipMask (XDISPLAY, XGC, None);
}

- (NSRect) xFrame
{
  return NSMakeRect (X, Y, WIDTH,HEIGHT);
}

- (Window) xWindow
{
  return XWINDOW;
}

- (GC) xGC
{
  return XGC;
}

- (void) _setVisible: (BOOL)flag
{
  visible = flag;
}

- (Class) classForCoder: (NSCoder*)aCoder
{
  if ([self class] == [XGWindow class])
    return [super class];
  return [self class];
}

- (void) setMaxSize: (NSSize)aSize
{
  [super setMaxSize: aSize];
  SIZ_HINTS.flags |= PMaxSize;
  SIZ_HINTS.max_height = maximum_size.height;
  SIZ_HINTS.max_width = maximum_size.width;
  XSetNormalHints(XDISPLAY, XWINDOW, &SIZ_HINTS);
}

- (void) setMinSize: (NSSize)aSize
{
  [super setMinSize: aSize];
  SIZ_HINTS.flags |= PMinSize;
  SIZ_HINTS.min_height = minimum_size.height;
  SIZ_HINTS.min_width = minimum_size.width;
  XSetNormalHints(XDISPLAY, XWINDOW, &SIZ_HINTS);
}

- (void) setResizeIncrements: (NSSize)aSize
{
  [super setResizeIncrements: aSize];
  SIZ_HINTS.flags |= PResizeInc;
  SIZ_HINTS.height_inc = increments.height;
  SIZ_HINTS.width_inc = increments.width;
  XSetNormalHints(XDISPLAY, XWINDOW, &SIZ_HINTS);
}

@end


