/* 
   XRWindow.m

   NSWindow for GNUstep GUI X/RAW Backend.

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: February 1996
   Author:  Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: May 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <stdlib.h>

#define BOOL XWINDOWSBOOL							// prevent X windows BOOL
#include <X11/Xmd.h>								// warning
#undef BOOL

#include <Foundation/NSArray.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSMapTable.h>

#include <AppKit/NSCursor.h>
#include <AppKit/GSTrackingRect.h>
#include <AppKit/PSMatrix.h>

#include <gnustep/xraw/XR.h>


#ifndef LIB_FOUNDATION_LIBRARY
static void __NSRetainNothing(void *table, const void *anObject) 	{}
static void __NSReleaseNothing(void *table, void *anObject)			{}

NSMutableArray  *_XRGlobalWindowList = nil;

static NSString* __NSDescribeObjects(void *table, const void *anObject)
{
    return [(NSObject*)anObject description];
}

static const NSMapTableValueCallBacks NSNonRetainedObjectMapValueCallBacks = {
				(void (*)(NSMapTable *, const void *))__NSRetainNothing,
				(void (*)(NSMapTable *, void *))__NSReleaseNothing,
				(NSString *(*)(NSMapTable *, const void *))__NSDescribeObjects
}; 
#endif /* LIB_FOUNDATION_LIBRARY */


#define XWINDOW (((XRWindow_struct *)be_wind_reserved)->xWindow)
#define XGC (((XRWindow_struct *)be_wind_reserved)->context)
#define XEXPOSE (((XRWindow_struct *)be_wind_reserved)->is_exposed)
#define exposedRects (((XRWindow_struct *)be_wind_reserved)->exposedRects)
#define xPixmap (((XRWindow_struct *)be_wind_reserved)->xPixmap)
#define X (((XRWindow_struct *)be_wind_reserved)->x)
#define Y (((XRWindow_struct *)be_wind_reserved)->y)
#define WIDTH (((XRWindow_struct *)be_wind_reserved)->width)
#define HEIGHT (((XRWindow_struct *)be_wind_reserved)->height)
#define DEPTH (((XRWindow_struct *)be_wind_reserved)->depth)
#define region (((XRWindow_struct *)be_wind_reserved)->region)
#define XDISPLAY (((XRWindow_struct *)be_wind_reserved)->xDisplay)


extern void XRWindowGeometryChanged(XRWindow *window, Drawable x_drawable);
void _setOriginHint(Display *anXDisplay, Window anXWindow, NSPoint origin);

//
// WindowMaker window manager interaction
//
typedef struct {
    CARD32 flags;
    CARD32 window_style;
    CARD32 window_level;
    CARD32 reserved;
    Pixmap miniaturize_pixmap;			// pixmap for miniaturize button 
    Pixmap close_pixmap;				// pixmap for close button 
    Pixmap miniaturize_mask;			// miniaturize pixmap mask 
    Pixmap close_mask;					// close pixmap mask 
    CARD32 extra_flags;
} GNUstepWMAttributes;

#define GSWindowStyleAttr 		(1<<0)
#define GSWindowLevelAttr 		(1<<1)
#define GSMiniaturizePixmapAttr	(1<<3)
#define GSClosePixmapAttr		(1<<4)
#define GSMiniaturizeMaskAttr	(1<<5)
#define GSCloseMaskAttr			(1<<6)
#define GSExtraFlagsAttr       	(1<<7)



@implementation XRWindow

//
// Class variables
//
static NSMapTable* windowsToXWindows = NULL;
static NSMapTable* windowsToTags = NULL;

//
// Class Initialization
//
+ (void)initialize
{
	if (self == [XRWindow class])
		{
        _XRGlobalWindowList = [[NSMutableArray alloc] initWithCapacity: 20];
		windowsToXWindows = NSCreateMapTable(NSNonOwnedPointerMapKeyCallBacks,
										NSNonRetainedObjectMapValueCallBacks,
										20);
		windowsToTags = NSCreateMapTable(NSIntMapKeyCallBacks,
										NSNonRetainedObjectMapValueCallBacks,
										20);
		}
}

+ (XRWindow*)_windowForXWindow:(Window)xWindow
{
	return NSMapGet (windowsToXWindows, (void*)xWindow);
}

+ (NSWindow*)_windowWithTag:(int)windowNumber
{
	return NSMapGet (windowsToTags, (void*)windowNumber);
}

+ (NSView *)_windowViewWithFrame:(NSRect)frameRect			// create the top
{															// most view in the
	return [[XRWindowView alloc] initWithFrame:frameRect];	// view's heirarchy
}

//
// Instance methods
//
- (id)initWithContentRect:(NSRect)contentRect
				styleMask:(unsigned int)aStyle
				backing:(NSBackingStoreType)bufferingType
				defer:(BOOL)flag
				screen:aScreen;
{											
int xScreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
XRContext* context = (XRContext *)[XRContext currentContext];
NSPoint origin;
NSSize size;
Window xAppRootWindow = [context xAppRootWindow];
XSetWindowAttributes winattrs;
unsigned long valuemask2;
XWMHints *wm_hints;
XGCValues values;
unsigned long valuemask = (GCForeground | GCBackground);
XClassHint chint; 
Atom win_decor_atom;
							 
GNUstepWMAttributes winAttrs;
															// Allocate the 
	be_wind_reserved = calloc(1, sizeof(XRWindow_struct));	// back-end struct

	XDISPLAY = [context xDisplay];
	exposedRects = [NSMutableArray new];
	region = XCreateRegion();
	xPixmap = None;
											// Provide some reasonable values 
	if (!contentRect.size.width)			// for the contentRect size so the 
		contentRect.size.width = 200;		// X Window server doesn't complain
	if (!contentRect.size.height)
		contentRect.size.height = 100;

    if ([_XRGlobalWindowList indexOfObjectIdenticalTo: self] == NSNotFound)
        [_XRGlobalWindowList addObject: self];
	[super initWithContentRect:contentRect 				// call super method
		   styleMask:aStyle
		   backing:bufferingType 
		   defer:flag 
		   screen:aScreen];

	NSDebugLog(@"XRWindow designated initializer\n");
														// get a unique integer
	window_num = XRUniqueWindowTag();		 			// identifier for self
									// translate NSWindow's coords to X coords
	origin = [context XPointFromUserPoint:NSMakePoint(contentRect.origin.x,
							contentRect.origin.y + contentRect.size.height)];
	size = [context XSizeFromUserSize:contentRect.size];
	origin.y = DisplayHeight(XDISPLAY, xScreen) - origin.y;

	if (!(wm_hints = XAllocWMHints())) 
		{
		NSLog(@"XRWindow: Failure allocating memory with XAllocWMHints()\n");
		exit(0);
		}
														// set default colors
	valuemask2 = (CWBackPixel|CWBorderPixel);					 
	winattrs.border_pixel = [(XRColor *)[NSColor blackColor] xColor].pixel;		
	winattrs.background_pixel = [(XRColor *)background_color xColor].pixel;
														// create an X window
	XWINDOW = XCreateWindow(XDISPLAY, RootWindow(XDISPLAY, xScreen),
							(int)origin.x, (int)origin.y,
							(int)size.width, (int)size.height,
							0,
							CopyFromParent,
							CopyFromParent,
							CopyFromParent,
							valuemask2,
							&winattrs);
														// Create an X GC for 
	values.foreground = winattrs.background_pixel;		// the content view  
	values.background = winattrs.background_pixel;		// set it's colors per
	values.function = GXcopy;							// chosen NSColor
	valuemask = (GCForeground | GCBackground | GCFunction);
	XGC = XCreateGC(XDISPLAY, RootWindow(XDISPLAY,xScreen), valuemask,&values);

														// Set the X event mask 
	XSelectInput(XDISPLAY, XWINDOW, ExposureMask | KeyPressMask | 
									KeyReleaseMask | ButtonPressMask | 
									ButtonReleaseMask | ButtonMotionMask | 
									StructureNotifyMask | PointerMotionMask | 
									EnterWindowMask | LeaveWindowMask | 
									FocusChangeMask | PropertyChangeMask | 
									ColormapChangeMask | KeymapStateMask | 
									VisibilityChangeMask);

	wm_hints->window_group = xAppRootWindow;
	wm_hints->initial_state = NormalState;				// Give the window
	wm_hints->input = True;								// manager some hints
	wm_hints->flags = StateHint | InputHint | WindowGroupHint;
	XSetWMHints(XDISPLAY, XWINDOW, wm_hints);
										// WindowMaker ignores the frame origin
										// unless it's also specified as a hint
	_setOriginHint(XDISPLAY, XWINDOW, origin);		
													// set class hints so WM
	chint.res_class = "XRWindow";					// creates an app icon
	chint.res_name = (char*)[[[NSProcessInfo processInfo] processName]cString];
	XSetClassHint(XDISPLAY, xAppRootWindow, &chint);

	[self _updateWindowGeometry];
														// push the arrow as
	[[NSCursor arrowCursor] push];						// the default cursor 
														
	if(aStyle & NSClosableWindowMask)					// if window has close 
		{												// button inform WM 
		Atom protocols_atom;
							// that we can handle
		Atom delete_win_atom;
							// WM_DELETE_WINDOW

		protocols_atom = XInternAtom(XDISPLAY, "WM_PROTOCOLS", False);
		delete_win_atom = XInternAtom(XDISPLAY, "WM_DELETE_WINDOW", False);

		if((delete_win_atom != (Atom)None) && (protocols_atom != (Atom)None))
			XSetWMProtocols(XDISPLAY, XWINDOW, &delete_win_atom, 1);
		}

	winAttrs.flags = GSWindowStyleAttr;
	winAttrs.window_style = aStyle;
	win_decor_atom = XInternAtom(XDISPLAY,"_GNUSTEP_WM_ATTR", False);
														// send WindowMaker WM
	if((win_decor_atom != (Atom)None))					// window style hints
		XChangeProperty(XDISPLAY, XWINDOW, win_decor_atom, win_decor_atom, 
						32, PropModeReplace, (unsigned char *)&winAttrs, 
						sizeof(GNUstepWMAttributes)/sizeof(CARD32));
															// Insert window 
	NSMapInsert (windowsToXWindows, (void*)XWINDOW, self);	// into the mapping
	NSMapInsert (windowsToTags, (void*)window_num, self);	// from X windows
															// to NS windows
	NSDebugLog(@"XRWindow end of designated initializer\n");

	return self;
}

- (void) close
{
  [super close];
  if (is_released_when_closed)
    {
      NSApplication *theApp = [NSApplication sharedApplication];

      [_XRGlobalWindowList removeObjectIdenticalTo: self];
      /*
       * Horrible kludge to handle case where an application has no menu - we
       * assume that this is the only window in the application and terminate.
       * We only do this for windows that release when closed - any other
       * window may be intended to re-open.
       */
      if ([theApp mainMenu] == nil)
        [theApp terminate: self];
    }
}

- (void)dealloc
{														// Remove from X to NS  
	NSMapRemove (windowsToXWindows, (void*)XWINDOW);	// windows mapping
	NSMapRemove (windowsToTags, (void*)[self windowNumber]);
  
	if (XWINDOW)									
		{
		XDestroyWindow(XDISPLAY, XWINDOW);				// Destroy the X Window
		XWINDOW = 0;
		XFreeGC (XDISPLAY, XGC);						// Free X window's GC
		}

	[exposedRects release];

	if (xPixmap) 
		{
		XFreePixmap (XDISPLAY, xPixmap);
		xPixmap = None;
		}

	XDestroyRegion (region);

	free(be_wind_reserved);								// Release the back-end
														// structure
	[super dealloc];
}

- (void)setTitle:(NSString*)aString
{
XTextProperty windowName;

	[super setTitle:aString];

	if (window_title && XWINDOW)
		{
		const char* newTitle = [window_title cString];

		XStringListToTextProperty((char**)&newTitle, 1, &windowName);
		XSetWMName(XDISPLAY, XWINDOW, &windowName);
		XSetWMIconName(XDISPLAY, XWINDOW, &windowName);
		}
}

- (void)orderOut:sender
{
	if (XEXPOSE && XWINDOW)
		XUnmapWindow([XRContext currentXDisplay], XWINDOW);

	XEXPOSE = NO;										// window not visible
	visible = NO;
	if (is_key)
		[self resignKeyWindow];
	if (is_main)
		[self resignMainWindow];
}

- (void)orderFront:sender
{														// if X window exists
	if (XWINDOW && !XEXPOSE)							// but is not exposed
		{												// map it raised
		XMapRaised([XRContext currentXDisplay],XWINDOW);
		XEXPOSE = YES;
		}												// else just raise the
	else												// window's tier
		XRaiseWindow([XRContext currentXDisplay], XWINDOW);				

	visible = YES;
    if ([_XRGlobalWindowList indexOfObjectIdenticalTo: self] == NSNotFound)
        [_XRGlobalWindowList addObject: self];
	[self update];
}

- (void)setContentViewSize:(NSSize)aSize
{
NSRect r;

	r.origin = NSZeroPoint;
	r.size = aSize;
	if (content_view)
		[content_view setFrame: r];
}

- (void)setContentSize:(NSSize)aSize
{
XRContext* context = [XRContext currentContext];
NSSize size;

	size = [context XSizeFromUserSize:aSize];
	NSDebugLog(@"content user size (%f, %f) device (%f, %f)",
				aSize.width, aSize.height, size.width, size.height);

	if (XWINDOW)										// Resize the window
		XResizeWindow(XDISPLAY, XWINDOW, (int)size.width, (int)size.height);

	[self _updateWindowGeometry];

	if (visible)
		[self display];
}

- (void)setFrame:(NSRect)rect display:(BOOL)flag
{
XRContext* context = [XRContext currentContext];
int xScreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
NSPoint origin;
NSSize size;
 
	origin = [context XPointFromUserPoint: NSMakePoint(rect.origin.x,
											rect.origin.y + rect.size.height)];
	origin.y = DisplayHeight(XDISPLAY, xScreen) - origin.y;
	size = [context XSizeFromUserSize:rect.size];
											// Reposition and resize the window  
	XMoveResizeWindow (XDISPLAY, XWINDOW, (int)origin.x, (int)origin.y,
						(int)size.width, (int)size.height);
	_setOriginHint(XDISPLAY, XWINDOW, origin);

	[self _updateWindowGeometry];

	[super setFrame:rect display:flag];
}

- (void)setFrameOrigin:(NSPoint)aPoint						// do nothing if
{															// frame unchanged
	if(aPoint.x == frame.origin.x && aPoint.y == frame.origin.y)	
		return;	
	else
		{													
		XRContext* context = [XRContext currentContext];
		int xScreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
		NSPoint origin;											

		[super setFrameOrigin:aPoint];

		origin = [context XPointFromUserPoint: NSMakePoint(frame.origin.x,
					  					frame.origin.y + frame.size.height)];
		origin.y = DisplayHeight(XDISPLAY, xScreen) - origin.y;	// translate to
																// screen coord
		NSDebugLog (@"display a window at (%f, %f) (%f, %f)", 
					origin.x, origin.y,
	      			frame.size.width, frame.size.height);

		XMoveWindow (XDISPLAY, XWINDOW, (int)origin.x, (int)origin.y);
		_setOriginHint(XDISPLAY, XWINDOW, origin);
		[self _updateWindowGeometry];
		}
}

- (void) flushWindow
{
  int x, y, width, height;
  XGCValues values;
  unsigned long valuemask;
										  // do nothing if backing is not buffered
  if (backing_type == NSBackingStoreNonretained)
    return;

  if (disable_flush_window) 		// if flushWindow is called
    {					// while flush is disabled
      needs_flush = YES;		// mark self as needing a
      return;				// flush, then return
    }

  values.function = GXcopy;							
  values.plane_mask = AllPlanes;							
  values.clip_mask = None;							
  valuemask = (GCFunction | GCPlaneMask | GCClipMask);
  XChangeGC(XDISPLAY, XGC, valuemask, &values);

  x = rectNeedingFlush.origin.x;		// width/height seems
  y = rectNeedingFlush.origin.y;		// to require +1 pixel
  width = rectNeedingFlush.size.width + 1;	// to copy out  
  height = rectNeedingFlush.size.height + 1;

  NSDebugLog (@"copy X rect ((%d, %d), (%d, %d))", x, y, width, height);
//  fprintf(stderr, " flushWindow ");

  if (width > 0 || height > 0)
    XCopyArea (XDISPLAY, xPixmap, XWINDOW, XGC, x, y, width, height, x, y);

  needs_flush = NO;
  rectNeedingFlush = NSZeroRect;
}

- (NSPoint)mouseLocationOutsideOfEventStream		// Return mouse location in
{													// reciever's base coords
NSPoint mouseLocation;				  				// ignoring the event loop  
XRContext* context = [XRContext currentContext];	
Window xDisplayRootWindow = [(XRContext *)context xDisplayRootWindow];
NSSize screenSize = [(XRScreen *)[NSScreen mainScreen] frame].size;
Window rootWin, childWin;
int currentX, currentY, winX, winY;
unsigned mask;
BOOL success;

	success = XQueryPointer (XDISPLAY, xDisplayRootWindow, &rootWin, &childWin,
							 &currentX, &currentY, &winX, &winY, &mask);
	if(!success)
		return NSZeroPoint;							
													// convert to NSCoord sys
	mouseLocation = NSMakePoint(currentX, screenSize.height - currentY);	

	return [self convertScreenToBase:mouseLocation];							
}													
//*****************************************************************************
//
// 		Non OPENSTEP spec Instance methods 
//
//*****************************************************************************

- (void)_captureMouse:sender
{
int ret = XGrabPointer([XRContext currentXDisplay], [self xWindow], False,
					   PointerMotionMask | ButtonReleaseMask | ButtonPressMask,
					   GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

	if (ret != GrabSuccess)
		NSLog(@"Failed to grab pointer\n");
	else
		{
		NSApplication *theApp = [NSApplication sharedApplication];
		NSDebugLog(@"Grabbed pointer\n");
		[(XRApplication *)theApp setGrabXWindow: [self xWindow]];
		}
}

- (void)_releaseMouse:sender
{
	XUngrabPointer([XRContext currentXDisplay], CurrentTime);
	[(XRApplication *)[NSApplication sharedApplication] setGrabXWindow: 0];
}

void _setOriginHint(Display *anXDisplay, Window anXWindow, NSPoint origin)
{
XSizeHints size_hints;

	size_hints.x = (int)origin.x;					// WindowMaker ignores the
	size_hints.y = (int)origin.y;					// frame origin unless it's
	size_hints.flags = PPosition | USPosition;		// also specified as a hint
	XSetNormalHints(anXDisplay, anXWindow, &size_hints);
}

- (void)_updateWindowGeometry
{
XRContext* context = [XRContext currentContext];
int xScreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
unsigned int borderWidth;
Window rootWindow;
int oldWidth = WIDTH;
int oldHeight = HEIGHT;
int x, y;

	if (XWINDOW) 
		{							// XGetGeometry returns coordinates
		XFlush (XDISPLAY);			// relative to it's parent which is the  	
									// window manager's title bar if it has one
		if (!XGetGeometry (XDISPLAY, XWINDOW, &rootWindow, &x, &y, &WIDTH, 
							&HEIGHT, &borderWidth, &DEPTH))
			return;					// set the NSWindow frame per the actual X 
									// Window frame.  if the window either has
									// no title or is a menu just use the value
									// returned by XGetGeometry().
		if(!(style_mask & NSTitledWindowMask))
			{		
			frame.origin = [context userPointFromXPoint: NSMakePoint (x, 
							DisplayHeight(XDISPLAY, xScreen) - (y + HEIGHT))];
			X = x;
			Y = y;
			}
		else						// window has a title bar so have X server
			{						// translate to display coordinates first
			XTranslateCoordinates(XDISPLAY, XWINDOW, 
									RootWindow(XDISPLAY, xScreen), 
									x, y, &X, &Y, &rootWindow);
			frame.origin = [context userPointFromXPoint: NSMakePoint ((float)X, 
							(float)y +	DisplayHeight(XDISPLAY, xScreen) - 
							(float)(Y + HEIGHT))];
			}

		frame.size = [context userSizeFromXSize:NSMakeSize (WIDTH, HEIGHT)];
		NSDebugLog (@"window geometry user ((%f, %f), (%f, %f)), "
			  		@"device ((%d, %d), (%d, %d))",
			  		frame.origin.x, frame.origin.y, 
					frame.size.width, frame.size.height,
			 		X, Y, WIDTH, HEIGHT);

		if (oldWidth != WIDTH || oldHeight != HEIGHT) 
			{											// Inform our window 
			if (content_view) 							// and content views		
				{										// of the frame change
				NSView *wv = [content_view superview];			
				NSRect rect = [self frame];	
	
				rect.origin = NSZeroPoint;
				rect.size = frame.size;
				[wv setFrame: rect];
						// xxx Actually the content view should be resized
						// by the window view
				[content_view setFrame:rect];
				[self _createBackingStore];
														// window view needs to
				[wv setNeedsDisplay:YES];				// be displayed
				}
			XRWindowGeometryChanged(self, [self xDrawable]);	// inform draw
			}													// engine of 
		}														// geometry chg
}

- (void)xSetFrameFromXFrame:(NSRect)rect
{
BOOL onlyMoved = NO;								// Determine if the window
													// was only moved 
	if (rect.size.width == WIDTH && rect.size.height == HEIGHT && 
			(rect.origin.x != X || rect.origin.y != Y))
		onlyMoved = YES;

	[self _updateWindowGeometry];

	if (onlyMoved)
		return;

//  PSWinitcontext (gc, drawable, 0, HEIGHT);

    /*
     * if no backing  store we need to redisplay window
     */
	if (backing_type != NSBackingStoreNonretained)
            [[content_view superview] setNeedsDisplay: YES];
}

- (Drawable)xDrawable										// return something
{															// to draw on
	if (backing_type != NSBackingStoreNonretained)	
		return xPixmap;										// backing store
	else													
		return XWINDOW;										// X window 
}															

- (void)_createBackingStore
{
int xScreen;

	if (backing_type == NSBackingStoreNonretained)
		return;

	NSDebugLog (@"_createBackingStore");

	if (xPixmap)
		XFreePixmap (XDISPLAY, xPixmap);

	NSDebugLog (@"create backing store: width = %d, height = %d, depth = %d",
					WIDTH, HEIGHT, DEPTH);

	xScreen = [(XRScreen *)[NSScreen mainScreen] xScreen];
	xPixmap = XCreatePixmap (XDISPLAY, RootWindow(XDISPLAY, xScreen),
								WIDTH, HEIGHT, DEPTH);
	if (!xPixmap) 		
		{
		NSLog (@"cannot create the pixmap for backing store!");
		return;
		}												// Clear the pixmap to 
														// avoid artifacts 
	XFillRectangle (XDISPLAY, xPixmap, XGC, 0, 0, WIDTH, HEIGHT);
}

- (void)_addExposedRectangle:(XRectangle)rectangle		// process expose event 	
{				
	XEXPOSE = YES;									// is this needed FIX ME?

	if (backing_type != NSBackingStoreNonretained) 	// window has a backing
		{											// store so just copy the
		XGCValues values;							// exposed rect from the
		unsigned long valuemask;					// pixmap to the X window

		NSDebugLog (@"copy exposed area ((%d, %d), (%d, %d))",
				rectangle.x, rectangle.y, rectangle.width, rectangle.height);

		XRSyncFocusWindowColor(background_color, self);

		values.function = GXcopy;							
		values.plane_mask = AllPlanes;							
		values.clip_mask = None;							
		values.foreground = [(XRColor*)background_color xColor].pixel;		  
		valuemask = (GCFunction | GCPlaneMask | GCClipMask | GCForeground);
		XChangeGC(XDISPLAY, XGC, valuemask, &values);
	
		XCopyArea (XDISPLAY, xPixmap, XWINDOW, XGC,
				   rectangle.x, rectangle.y, rectangle.width, rectangle.height,
				   rectangle.x, rectangle.y);
		}
	else 			// no backing store, so keep a list of exposed rects to be
		{			// processed in the _processExposedRectangles method
		XRContext* context = [XRContext currentContext];
		NSRect rect;			// Add this rectangle to the region. The region 
								// is used in -_processExposedRectangles to set 
								// the clipping path.
		XUnionRectWithRegion (&rectangle, region, region);
	
								// Transform the rectangle's coordinates to PS 
								// coordinates and add this new rectangle to 
								// the list of exposed rectangles.
		rect.origin = [context userPointFromXPoint:NSMakePoint(rectangle.x,
									rectangle.y)];
		rect.size = [context userSizeFromXSize:NSMakeSize(rectangle.width,
									rectangle.height)];
		[exposedRects addObject:[NSValue valueWithRect:rect]];
		}
}

- (void)_processExposedRectangles					// handle X expose events
{					
	if (backing_type != NSBackingStoreNonretained)			// do nothing if
		return;												// view is buffered

//  fprintf(stderr, " _processExposedRectangles ");

							// Set the clipping path to the exposed rectangles 
							// so that further drawing will not affect the 
							// non-exposed region 
	XSetRegion (XDISPLAY, XGC, region);

							// We should determine the views that need to be 
							// redisplayed. Until we fully support scalation 
	[self display];			// and rotation of views redisplay everything.

							// Restore the exposed rectangles and the region
	[exposedRects removeAllObjects];
	XDestroyRegion (region);
	region = XCreateRegion();
	XSetClipMask (XDISPLAY, XGC, None);
}

- (NSRect)xFrame					{ return NSMakeRect (X, Y, WIDTH,HEIGHT); }
- (Pixmap)xBackingPixmap			{ return xPixmap; }
- (unsigned int)xHeight				{ return HEIGHT; }
- (Window)xWindow					{ return XWINDOW; }
- (GC)xGC							{ return XGC; }
- (void)xSetUnmapped:(BOOL)flag		{ visible = flag; }		// Unmapped X win
									 						// are not visible
- (void)_windowNeedsFlushInRect:(NSRect)rect		// add rect to area needing 
{													// flush.  X window device
	needs_flush = YES;								// coords are assumed
	rectNeedingFlush = NSUnionRect(rectNeedingFlush, rect);
}

- (Class) classForCoder: aCoder
{
  if ([self class] == [XRWindow class])
    return [super class];
  return [self class];
}

- (void) _windowNeedsFlushInXRect: (XRectangle)xRect      // add rect to area needing
{                       // flush.  X window device
  needs_flush = YES;      // coords are assumed
  rectNeedingFlush = NSUnionRect(rectNeedingFlush,
				 NSMakeRect(xRect.x,     xRect.y,
					    xRect.width, xRect.height));
}

@end
