
#include<iostream.h>
#include<mMatrix.h>
#include"ode.h"

template<class T,class Real>
ostream&
operator<<(ostream &os,OIVP<T,Real>& a)
{
  os << "dimension: " << a.dim << "\n";
  os << "position: " << a.t << "\n";
  os << "step: " << a.step << "\n";
  os << "tolerance: " << a.tol << "\n";
  os << "start: " << a.start << "\n";
  os << "stop: " << a.stop << "\n";
  os << "x: " << a.x << "\n";
  os << "f: " << a.f << "\n";
  os << "x0: " << a.x0 << "\n";
  os << "error: " << a.error << "\n\n";
  return os;
}

template<class T,class Real>
OIVP<T,Real>::OIVP(const OIVP<T,Real> &a) : x(a.x.size()), f(a.f.size()),
x0(a.x0.size()), error(a.error.size()), scale(a.scale.size()),
plot(a.plot.size()), midpoint(a.midpoint.size()),
midval(a.midval.rsize(),a.midval.csize())
{
  x=a.x;
  f=a.f;
  x0=a.x0;
  error=a.error;
  scale=a.scale;
  plot=a.plot;
  midpoint=a.midpoint;
  midval=a.midval;
}

template<class T,class Real>
void
OIVP<T,Real>::integrate()
{
  precondition();                    // evaluate any preconditions
  t=start;                           // starting value
  evaluate(start,x0);                // compute initial derivatives
  int count=0;
  Real tplot=start;
  Real plotstep=(stop-start)/(plotpoint-1.0);
  if(plotpoint) {
    if(plotpoint!=plot.csize()) plot.resize(dim,plotpoint);
    if(plotstep!=0.0) assign(x,plot.col(++count));    // initial values
    tplot+=plotstep;                   // next plotting point
  }
  while(t!=stop) {
    takestep();
    changestep();
//cerr << t << "\t" << step << "\n";
    if(plotpoint) {
      while(tplot<=t) {
        assign(x-f*(t-tplot),plot.col(++count));  // interpolate plotted value
        tplot+=plotstep;
      }
    }
    if(t+step>stop) step=stop-t;         // don't overshoot
  }
  if(plotpoint) assign(x,plot.col(++count));    // plot final values
  postcondition();                    // evaluate any postconditions
}


template<class T,class Real>
RungeKutta<T,Real>::RungeKutta(const RungeKutta<T,Real> &a) : IOVP<T,Real>(a), 
a(a.a.size()), c(a.c.size()), cs(a.cs.size()), tmp(a.tmp.size()), 
b(a.b.rsize(),a.b.csize())
{
  a=a.a;
  c=a.c;
  cs=a.cs;
  tmp=a.tmp;
  b=a.b;
}

template<class T,class Real>
RungeKutta<T,Real>::RungeKutta(int d)
{
// initializes all data structures.
  dim=d;
  a.reserve(6);
  c.reserve(6);
  cs.reserve(6);
  b.reserve(6,5);
  a(1)=0.0;
  a(2)=1.0/5.0;
  a(3)=3.0/10.0;
  a(4)=3.0/5.0;
  a(5)=1.0;
  a(6)=7.0/8.0;
  c(1)=37.0/378.0;
  c(2)=0.0;
  c(3)=250.0/621.0;
  c(4)=125.0/594.0;
  c(5)=0.0;
  c(6)=512.0/1771.0;
  cs(1)=2825.0/27648.0;
  cs(2)=0.0;
  cs(3)=18575.0/48384.0;
  cs(4)=13525.0/55296.0;
  cs(5)=277.0/14336.0;
  cs(6)=1.0/4.0;
  b=(Real)0.0;
  b(2,1)=1.0/5.0;
  b(3,1)=3.0/40.0;
  b(3,2)=9.0/40.0;
  b(4,1)=3.0/10.0;
  b(4,2)=-9.0/10.0;
  b(4,3)=6.0/5.0;
  b(5,1)=-11.0/54.0;
  b(5,2)=5.0/2.0;
  b(5,3)=-70.0/27.0;
  b(5,4)=35.0/27.0;
  b(6,1)=1631.0/55296.0;
  b(6,2)=175.0/512.0;
  b(6,3)=575.0/13824.0;
  b(6,4)=44275.0/110592.0;
  b(6,5)=253.0/4096.0;
  midpoint.reserve(6);
  init();             // initialize everything depending on dim
}

template<class T,class Real>
void 
RungeKutta<T,Real>::init()
{
// initializes all data structures.  dim must be preset
  x.reserve(dim);
  f.reserve(dim);
  x0.reserve(dim);
  tmp.reserve(dim);
  error.reserve(dim);
  scale.reserve(dim);
  scale=(T)1.0;
  midval.reserve(dim,6);
}


template<class T,class Real>
void 
RungeKutta<T,Real>::takestep()
{
// takes a single RK step, without regard to success or failure
  x0=x;                 // move current point to previous
  error=(T)0.0;            // clear error estimate
  for(int j=1;j<=6;j++) {    // for each internal point
    midpoint(j)=t+a(j)*step;       // internal position
    tmp=x0;                         // initialize guess with previous value
    for(int k=1;k<=6;k++) tmp+=(T)b(j,k)*midval.col(k); // add shifts
//    midval.col(j)=step*evaluate(midpoint(j),tmp);    // next shift
assign((T)step*evaluate(midpoint(j),tmp),midval.col(j));
    x+=(T)c(j)*midval.col(j);        // update current result with midpoint info
    error+=(T)(c(j)-cs(j))*midval.col(j);    // update error
  }
  t+=step;                          // update external position
}


template<class T,class Real>
void 
RungeKutta<T,Real>::changestep()
{
// modifies the step size based on the error information generated in the 
// previous step.  If the previous step was unacceptable, it restores
// the original x and t (Note that x0 is no longer valid).
//  Real relerr=0.5*(inf(error/scale)+sup(error/scale))/tol; // mode 
//  Real relerr=0.5*(inf(error/scale)+sup(error/scale))/tol; // max 
  Real relerr=sqrt(norm(error/scale))/tol; // avg relative error
  if(relerr < 1.0) {     // the previous step succeeded, leave in updated state
    if(relerr > cutoff) step*=safety*pow(relerr,growexp);  // increase size
    else step*=growfact;                 // maximum growth
  } else {                               // it failed
    x=x0;                                // throw out computed values
    t-=step;
    step*=min(double(shrinkfact),safety*pow(relerr,shrinkexp));
  }
} 

