#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2009 Free Software Foundation
#
# FILE:
# GRServer.py
#
# DESCRIPTION:
# Class that defines the Server component of GNUe Reports.
#
# NOTES:
#

import sys
import time
import thread

from gnue.common.apps import GDebug, GServerApp, GConfig
from gnue.common.apps.i18n import utranslate as u_
from gnue.common.rpc import server
from gnue.reports.base.GRConfig import ConfigOptions
from gnue.reports.server.GRRpcWrapper import GRRpcHandler
from gnue.common.datasources.GLoginHandler import SilentLoginHandler
from gnue.reports import VERSION


class GRServer (GServerApp.GServerApp): 

  #
  # GBaseApp() overrides
  #
  VERSION = VERSION
  COMMAND = "gnue-reports-server"
  NAME = "GNUe Reports Server"
  USAGE = GServerApp.GServerApp.USAGE
  COMMAND_OPTIONS = [    
    [ 'rpctype',None,'rpc-type',1,'xmlrpc', 'type',
      u_('Set the GNURPC connection type. The currently supported values ')+
      u_("for <type> are 'xmlrpc','xmlrpc.py_xmlrpc', 'xmlrpc.pw_xmlrpc'")+
      u_("and 'pyro'. For more information on GNURPC connection types have ")+
      u_('a look at common/doc/RPC-abstraction. ')],
    [ 'rpcport',None,'rpc-port',1,8766, 'port',
      u_('Set the GNURPC port. For more information on ')+
      u_('GNURPC have a look at common/doc/RPC-abstraction. ')],
    [ 'username',"u",'username',1,None, 'user',
      u_('Set the username for the used database.')],
    [ 'password',"p",'password',1,None, 'password',
      u_('Set the password for the used database.')]]
  SUMMARY = \
     "GNUe Reports is the primary reporting agent of the GNU Enterprise system."

  
  #
  # Initialize the class
  #
  def __init__(self):
    GServerApp.GServerApp.__init__(self, application='reports',defaults=ConfigOptions)
    
    self.configurationManager.registerAlias('gConfigReports', 'reports')

  ## set a list of transports
  def setTransports(self,transports):
    
    self._transports=transports


  def phaseInit(self):
    
    rpctype = self.OPTIONS['rpctype']

    if rpctype in ('xmlrpc','xmlrpc.pw_xmlrpc','xmlrpc.py_xmlrpc'):
      port = self.OPTIONS["rpcport"]
      print "Exporting our services via %s (port %s) ..." % (rpctype,port)
      params = {'port': int (port)} #,
#                'allowed_hosts': gConfig ('allowed_hosts')}


      self.setTransports({rpctype: params})

    elif rpctype == "pyro":
      print "Exporting our services via Pyro (EXPERIMENTAL!) ..."
      self.setTransports ({'pyro':{}})

    elif rpctype == "sockets":
      # Sockets not working yet
      print "Exporting our services via sockets (EXPERIMENTAL!) ..."
      self.setTransports ({'sockets':{}})

    else:
      # wrong transport protocol set. exiting
      print "The protocol you've set is currently not supported."
      sys.exit (-1)

    loginhandler = SilentLoginHandler (_username = self.OPTIONS ['username'],
                                       _password = self.OPTIONS ['password'])
    self.connections.setLoginHandler (loginhandler)


  def run(self):
           
    # Create the various servers
    servers = server.bind (self._transports, self.requestServer())

    # be verbose
    print _("\n... GNUe Reports server up and running ...\n")

    # Daemonize (if appropriate)
    GServerApp.GServerApp.run(self)

    # Start the server for the different protocolls
    for adapter in servers.values():
      thread.start_new_thread(adapter.serve,())

    # wait for the servers shut down
    try:
      while 1==1:
        time.sleep(1999999)
    except KeyboardInterrupt:
      print "Reports server is shuting down....ok"


  def requestServer(self):
    return GRRpcHandler(self.connections)


if __name__ == '__main__':
  grserv=GRServer()
  grserv.phaseInit()
  grserv.run()
