# GNU Enterprise Forms - Curses UI Driver - MenuItem Widget
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: menuitem.py 9956 2009-10-11 18:54:57Z reinhard $

import curses

from gnue.forms.uidrivers.curses.widgets._base import UIHelper


# =============================================================================
# Wrap an UI layer around a wxMenu widget
# =============================================================================

class UIMenuItem(UIHelper):
    """
    Implements a menu item object.
    """

    # -------------------------------------------------------------------------
    # Create a menu item widget
    # -------------------------------------------------------------------------

    def _create_widget_(self, event, spacer):
        """
        Creates a new MenuItem widget.
        """

        self.hotkey = self._gfObject.hotkey
        self.check = (self._gfObject.action_off is not None)
        self.is_checked = False
        self.is_enabled = True

        if self._gfObject.label is not None:
            if self.hotkey is not None and self.hotkey in _HOTKEYS:
                item = _HOTKEYS[self.hotkey]
                self._uiForm.register_hotkey(self, item)
            else:
                self.hotkey = ''

        event.container.add_subitem(self)

        return None


    # -------------------------------------------------------------------------
    # Events
    # -------------------------------------------------------------------------

    def _event_fire(self):
        """
        Call the action bound to this menu item
        """

        if self.is_enabled and self._gfObject.label is not None:
            self._gfObject._event_fire()
            result = True
        else:
            result = False

        return result


    # -------------------------------------------------------------------------
    # Check/uncheck menu item
    # -------------------------------------------------------------------------

    def _ui_switch_on_(self):

        self.is_checked = True
        return

    # -------------------------------------------------------------------------

    def _ui_switch_off_(self):
        
        self.is_checked = False
        return

    # -------------------------------------------------------------------------
    # Enable/disable menu item
    # -------------------------------------------------------------------------

    def _ui_enable_(self):
        
        self.is_enabled = True
        self._uiForm.update_fkey_bar()
        return

    # -------------------------------------------------------------------------

    def _ui_disable_(self):

        self.is_enabled = False
        self._uiForm.update_fkey_bar()
        return


# =============================================================================
# Definition of the available hotkeys
# =============================================================================

_HOTKEYS = {
    "INS"  : (curses.KEY_IC,),
    "DEL"  : (curses.KEY_DC,),
    "HOME" : (curses.KEY_HOME, curses.KEY_FIND),
    "END"  : (curses.KEY_END,),
    "PGUP" : (curses.KEY_PPAGE,),
    "PGDN" : (curses.KEY_NPAGE,),
    "UP"   : (curses.KEY_UP,),
    "DOWN" : (curses.KEY_DOWN,),
    "LEFT" : (curses.KEY_LEFT,),
    "RIGHT": (curses.KEY_RIGHT,),
    "TAB"  : (9,),
    "ENTER": (10, curses.KEY_ENTER,),
    "BACK" : (curses.KEY_BACKSPACE,),

    "SHIFT+INS"  : (curses.KEY_SIC,),
    "SHIFT+DEL"  : (curses.KEY_SDC,),
    "SHIFT+HOME" : (curses.KEY_SHOME,),
    "SHIFT+END"  : (curses.KEY_SEND,),
    "SHIFT+LEFT" : (curses.KEY_SLEFT,),
    "SHIFT+RIGHT": (curses.KEY_SRIGHT,),
    "SHIFT+TAB"  : (curses.KEY_BTAB,),
}

# Add the function keys to the map.  We treat the modifiers SHIFT, CTRL and the
# combination of both to add a multiple of 12 to the function key name.  So F1
# is F1, SHIFT+F1 is F13, CTRL+F1 is F25 and SHIFT+CTRL-F1 is F37.

for i in range(1, 13):
    for mul,mod in enumerate(["", "SHIFT+", "CTRL+", "SHIFT+CTRL+"]):
        hot_name = "%sF%d" % (mod, i)
        key_name = "KEY_F%d" % (i + mul * 12)

        key = getattr(curses, key_name, None)
        if key is not None:
            _HOTKEYS[hot_name] = (key,)

# Add the Control-Characters for A - Z (CTRL+A, CTRL+B, ...)
for i in range(26):
    _HOTKEYS["CTRL+%c" % (65 + i)] = chr(i+1)


# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass': UIMenuItem,
  'provides' : 'GFMenuItem',
  'container': False
}
