#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2009 Free Software Foundation
#
# $Id: GCParser.py 9953 2009-10-11 18:50:17Z reinhard $

import re

from gnue.common.definitions import GParser, GObjects, GRootObj
from gnue.common.definitions.GParserHelpers import GContent
from gnue.common.formatting import GTypecast
from gnue.common.apps import errors
from gnue.appserver import repository


xmlElements   = None
_LENGTH_SCALE = re.compile ('^\w+\s*\(\s*(\d*)\s*[\.\,]{0,1}\s*(\d*)\s*\)\s*')

# =============================================================================
# Exceptions
# =============================================================================

class Error (errors.ApplicationError):
  pass

class GCDError (errors.UserError):
  pass

class NoModuleError (GCDError):
  def __init__ (self, classname):
    msg = u_("Class '%s' has no module") % classname
    GCDError.__init__ (self, msg)

class DuplicateDefinitionError (GCDError):
  def __init__ (self, name, asLength):
    if asLength:
      msg = u_("'%s' has a length in type and an extra length attribute") % name
    else:
      msg = u_("'%s' has a scale in type and an extra scale attribute") % name
    GCDError.__init__ (self, msg)

class ModuleMismatchError (GCDError):
  def __init__ (self, classname, module):
    msg = u_("The class '%(class)s' mismatches the given module attribute "
             "'%(module)s'") % {'class': classname, 'module': module}
    GCDError.__init__ (self, msg)

class InvalidCalculatedError (GCDError):
  def __init__ (self, classname, propName):
    msg = u_("Calculated property '%(class)s.%(property)s' has parameters") \
          % {'class': classname, 'property': propName}
    GCDError.__init__ (self, msg)

class MissingNameError (GCDError):
  pass

class ModuleNameError (GCDError):
  def __init__ (self, name):
    msg = u_("The module name '%s' contains underscores") % name
    GCDError.__init__ (self, msg)

class ReferenceNotAllowedError (GCDError):
  def __init__ (self, refname, typename, name):
    msg = u_("Reference '%(ref)s' not allowed for '%(type)s' instance "
             "%(name)s") % {'ref': refname, 'type': typename, 'name': name}
    GCDError.__init__ (self, msg)

# =============================================================================
# load an XML object tree from a given stream and return it's root object
# =============================================================================

def loadFile (stream, initialize = True):
  return GParser.loadXMLObject (stream, xmlSchemaHandler, 'GCModule',
      'module', initialize)


# =============================================================================
# Build a dictionary tree with all available XML elements
# =============================================================================

def getXMLelements ():
  global xmlElements

  if xmlElements is None:
    xmlElements = {
      'module': {
        'BaseClass' : GCModule,
        'Required'  : True,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'comment': {
            'Typecast': GTypecast.text}
        },
        'ParentTags': (None)
      },

      'class': {
        'BaseClass': GCClass,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'module': {
            'Typecast': GTypecast.name},
          'comment': {
            'Typecast': GTypecast.text},
          'filter': {
            'Typecast': GTypecast.name},
        },
        'ParentTags': ('module',)
      },

      'property': {
        'BaseClass': GCProperty,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'type': {
            'Required': True,
            'Typecast': GTypecast.text},
          'file': {
            'Typecast': GTypecast.name },
          'nullable': {
            'Default' : True,
            'Typecast': GTypecast.boolean},
          'comment': {
            'Typecast': GTypecast.text},
          'length': {
            'Typecast': GTypecast.integer},
          'scale': {
            'Typecast': GTypecast.integer},
          'language': {
            'Default' : 'python',
            'Typecast': GTypecast.name}
        },
        'MixedContent'  : True,
        'KeepWhitespace': True,
        'ParentTags'    : ('class',)
      },

      'procedure': {
        'BaseClass'     : GCProcedure,
        'Attributes'    : {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'type': {
            'Typecast': GTypecast.text},
          'file': {
            'Typecast': GTypecast.name },
          'nullable': {
            'Default' : True,
            'Typecast': GTypecast.boolean},
          'comment': {
            'Typecast': GTypecast.text},
          'length': {
            'Typecast': GTypecast.integer},
          'scale': {
            'Typecast': GTypecast.integer},
          'language': {
            'Default' : 'python',
            'Typecast': GTypecast.name}
        },
        'MixedContent'  : True,
        'KeepWhitespace': True,
        'ParentTags'    : ('class',)
      },

      'parameter': {
        'BaseClass': GCParameter,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'type': {
            'Required': True,
            'Typecast': GTypecast.text},
          'comment': {
            'Typecast': GTypecast.text},
          'length': {
            'Typecast': GTypecast.integer},
          'scale': {
            'Typecast': GTypecast.integer}
        },
        'ParentTags': ('procedure',)
      },

      'index': {
        'BaseClass': GCIndex,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'unique': {
            'Typecast': GTypecast.boolean,
            'Default': True},
          },
        'ParentTags': ('class',),
      },
      'indexfield': {
        'BaseClass': GCIndexField,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique': True,
            'Typecast': GTypecast.name},
        },
        'ParentTags': ('index',)
      },
    }

  return GParser.buildImportableTags ('module', xmlElements)


# =============================================================================
# Class called by the XML parser to process the XML file
# =============================================================================

class xmlSchemaHandler (GParser.xmlHandler):
  def __init__ (self):
    GParser.xmlHandler.__init__ (self)
    self.xmlElements = getXMLelements ()


# =============================================================================
# Base class for all GNUe Class Definition classes
# =============================================================================

class GCObject (GObjects.GObj):
  pass


# =============================================================================
# Base class for all objects with a type-attribute
# =============================================================================

class GCTypeDefinition (GCObject):
  def __init__ (self, parent = None, objType = None, refsAllowed = True):
    GCObject.__init__ (self, parent, type = objType)

    self.datatype         = None
    self.length           = None
    self.scale            = None
    self.isReference      = False
    self._refsAllowed     = refsAllowed

    self._inits.extend ([None, self._validate])


  # ---------------------------------------------------------------------------
  # If an instance has a type attribute, verify it
  # ---------------------------------------------------------------------------

  def _validate (self):
    if hasattr (self, 'type'):
      fieldType = self.type.lower ().strip ()
      tpMatch   = re.compile ('^(\w+)').match (fieldType)
      if tpMatch is None:
        raise repository.TypeNameError, (fieldType)

      typename = tpMatch.groups () [0]
      if not repository.BASE_TYPES.has_key (typename) and \
         len (typename.split ('_')) != 2:
        raise repository.TypeNameError, (fieldType)

      # try to extract length and scale from fieldType
      lsMatch = _LENGTH_SCALE.match (fieldType)
      if lsMatch is not None:
        (lstr, sstr) = lsMatch.groups ()

        if len (lstr):
          if self.length is not None and self.length:
            raise DuplicateDefinitionError, (self.name, True)
          self.length = int (lstr)

        if len (sstr):
          if self.scale is not None and self.scale:
            raise DuplicateDefinitionError, (self.name, False)
          self.scale = int (sstr)

      if repository.BASE_TYPES.has_key (typename):
        repository.verifyBasetype (typename, self.length, self.scale)
      else:
        if not self._refsAllowed:
          raise ReferenceNotAllowedError, (typename, self._type, self.name)

        self.isReference = True

        if self.length:
          raise repository.TypeFormatError, \
              u_("Reference type '%s' must not have a length") % typename
        if self.scale:
          raise repository.TypeFormatError, \
              u_("Reference type '%s' must not have a scale") % typename

        typename    = repository.REF_TYPE
        self.length = repository.REF_LENGTH

      self.datatype = typename


# =============================================================================
# The module object, root object of a GCD tree
# =============================================================================

class GCModule (GRootObj.GRootObj, GCObject):
  def __init__ (self, parent = None):
    GRootObj.GRootObj.__init__ (self, 'module', getXMLelements, self.__module__)
    GCObject.__init__ (self, parent, type = 'GCModule')
    self._inits.extend ([None, self._validate])

  # ---------------------------------------------------------------------------
  # Validate a module tag
  # ---------------------------------------------------------------------------

  def _validate (self):
    if not len (self.name):
      raise MissingNameError, u_("Module has no name")

    if '_' in self.name:
      raise ModuleNameError, self.name


# =============================================================================
# The class object
# =============================================================================

class GCClass (GCObject):
  def __init__ (self, parent):
    GCObject.__init__ (self, parent, type = 'GCClass')
    self.fullName = None
    self.action   = 'create'
    self._inits.extend ([None, self._validate, self._complete])

  # ---------------------------------------------------------------------------
  # Validate a class definition
  # ---------------------------------------------------------------------------

  def _validate (self):
    parent = self.getParent ()

    if not len (self.name):
      raise MissingNameError, u_("Class has no name")

    if not isinstance (parent, GCModule):
      raise NoModuleError, self.name

    nameParts = self.name.split ('_')
    if len (nameParts) > 1:
      if hasattr (self, 'module') and self.module != nameParts [0]:
        raise ModuleMismatchError, (self.name, self.module)

      (self.module, self.name) = nameParts [:2]

    if not hasattr (self, 'module'):
      self.module = parent.name

    # if this class is not defined by the top level module, this is just a
    # definition for a class extension
    if self.module != parent.name:
      self.action = 'extend'

    self.fullName = repository.createName (self.module, self.name)


  # ---------------------------------------------------------------------------
  # Complete a class definition if neccessary
  # ---------------------------------------------------------------------------

  def _complete (self):
    # check if there's a gnue_id in new classes
    if self.action == 'create':
      self.__addProperty ('id', 'Object ID', pType = 'string', length = 32)
      self.__addProperty ('createdate', 'Date instance was created',
          pType = 'datetime', nullable = True)
      self.__addProperty ('createuser', 'User who created the instance',
          pType = 'string', length = 8, nullable = True)
      self.__addProperty ('modifydate', 'Date instance was last modified',
          pType = 'datetime', nullable = True)
      self.__addProperty ('modifyuser', 'User who last modified the instance',
          pType = 'string', length = 8, nullable = True)

      if hasattr (self, 'filter'):
        (fModule, fName) = repository.splitName (self.filter)
        self.__addProperty (fName, module = fModule)


    # transform all 'calculated properties' into procedures
    for prop in self.findChildrenOfType ('GCProperty') [:]:
      if True in [isinstance (c, GContent) for c in prop._children]:
        if prop.findChildOfType ('GCParameter'):
          raise InvalidCalculatedError, (self.fullName, prop.fullName)

        proc = GCProcedure (self)
        proc.name      = 'get%s' % prop.name
        proc.type      = prop.datatype
        proc.length    = prop.length
        proc.scale     = prop.scale
        proc.nullable  = prop.nullable
        proc.language  = prop.language
        if hasattr (prop, 'comment'):
          proc.comment = prop.comment

        proc._children = prop._children

        for child in proc._children:
          child.setParent (proc)

        proc._validate ()

        self._children.remove (prop)

        # Release references of the no longer needed property instance (for gc)
        del prop._inits [:]
        prop.setParent (None)


  # ---------------------------------------------------------------------------
  # Add a property to the class
  # ---------------------------------------------------------------------------

  def __addProperty (self, name, comment = None, module = "gnue", pType = None,
      length = None, nullable = False):
    items = [c.fullName for c in self.findChildrenOfType ('GCProperty')]

    fullName = repository.createName (module, name)
    if fullName in items:
      return None

    prop = GCProperty (self)
    prop.name     = name
    prop.type     = pType or fullName
    if length is not None:
      prop.length = length
    if comment is not None:
      prop.comment  = comment
    prop.nullable = nullable
    prop._validate ()

    prop.module   = module
    prop.fullName = fullName

    return prop





# =============================================================================
# The property object
# =============================================================================

class GCProperty (GCTypeDefinition):
  def __init__ (self, parent):
    GCTypeDefinition.__init__ (self, parent, objType = 'GCProperty')
    self.module   = None
    self.fullName = None


  # ---------------------------------------------------------------------------
  # Validate a property definition
  # ---------------------------------------------------------------------------

  def _validate (self):
    GCTypeDefinition._validate (self)
    self.module   = self.findParentOfType ('GCModule').name
    self.fullName = repository.createName (self.module, self.name)


# =============================================================================
# The procedure object
# =============================================================================

class GCProcedure (GCTypeDefinition):
  def __init__ (self, parent):
    GCTypeDefinition.__init__ (self, parent, objType = 'GCProcedure')
    self.module   = None
    self.fullName = None


  # ---------------------------------------------------------------------------
  # Validate a procedure definition
  # ---------------------------------------------------------------------------

  def _validate (self):
    GCTypeDefinition._validate (self)
    self.module   = self.findParentOfType ('GCModule').name
    self.fullName = repository.createName (self.module, self.name)


# =============================================================================
# The parameter object
# =============================================================================

class GCParameter (GCTypeDefinition):
  def __init__ (self, parent):
    GCTypeDefinition.__init__ (self, parent, 'GCParameter', False)


# =============================================================================
# The Index object
# =============================================================================

class GCIndex (GCObject):
  def __init__ (self, parent):
    GCObject.__init__ (self, parent, type = 'GCIndex')
    self.fields     = []
    self.fullName   = None
    self.module     = None
    self.fieldNames = []
    self._inits.extend ([None, self._complete, self._getFieldNames])

  def _complete (self):
    self.module     = self.findParentOfType ('GCModule').name
    self.fullName   = "ix_%s" % repository.createName (self.module, self.name)
    self.fields     = self.findChildrenOfType ('GCIndexField')

  def _getFieldNames (self):
    self.fieldNames = [f.fullName for f in self.fields]

# =============================================================================
# The index field object
# =============================================================================

class GCIndexField (GCObject):
  def __init__ (self, parent):
    GCObject.__init__ (self, parent, type = 'GCIndexField')
    self._inits.extend ([None, self._complete])

  def _complete (self):
    if not '_' in self.name:
      self.module   = self.getParent ().module
      self.fullName = repository.createName (self.module, self.name)
    else:
      self.module   = repository.splitName (self.name) [0]
      self.fullName = self.name
