/* unixmail.c -- Functions for manipulating unix mail messages. */

/* Copyright (C) 1990 Free Software Foundation

   Authored by Brian Fox.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <stdio.h>
#include <pwd.h>
#include <sys/errno.h>
#include "general.h"
#include "unixmail.h"
#include "regex.h"

extern int errno;

#if !defined (SPOOLDIR)
#define SPOOLDIR "/usr/spool/mail"
#endif

extern struct passwd *getpwuid ();

/* Global input/output buffer. */
char io_buffer[1024];

#define DEFAULT_FORMAT_SPEC "%18From %11Date %47Subject"

/* Produce an array of displayable lines for the messages in FILENAME.

   FORMAT_SPEC specifies how the messages are to be formatted.  If
   FORMAT_SPEC is null, then a default format is used.

   If FILTER_REGEXP is non-null, it is a regular expression to match
   against the entire contents of the FILTER_FIELD.

   If ALL is non-zero, it says to include the headers of messages that
   have already been read, i.e., messages which have a Status field
   containing an "R".

   If an error occurred reading the mail file, return a NULL pointer.
   If there was no error, and no messages, return an empty array instead. */
char **
from_headers (filename, format_spec, filter_regexp, filter_field, all)
     char *filename, *format_spec, *filter_regexp, *filter_field;
     int all;
{
  FILE *stream;
  char **array = (char **)NULL;
  int array_size = 0, array_index = 0;
  int rmail = 0;
  char *current_line;
  
  stream = fopen (filename, "r");

  if (!stream)
    return ((char **)NULL);
  else
    {
      array = (char **)xmalloc ((array_size = 10) * sizeof (char *));
      array[0] = (char *)NULL;
    }

  if (!format_spec)
    format_spec = DEFAULT_FORMAT_SPEC;

  if (filter_regexp)
    re_comp (filter_regexp);

  while (fgets (io_buffer, sizeof (io_buffer), stream))
    {
      int found = 0;

      /* Find start of headers.  We can search RMAIL and Unix Mail
	 files. */
      if (!rmail && strncmp (io_buffer, "From ", 5) == 0)
	found = 1;
      else if (strcmp (io_buffer, "\037\f\n") == 0)
	{
	  rmail = 1;
	  fgets (io_buffer, sizeof (io_buffer), stream);

	  if (strcmp (io_buffer, "0, unseen,,\n") == 0)
	    fgets (io_buffer, sizeof (io_buffer), stream);

	  found = 1;
	}
	
      if (found)
	{
	  HEADER **headers, **snarf_headers ();
	  char *header_value ();

	  headers = snarf_headers (stream);

	  /* Normally we only show the headers of new messages.  However,
	     this can be overridden with ALL. */
	  if (!all)
	    {
	      char *status = header_value ("Status", headers);

	      /* If the message has already been read, then don't include
		 it in the count of new messages. */
	      if (rindex (status, 'R'))
		{
		  free_headers (headers);
		  continue;
		}
	    }

	  if (filter_regexp)
	    {
	      if (re_exec (header_value (filter_field, headers)) == 1)
		current_line = displayable_string (format_spec, headers);
	      else
		current_line = (char *)NULL;
	    }
	  else
	    current_line = displayable_string (format_spec, headers);

	  /* If this message is to be part of the list, then add it now. */
	  if (current_line)
	    {
	      if (array_index + 2 > array_size)
		array = (char **)
		  xrealloc (array, (array_size += 10) * sizeof (char *));

	      array[array_index++] = current_line;
	      array[array_index] = (char *)NULL;
	    }

	  free_headers (headers);
	}
    }
  fclose (stream);
  return (array);
}

/* Just like from-headers, but prints the headers and message bodies
   to STREAM. */
from_message_bodies (filename, format_spec, filter_regexp,
		     filter_field, all, stream)
     char *filename, *format_spec, *filter_regexp, *filter_field;
     int all;
     FILE *stream;
{
  FILE *input_stream;
  int rmail = 0, skip_message;
  char *current_line;

  input_stream = fopen (filename, "r");

  if (!input_stream)
    return;

  if (!format_spec)
    format_spec = DEFAULT_FORMAT_SPEC;

  if (filter_regexp)
    re_comp (filter_regexp);

  skip_message = 1;

  while (fgets (io_buffer, sizeof (io_buffer), input_stream))
    {
      int found = 0;

      /* Find start of headers for a given message.  We search RMAIL
	 and Unix mail files. */
      if (!rmail && strncmp (io_buffer, "From ", 5) == 0)
	found = 1;
      else if (strcmp (io_buffer, "\037\f\n") == 0)
	{
	  rmail = 1;
	  fgets (io_buffer, sizeof (io_buffer), input_stream);

	  if (strcmp (io_buffer, "0, unseen,,\n") == 0)
	    fgets (io_buffer, sizeof (io_buffer), input_stream);

	  found = 1;
	}

      if (found)
	{
	  HEADER **headers, **snarf_headers ();
	  char *header_value ();

	  skip_message = 0;

	  headers = snarf_headers (input_stream);

	  /* Check to see if this message is new. */
	  if (!all)
	    {
	      char *status = header_value ("Status", headers);

	      if (rindex (status, 'R'))
		{
		  free_headers (headers);
		  skip_message = 1;
		  continue;
		}
	    }

	  if (filter_regexp)
	    {
	      if (re_exec (header_value (filter_field, headers)) == 1)
		current_line = displayable_string (format_spec, headers);
	      else
		current_line = (char *)NULL;
	    }
	  else
	    current_line = displayable_string (format_spec, headers);

	  if (current_line)
	    {
	      fprintf (stream, "%s\n", current_line);
	      free (current_line);
	    }
	  else
	    skip_message = 1;

	  free_headers (headers);
	}
      else if (!skip_message)		/* If in message body. */
	fprintf (stream, "%s", io_buffer);
    }
  fclose (input_stream);
}
	  
/* In GNU From, we allow the user to control the format of the
   output data.  The following variable contains the string which
   is the format specification for which headers and how wide to
   display them.  The specification for inserting a header is as
   follows:

     %[width]FIELD_NAME

   So, a specification string of:

     "%18From on %11Date: RE:%Subject"

   might display:

     Brian Fox         on Sat, 29 Dec: RE:This is a test message

   The default string is "%18From %11Date  %48Subject", and can be
   changed with the "-format" flag, or from your .fromrc file. */

/* Create a new string from SPEC and HEADERS which is a single, displayable
   line of header information. */
char *
displayable_string (spec, headers)
     char *spec;
     HEADER **headers;
{
  register int i, end, c;
  int width, size, string_index;
  char *string, *parse_sender ();

  size = string_index = 0;
  string = (char *)xmalloc (size = (80 + strlen (spec)));
  *string = '\0';

  for (i = 0; c = spec[i]; i++)
    {
      if (c != '%')
	{
	add_character:
	  if (string_index + 2 >= size)
	    string = (char *)xrealloc (string, (size += 80));

	  string[string_index++] = c;
	  string[string_index] = '\0';
	  continue;
	}
      else
	{
	  char *name, *value;

	  /* Parse out the width specification and the field name.
	     Insert the results at STRING_INDEX. */

	  i++;

	  /* If the specification is another percent sign, then just
	     include that here. */
	  if (spec[i] == '%')
	    goto add_character;

	  /* Get possible width specification. */
	  width = 0;

	  for (end = i; digit (spec[end]); end++)
	    width = (10 * width) + (digit_value (spec[end]));

	  /* Now END points at the start of the field name.  Anything
	     that can be a field-name character can be used. */
	  i = end;

	  for (; tag_character (spec[end]); end++);

	  name = (char *)xmalloc (1 + (end - i));
	  strncpy (name, spec + i, end - i);
	  name[end - i] = '\0';

	  i = end;

	  if (*name)
	    i--;

	  value = header_value (name, headers);
	  free (name);

	  /* Special case for "From" field, we call parse_sender ()
	     on the header value. */
	  if (stricmp (name, "From") == 0)
	    value = parse_sender (value);

	  /* Make this string reallocable. */
	  value = savestring (value);

	  /* Pad or suppress the value to make it match the width
	     specification. */
	  if (width)
	    {
	      if (width > strlen (value))
		{
		  value = (char *)xrealloc (value, 1 + width);
		  for (end = strlen (value); end < width; end++)
		    value[end] = ' ';
		}
	      value[width] = '\0';
	    }

	  /* Add the value to the string. */
	  if (string_index + strlen (value) + 1 >= size)
	    string = (char *)
	      xrealloc (string, (size += (20 + strlen (value))));

	  strcat (string, value);
	  free (value);
	  string_index = strlen (string);
	}
    }

  /* Strip trailing whitespace from string. */
  if (string)
    for (i = (strlen (string) - 1); i >= 0 && whitespace (string[i]); i--)
      string[i] = '\0';

  return (string);
}

/* Return an array of the headers read from stream.  This reads lines
   from stream until a non-header line is found. */
HEADER **
snarf_headers (stream)
     FILE *stream;
{
  register int i, c;
  HEADER **headers = (HEADER **)NULL;
  int header_index, header_size, label_parsed = 0;

  header_index = header_size = 0;

  while (fgets (io_buffer, sizeof (io_buffer), stream))
    {
      if (io_buffer[strlen (io_buffer) - 1] == '\n')
	io_buffer[strlen (io_buffer) - 1] = '\0';

      /* See if this is a header line.  Must find a ':' before first
	 whitespace character.  Exception:  Lines that begin with whitespace,
	 contain text, and follow a header tag line are continuations
	 of the previous header. */
      if (whitespace (io_buffer[0]))
	  {
	    for (i = 0; io_buffer[i]; i++)
	      if (!whitespace (io_buffer[i]))
		break;

	    if (io_buffer[i] && label_parsed)
	      continue;
	    else
	      break;
	  }

      for (i = 0; tag_character (io_buffer[i]); i++);

      /* If we find a colon, the text preceding it is the tag,
	 and the text after it is the value. */
      if (io_buffer[i] == ':')
	{
	  char *tag, *value;

	  label_parsed = 1;
	  io_buffer[i] = '\0';
	  tag = savestring (io_buffer);
	  value = savestring (&io_buffer[i + 1]);

	  strip_whitespace (value);

	  if (header_index + 1 >= header_size)
	    headers = (HEADER **)
	      xrealloc (headers, (header_size += 5) * sizeof (HEADER *));

	  headers[header_index] = (HEADER *)xmalloc (sizeof (HEADER));
	  headers[header_index]->tag = tag;
	  headers[header_index]->value = value;
	  headers[++header_index] = (HEADER *)NULL;
	}
      else
	break;
    }

  return (headers);
}

/* Remove the leading and trailing whitespace from STRING. */
void
strip_whitespace (string)
     char *string;
{
  register int i;

  for (i = 0; whitespace (string[i]); i++);

  strcpy (string, string + i);

  for (i = strlen (string); i; i--)
    if (whitespace (string[i - 1]))
      string[i - 1] = '\0';
    else
      break;
}

/* Return a string which is the value of TAG in HEADERS.
   If TAG isn't found, then return an empty (not null!) string. */
char *
header_value (tag, headers)
     char *tag;
     HEADER **headers;
{
  register int i;
  HEADER *header;
  char *value = (char *)NULL;

  /* Do it backwards.  In that way, we get the original sender, original
     date, original subject, etc. */

  if (!headers)
    return ("");

  for (i = 0; header = headers[i]; i++)
    ;

  for (--i; (i >= 0) && (header = headers[i]); i--)
    {
      if (stricmp (header->tag, tag) == 0)
	{
	  value = header->value;
	  break;
	}
    }

  if (value)
    return (value);
  else
    return ("");
}

static char sender_buff[1024];

/* Parse out the value of the sender from STRING. */
char *
parse_sender (string)
     char *string;
{
  register int start, i;

  sender_buff[0] = '\0';

  /* If there is text surrounded by parenthesis, the text inside is the
     sender. */
  for (i = 0; string[i] && string[i] != '('; i++);

  if (string[i] == '(')
    {
      start = ++i;

      for (; string[i] && string[i] != ')'; i++);

      strncpy (sender_buff, string + start, i - start);
      sender_buff[i - start] = '\0';
      goto clean;
    }

  /* If there are angle brackets on the line, then the text that precedes
     the angle brackets could be the sender, or the text after the brackets
     could be the sender. */
  for (i = 0; string[i] && string[i] != '<'; i++);

  if (string[i] == '<')
    {
      /* Examine the text preceding point.  If it has alphabetics in it,
	 then it is the name of the sender. */

      for (start = 0; start < i; start++)
	{
	  if (!whitespace (string[start]))
	    {
	      strncpy (sender_buff, string + start, i - start);
	      sender_buff[i - start] = '\0';
	      goto clean;
	    }
	}
    }
      
  /* Try for text after the closing angle bracket. */
  for (i = 0; string[i] && string[i] != '>'; i++);

  if (string[i] == '>')
    {
      /* Examine the text after point.  If it has alphabetics in it,
	 then it is the name of the sender. */

      for (start = ++i; string[start]; start++)
	{
	  if (!whitespace (string[start]))
	    {
	      strcpy (sender_buff, string + start);
	      goto clean;
	    }
	}
    }

  /* Otherwise, the entire string is the sender. */
  strcpy (sender_buff, string);

 clean:

  /* If the sender's name is surrounded with double quotes, then
     strip them off. */
  if (*sender_buff == '"')
    {
      strcpy (sender_buff, sender_buff + 1);

      for (i = 0; sender_buff[i] && sender_buff[i] != '"'; i++);
      sender_buff[i] = '\0';
    }

  strip_whitespace (sender_buff);

  return (sender_buff);
}

/* Release the memory allocated to the HEADERS passed. */
void
free_headers (headers)
     HEADER **headers;
{
  register int i;
  HEADER *header;

  if (!headers)
    return;

  for (i = 0; header = headers[i]; i++)
    {  
      free (header->tag);
      free (header->value);
      free (header);
    }

  free (headers);
}

/* strcmp (), but caseless. */
int
stricmp (string1, string2)
     char *string1, *string2;
{
  register char ch1, ch2;

  while (*string1 && *string2)
    {
      ch1 = *string1++;
      ch2 = *string2++;
      if (to_upper(ch1) != to_upper(ch2))
	return (1);
    }
  return (*string1 | *string2);
}

void *
xmalloc (size)
     int size;
{
  register char *temp;

  temp = (char *) malloc (size);

  if (!temp)
    memory_die ();

  return (void *)(temp);
}

void *
xrealloc (pointer, size)
     void *pointer;
     int size;
{
  char *temp;

  if (!pointer)
    return (xmalloc (size));

  pointer = (void *) realloc (pointer, size);

  if (!pointer)
    memory_die ();

  return (pointer);
}

memory_die ()
{
  fprintf (stderr, "Out of virtual memory");
  abort ();
}
