/* glpk.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPK_H
#define _GLPK_H

#ifndef _GLPAPI_H
typedef struct { int dummy; } LPI;
#endif

#define GLP_UNDEF    0  /* solution is undefined */
#define GLP_OPT      1  /* solution is optimal */
#define GLP_FEAS     2  /* solution is feasible */
#define GLP_INFEAS   3  /* solution is infeasible */
#define GLP_NOFEAS   4  /* problem has no feasible solution */
#define GLP_UNBND    5  /* problem has unbounded solution */
#define GLP_INTOPT   6  /* solution is integer optimal */
#define GLP_INTSOL   7  /* solution is integer feasible */
#define GLP_DISINT   8  /* solution is integer infeasible */
#define GLP_NOINT    9  /* problem has no integer solution */

extern void glp_check_data(LPI *lp, int check);
/* turn on/off data consistency checking flag */

extern int glp_check_name(char *name);
/* check symbolic name for correctness */

extern void glp_clear_mat_cols(LPI *lp, int flag[]);
/* nullify columns of constraint matrix */

extern void glp_clear_mat_rows(LPI *lp, int flag[]);
/* nullify rows of constraint matrix */

extern LPI *glp_create_prob(char *name);
/* create problem instance */

extern void glp_delete_cols(LPI *lp, int flag[]);
/* delete columns from problem instance */

extern void glp_delete_prob(LPI *lp);
/* delete problem instance */

extern void glp_delete_rows(LPI *lp, int flag[]);
/* delete rows from problem instance */

extern int glp_find_col(LPI *lp, char *name);
/* find column by name */

extern int glp_find_row(LPI *lp, char *name);
/* find row by name */

extern int glp_get_bstat(LPI *lp);
/* determine basis status */

extern void glp_get_col_bnds(LPI *lp, int j, int *type, double *lb,
      double *ub);
/* determine column bounds */

extern int glp_get_col_coef(LPI *lp, int j, int rn[], double aj[]);
/* get column of constraint matrix */

extern int glp_get_col_kind(LPI *lp, int j);
/* get column kind */

extern char *glp_get_col_name(LPI *lp, int j);
/* get column name */

extern void glp_get_col_soln(LPI *lp, int j, int *tagx, double *valx,
      double *dx);
/* obtain column solution information */

extern int glp_get_num_bin(LPI *lp);
/* determine number of binary columns */

extern int glp_get_num_cols(LPI *lp);
/* determine number of columns */

extern int glp_get_num_int(LPI *lp);
/* determine number of integer columns */

extern int glp_get_num_nz(LPI *lp);
/* determine number of constraint coefficients */

extern int glp_get_num_rows(LPI *lp);
/* determine number of rows */

extern double glp_get_obj_coef(LPI *lp, int j);
/* get objective function coefficient */

extern int glp_get_obj_sense(LPI *lp);
/* get objective function sense */

extern double glp_get_obj_val(LPI *lp);
/* obtain objective function value */

extern char *glp_get_prob_name(LPI *lp);
/* obtain problem name */

extern void glp_get_row_bnds(LPI *lp, int i, int *type, double *lb,
      double *ub);
/* determine row bounds */

extern int glp_get_row_coef(LPI *lp, int i, int cn[], double ai[]);
/* get row of constraint matrix */

extern char *glp_get_row_name(LPI *lp, int i);
/* get row name */

extern void glp_get_row_soln(LPI *lp, int i, int *tagx, double *valx,
      double *dx);
/* obtain row solution information */

extern int glp_get_status(LPI *lp);
/* determine solution status */

extern void glp_new_aij(LPI *lp, int i, int j, double aij);
/* create new constraint coefficient */

extern void glp_new_col(LPI *lp, char *name);
/* create new column */

extern void glp_new_row(LPI *lp, char *name);
/* create new row */

extern int glp_print_soln(LPI *lp, char *fname);
/* write problem solution using printable format */

extern void glp_put_col_soln(LPI *lp, int j, int tagx, double valx,
      double dx);
/* store column solution information */

extern void glp_put_row_soln(LPI *lp, int i, int tagx, double valx,
      double dx);
/* store row solution information */

extern void glp_put_soln_info(LPI *lp, int bstat, int status,
      double objv);
/* store main solution information */

extern LPI *glp_read_lpm1(char *infile, char *outfile);
/* read linear programming model written in GLPK/L */

extern LPI *glp_read_mps1(char *fname);
/* read problem data using MPS format */

extern void glp_set_col_bnds(LPI *lp, int j, int type, double lb,
      double ub);
/* set column bounds */

extern void glp_set_col_kind(LPI *lp, int j, int kind);
/* set column kind */

extern void glp_set_col_name(LPI *lp, int j, char *name);
/* assign (change) column name */

extern void glp_set_obj_coef(LPI *lp, int j, double coef);
/* set objective function coefficient */

extern void glp_set_obj_sense(LPI *lp, int sense);
/* set objective function sense */

extern void glp_set_prob_name(LPI *lp, char *name);
/* assign problem name */

extern void glp_set_row_bnds(LPI *lp, int i, int type, double lb,
      double ub);
/* set row bounds */

extern void glp_set_row_name(LPI *lp, int i, char *name);
/* assign (change) row name */

struct rsm1
{     /* revised simplex method parameter block */
      int what;
      /* this parameter specifies what basis solution should be found
         by the solver:
         0 - initial solution
         1 - primal feasible solution
         2 - optimal solution */
      int form;
      /* this parameter specifies what form of the basis matrix should
         be used by the solver:
         0 - EFI
         1 - RFI + Bartels & Golub updating technique
         2 - RFI + Forrest & Tomlin updating technique */
      int scale;
      /* if this flag is set, the solver performs automatic scaling the
         problem; otherwise the scaling is not used */
      int dual;
      /* if this flag is set, the solver uses the dual simplex method
         in order to find the primal feasible basis solution; otherwise
         the solver uses the primal simplex method */
      int steep;
      /* if this flag is set, the solver uses the steepest edge pricing
         (for both primal and dual simplex) proposed by Goldfarb & Reid;
         otherwise the solver uses the standard "textbook" pricing */
      int relax;
      /* if this flag is set, the solver uses two-pass ratio test (for
         both primal and dual simplex) proposed by P.Harris; otherwise
         the solver uses the standard "textbook" ratio test */
      double tol_bnd;
      /* relative tolerance which is used to see if the solution is
         primal feasible */
      double tol_dj;
      /* relative tolerance which is used to see if the solution is
         dual feasible */
      double tol_piv;
      /* relative tolerance which is used to choose the pivot element
         of the simplex table */
      int iter_max;
      /* maximal allowed number of iterations (0 means no limit) */
      int round;
      /* if this flag is set, the solver replaces computed values of
         basic variables and reduced costs of non-basic variables which
         are close to zero by exact zeros; otherwise the solver remains
         all computed values and reduced costs "as is" */
};

extern void glp_init_rsm1(struct rsm1 *parm);
/* initialize parameter block by default values */

extern int glp_call_rsm1(LPI *lpi, struct rsm1 *parm);
/* solve LP problem using revised simplex method */

extern int glp_call_ipm1(LPI *lpi);
/* solve LP problem using interior point method */

struct bbm1
{     /* branch-and-bound method parameter block */
      int what;
      /* this parameter specifies what basis solution should be found
         by the solver:
         0 - initial relaxed solution
         1 - feasible integer solution
         2 - optimal integer solution */
      int branch;
      /* this parameter specifies what branching heuristic should be
         used by the solver: */
#define BB_FIRST  0  /* branch on the first variable */
#define BB_LAST   1  /* branch on the last variable */
#define BB_DRTOM  2  /* branch using heuristic by Driebeck and Tomlin */
      int btrack;
      /* this parameter specifies what backtracking heuristic should be
         used by the solver: */
#define BB_FIFO   0  /* backtrack using FIFO heuristic */
#define BB_LIFO   1  /* backtrack using LIFO heuristic */
#define BB_BESTP  2  /* backtrack using the best projection heuristic */
      double tol_int;
      /* absolute tolerance which is used to see if the solution is
         integer feasible */
      double tol_obj;
      /* relative tolerance which is used to check if current value of
         the objective function is not better than for the best integer
         feasible solution found */
      int form;
      /* this parameter specifies what form of the basis matrix should
         be used by the solver:
         0 - EFI
         1 - RFI + Bartels & Golub updating technique
         2 - RFI + Forrest & Tomlin updating technique */
      int scale;
      /* if this flag is set, the solver performs automatic scaling the
         problem; otherwise the scaling is not used */
      int dual;
      /* if this flag is set, the solver uses the dual simplex method
         in order to find the primal feasible basis solution; otherwise
         the solver uses the primal simplex method */
      int steep;
      /* if this flag is set, the solver uses the steepest edge pricing
         proposed by Goldfarb & Reid; otherwise the standard "textbook"
         pricing is used */
      int relax;
      /* if this flag is set, the solver uses two-pass ratio test
         proposed by P.Harris; otherwise the standard "textbook" ratio
         test is used */
      double tol_bnd;
      /* relative tolerance which is used to see if the solution is
         primal feasible */
      double tol_dj;
      /* relative tolerance which is used to see if the solution is
         dual feasible */
      double tol_piv;
      /* relative tolerance which is used to choose the pivot element
         of the simplex table */
      int iter_max;
      /* maximal allowed number of iterations (0 means no limit) */
      int round;
      /* if this flag is set, the solver replaces computed values of
         basic continuous variables which are close to zero by exact
         zeros; otherwise all computed values are remained "as is" */
};

extern void glp_init_bbm1(struct bbm1 *parm);
/* initialize parameter block by default values */

extern int glp_call_bbm1(LPI *lpi, struct bbm1 *parm);
/* solve MIP problem using branch-and-bound method */

#ifdef GLP_OLD_API

#define GLP_MAX_NAME 255
/* maximal length of item name (not including '\0') */

#define GLP_NO       0     /* false */
#define GLP_YES      1     /* true */

#define GLP_ROW      101   /* item is row */
#define GLP_COL      102   /* item is column */

#define GLP_FR       201   /* free variable:  -inf <  x[k] <  +inf */
#define GLP_LO       202   /* lower bound:    l[k] <= x[k] <  +inf */
#define GLP_UP       203   /* upper bound:    -inf <  x[k] <= u[k] */
#define GLP_DB       204   /* double bound:   l[k] <= x[k] <= u[k] */
#define GLP_FX       205   /* fixed variable: l[k]  = x[k]  = u[k] */

#define GLP_MIN      301   /* objective function should be minimized */
#define GLP_MAX      302   /* objective function should be maximized */

#define GLP_INI      311   /* only initial solution is required */
#define GLP_ANY      312   /* any feasible solution is required */
#define GLP_FIN      313   /* final solution is required */

#define GLP_BS       501   /* basic variable */
#define GLP_NL       502   /* non-basic variable on its lower bound */
#define GLP_NU       503   /* non-basic variable on its upper bound */
#define GLP_NF       504   /* non-basic free variable */
#define GLP_NS       505   /* non-basic fixed variable */

#define GLP_EFI      601   /* EFI */
#define GLP_RFI_BG   602   /* RFI + Bartels & Golub updating tech */
#define GLP_RFI_FT   603   /* RFI + Forrest & Tomlin updating tech */

#define GLP_FIRST    701   /* branch on the first variable */
#define GLP_LAST     702   /* branch on the last variable */
#define GLP_DRTOM    703   /* branch using Driebeek & Tomlin heur */

#define GLP_FIFO     801   /* backtrack using FIFO strategy */
#define GLP_LIFO     802   /* backtrack using LIFO strategy */
#define GLP_BESTP    803   /* backtrack using best projection heur */

#define GLP_D_PARS   0x01  /* dump control parameters */
#define GLP_D_ROWS   0x10  /* dump rows */
#define GLP_D_RMAT   0x20  /* dump coefficients for each row */
#define GLP_D_COLS   0x40  /* dump columns */
#define GLP_D_CMAT   0x80  /* dump coefficients for each column */
#define GLP_D_ALL    0xFF  /* dump all information */

extern int glp_check_name(char *name);
/* check symbolic name for correctness */

extern int glp_create_item(int what, char *name);
/* create new row or column */

extern int glp_dump_ws(char *fname, int what);
/* dump GLPK API workspace */

extern int glp_find_item(int what, char *name);
/* find row or column by name */

extern int glp_first_coef(int what);
/* find the first constraint coefficient */

extern int glp_first_item(int what);
/* find the first row or column */

extern int glp_get_activity(int what, int *tagx, double *valx,
      double *dx);
/* get activity of the current row or column */

extern int glp_get_bounds(int what, int *type, double *lb, double *ub);
/* get bounds of the current row or column */

extern int glp_get_coef(double *val);
/* get the current constraint coefficient */

extern int glp_get_cpar(char *name, char *val);
/* get value of text control parameter */

extern int glp_get_kind(int what, int *kind);
/* get kind of the current row or column */

extern int glp_get_ipar(char *name, int *val);
/* get value of integer control parameter */

extern int glp_get_name(int what, char *name);
/* get name of the current row or column */

extern int glp_get_rpar(char *name, double *val);
/* get value of real control parameter */

extern int glp_get_seqn(int what, int *seqn);
/* get seqn attribute of the current row or column */

extern int glp_initialize(void);
/* initialize GLPK application program interface */

extern int glp_integer(void);
/* solve MIP problem using branch-and-bound procedure */

extern int glp_interior(void);
/* solve problem by means of the interior point method */

extern int glp_new_coef(double val);
/* create new constraint coefficient */

extern int glp_next_coef(int what);
/* find the next constraint coefficient */

extern int glp_next_item(int what);
/* find the next row or column */

extern int glp_print_sol(char *fname);
/* write problem solution using printable format */

extern int glp_read_lpm(char *fname);
/* read problem data using LPM format */

extern int glp_read_mps(char *fname);
/* read problem data using MPS format */

extern int glp_set_activity(int what, int tagx, double valx, double dx);
/* set activity of the current row or column */

extern int glp_set_bounds(int what, int type, double lb, double ub);
/* set bounds of the current row or column */

extern int glp_set_cpar(char *name, char *val);
/* set value of text control parameter */

extern int glp_set_ipar(char *name, int val);
/* set value of integer control parameter */

extern int glp_set_kind(int what, int kind);
/* set kind of the current row or column */

extern int glp_set_name(int what, char *name);
/* change name of the current row or column */

extern int glp_set_rpar(char *name, double val);
/* set value of real control parameter */

extern int glp_set_seqn(int what, int seqn);
/* set seqn attribute of the current row or column */

extern int glp_simplex(void);
/* solve problem by means of the revised simplex method */

extern int glp_terminate(void);
/* terminate GLPK application program interface */

extern int glp_write_mps(char *fname);
/* write problem data using MPS format */

#endif /* GLP_OLD_API */

#endif

/* eof */
