/* glprsm/update_gvec.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- update_gvec - update the vector gamma.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void update_gvec(RSM *rsm, double gvec[], int p, int q, double ap[],
--    double aq[], double w[]);
--
-- *Description*
--
-- The update_gvec routine replaces the vector gamma which corresponds
-- to the current basis by the updated vector which corresponds to the
-- adjacent basis.
--
-- On entry the array gvec should contain elements of the vector
-- gamma = (gamma_1, ..., gamma_n) for the current basis in locations
-- gvec[1], ..., gvec[n] respectively. On exit this array will contain
-- elements of the updated vector for the adjacent basis in the same
-- locations.
--
-- The parameter p specifies basic variable (xB)p which has been chosen
-- to leave the basis.
--
-- The parameter q specifies non-basic variable (xN)q which has been
-- chosen to enter the basis.
--
-- On entry the array ap should contain elements of p-th row of the
-- current simplex table in locations ap[1], ..., ap[n]. This array can
-- be computed by means of the eval_row routine. It is not changed on
-- exit.
--
-- On entry the array aq should contain elements of q-th column of the
-- current simplex table in locations aq[1], ..., aq[m]. This array can
-- be computed by means of the eval_col routine. It is not changed on
-- exit.
--
-- The working array w should have at least 1+m locations, where m is
-- order of the basis matrix B.
--
-- The update_gvec routine assumes that the representation of the basis
-- matrix corresponds to the current basis, not to the adjacent one.
-- Therefore this routine should be called before changing the basis.
--
-- For further details see the program documentation. */

void update_gvec(RSM *rsm, double gvec[], int p, int q, double ap[],
      double aq[], double w[])
{     ELEM *e;
      int m = rsm->m, n = rsm->n, i, j, k;
      double apj, tj, t1, t2;
      insist(1 <= p && p <= m && 1 <= q && q <= n);
      gvec[q] = 1.0;
      for (i = 1; i <= m; i++) gvec[q] += aq[i] * aq[i];
      for (i = 1; i <= m; i++) w[i] = aq[i];
      btran(rsm, w);
      for (j = 1; j <= n; j++)
      {  if (j == q) continue;
         apj = - ap[j] / aq[p];
         tj = 0.0;
         k = rsm->indn[j];
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            tj += e->val * w[e->i];
         t1 = gvec[j] + apj * apj * gvec[q] - 2.0 * apj * tj;
         t2 = 1.0 + apj * apj;
         gvec[j] = t1 > t2 ? t1 : t2;
      }
      gvec[q] /= (aq[p] * aq[p]);
      return;
}

/* eof */
