/* glpapi/glp_set_col_bnds.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_set_col_bnds - set column bounds.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_set_col_bnds(LPI *lp, int j, int type, double lb,
--    double ub);
--
-- *Description*
--
-- The routine glp_set_col_bnds sets (changes) type and bounds of the
-- j-th column.
--
-- Parameters type, lb, and ub specify respectively the type, the lower
-- bound, and the upper bound, which should be set for the j-th column:
--
--    Type        Bounds            Note
--    ----------------------------------------
--    'F'   -inf <  x <  +inf   free variable
--    'L'     lb <= x <  +inf   lower bound
--    'U'   -inf <  x <=  ub    upper bound
--    'D'     lb <= x <=  ub    double bound
--    'S'           x  =  lb    fixed variable
--
-- where x is the corresponding structural variable.
--
-- If the column has no lower bound, the parameter lb is ignored. If
-- the column has no upper bound, the parameter ub is ignored. If the
-- column is of fixed type, the parameter lb is used, and the parameter
-- ub is ignored. */

void glp_set_col_bnds(LPI *lp, int j, int type, double lb, double ub)
{     if (!(1 <= j && j <= lp->n))
         fault("glp_set_col_bnds: j = %d; invalid column number", j);
      lp->col[j]->type = type;
      switch (type)
      {  case 'F':
            lp->col[j]->lb = lp->col[j]->ub = 0.0;
            break;
         case 'L':
            lp->col[j]->lb = lb, lp->col[j]->ub = 0.0;
            break;
         case 'U':
            lp->col[j]->lb = 0.0, lp->col[j]->ub = ub;
            break;
         case 'D':
            lp->col[j]->lb = lb, lp->col[j]->ub = ub;
            break;
         case 'S':
            lp->col[j]->lb = lp->col[j]->ub = lb;
            break;
         default:
            fault("glp_set_col_bnds: type = %d; invalid column type",
               type);
      }
      lp->basis = '?', lp->status = GLP_UNDEF;
      return;
}

/* eof */
