/* glplpx7.c (simplex table routines) */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include "glpspx.h"

/*----------------------------------------------------------------------
-- lpx_eval_tab_row - compute row of the simplex table.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_eval_tab_row(LPX *lp, int k, int ndx[], double val[]);
--
-- *Description*
--
-- The routine lpx_eval_tab_row computes a row of the current simplex
-- table for the basic variable, which is specified by the number k:
-- if 1 <= k <= m, x[k] is k-th auxiliary variable; if m+1 <= k <= m+n,
-- x[k] is (k-m)-th structural variable, where m is number of rows, and
-- n is number of columns. The current basis must be valid.
--
-- The routine stores column indices and numerical values of non-zero
-- elements of the computed row using sparse format to the locations
-- ndx[1], ..., ndx[len] and val[1], ..., val[len] respectively, where
-- 0 <= len <= n is number of non-zeros returned on exit.
--
-- Element indices strored in the array ndx have the same sense as the
-- index k, i.e. indices 1 to m denote auxiliary variables and indices
-- m+1 to m+n denote structural ones (all these variables are obviously
-- non-basic by the definition).
--
-- The computed row shows how the specified basic variable x[k] = xB[i]
-- depends on non-basic variables:
--
--    xB[i] = alfa[i,1]*xN[1] + alfa[i,2]*xN[2] + ... + alfa[i,n]*xN[n],
--
-- where alfa[i,j] are elements of the simplex table row, xN[j] are
-- non-basic (auxiliary and structural) variables.
--
-- Even if the problem is (internally) scaled, the routine returns the
-- specified simplex table row as if the problem were unscaled.
--
-- *Returns*
--
-- The routine returns number of non-zero elements in the simplex table
-- row stored in the arrays ndx and val.
--
-- *Unscaling*
--
-- Let A~ = (I | -A) is the augmented constraint matrix of the original
-- (unscaled) problem. In the scaled LP problem instead the matrix A the
-- scaled matrix A" = R*A*S is actually used, so
--
--    A~" = (I | A") = (I | R*A*S) = (R*I*inv(R) | R*A*S) =
--                                                                   (1)
--        = R*(I | A)*S~ = R*A~*S~,
--
-- is the scaled augmented constraint matrix, where R and S are diagonal
-- scaling matrices used to scale rows and columns of the matrix A, and
--
--    S~ = diag(inv(R) | S)                                          (2)
--
-- is an augmented diagonal scaling matrix.
--
-- By the definition the simplex table is the matrix
--
--    T = - inv(B) * N,                                              (3)
--
-- where B is the basis matrix that consists of basic columns of the
-- augmented constraint matrix A~, and N is a matrix that consists of
-- non-basic columns of A~. From (1) it follows that
--
--    A~" = (B" | N") = (R*B*SB | R*N*SN),                           (4)
--
-- where SB and SN are parts of the augmented scaling matrix S~ that
-- correspond to basic and non-basic variables respectively. Thus, in
-- the scaled case
--
--    T" = - inv(B") * N" = - inv(R*B*SB) * (R*N*SN) =
--
--       = - inv(SB)*inv(B)*inv(R) * (R*N*SN) =                      (5)
--
--       = inv(SB) * (-inv(B) * N) * SN = inv(SB) * T * SN,
--
-- that allows us expressing the original simplex table T through the
-- scaled simplex table T":
--
--    T = SB * T" * inv(SN).                                         (6)
--
-- The formula (6) is used by the routine for unscaling elements of the
-- computed simplex table row. */

int lpx_eval_tab_row(LPX *lp, int k, int ndx[], double val[])
{     int m = lp->m, n = lp->n;
      int i, j, len;
      double *rho, *row, sb_i, sn_j;
      if (!(1 <= k && k <= m+n))
         fault("lpx_eval_tab_row: k = %d; variable number out of range",
            k);
      if (lp->b_stat != LPX_B_VALID)
         fault("lpx_eval_tab_row: current basis is undefined");
      if (lp->tagx[k] != LPX_BS)
         fault("lpx_eval_tab_row: k = %d; variable should be basic", k);
      i = lp->posx[k]; /* x[k] = xB[i] */
      insist(1 <= i && i <= m);
      /* allocate working arrays */
      rho = ucalloc(1+m, sizeof(double));
      row = ucalloc(1+n, sizeof(double));
      /* compute i-th row of the inverse inv(B) */
      spx_eval_rho(lp, i, rho);
      /* compute i-th row of the simplex table */
      spx_eval_row(lp, rho, row);
      /* unscale and store the required row */
      sb_i = (k <= m ? 1.0 / lp->rs[k] : lp->rs[k]);
      len = 0;
      for (j = 1; j <= n; j++)
      {  if (row[j] != 0.0)
         {  k = lp->indx[m+j]; /* x[k] = xN[j] */
            sn_j = (k <= m ? 1.0 / lp->rs[k] : lp->rs[k]);
            len++;
            ndx[len] = k;
            val[len] = (sb_i / sn_j) * row[j];
         }
      }
      /* free working arrays */
      ufree(rho);
      ufree(row);
      /* return to the calling program */
      return len;
}

/*----------------------------------------------------------------------
-- lpx_eval_tab_col - compute column of the simplex table.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_eval_tab_col(LPX *lp, int k, int ndx[], double val[]);
--
-- *Description*
--
-- The routine lpx_eval_tab_col computes a column of the current simplex
-- table for the non-basic variable, which is specified by the number k:
-- if 1 <= k <= m, x[k] is k-th auxiliary variable; if m+1 <= k <= m+n,
-- x[k] is (k-m)-th structural variable, where m is number of rows, and
-- n is number of columns. The current basis must be valid.
--
-- Note that the current basis should be valid.
--
-- The routine stores row indices and numerical values of non-zero
-- elements of the computed column using sparse format to the locations
-- ndx[1], ..., ndx[len] and val[1], ..., val[len] respectively, where
-- 0 <= len <= m is number of non-zeros returned on exit.
--
-- Element indices strored in the array ndx have the same sense as the
-- index k, i.e. indices 1 to m denote auxiliary variables and indices
-- m+1 to m+n denote structural ones (all these variables are obviously
-- basic by the definition).
--
-- The computed column shows how basic variables depend on the specified
-- non-basic variable x[k] = xN[j]:
--
--    xB[1] = ... + alfa[1,j]*xN[j] + ...
--    xB[2] = ... + alfa[2,j]*xN[j] + ...
--             . . . . . .
--    xB[m] = ... + alfa[m,j]*xN[j] + ...
--
-- where alfa[i,j] are elements of the simplex table column, xB[i] are
-- basic (auxiliary and structural) variables.
--
-- Even if the problem is (internally) scaled, the routine returns the
-- specified simplex table column as if the problem were unscaled.
--
-- *Returns*
--
-- The routine returns number of non-zero elements in the simplex table
-- column stored in the arrays ndx and val.
--
-- *Unscaling*
--
-- The routine unscales elements of the computed simplex table column
-- using the formula (6), which is derived and explained in description
-- of the routine lpx_eval_tab_row (see above). */

int lpx_eval_tab_col(LPX *lp, int k, int ndx[], double val[])
{     int m = lp->m, n = lp->n;
      int i, j, len;
      double *col, sb_i, sn_j;
      if (!(1 <= k && k <= m+n))
         fault("lpx_eval_tab_col: k = %d; variable number out of range",
            k);
      if (lp->b_stat != LPX_B_VALID)
         fault("lpx_eval_tab_col: current basis is undefined");
      if (lp->tagx[k] == LPX_BS)
         fault("lpx_eval_tab_col; k = %d; variable should be non-basic",
            k);
      j = lp->posx[k] - m; /* x[k] = xN[j] */
      insist(1 <= j && j <= n);
      /* allocate working array */
      col = ucalloc(1+m, sizeof(double));
      /* compute j-th column of the simplex table */
      spx_eval_col(lp, j, col, 0);
      /* unscale and store the required column */
      sn_j = (k <= m ? 1.0 / lp->rs[k] : lp->rs[k]);
      len = 0;
      for (i = 1; i <= m; i++)
      {  if (col[i] != 0.0)
         {  k = lp->indx[i]; /* x[k] = xB[i] */
            sb_i = (k <= m ? 1.0 / lp->rs[k] : lp->rs[k]);
            len++;
            ndx[len] = k;
            val[len] = (sb_i / sn_j) * col[i];
         }
      }
      /* free working array */
      ufree(col);
      /* return to the calling program */
      return len;
}

/* eof */
