/* glprsm2.c (afi) */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <math.h>
#include <stddef.h>
#include <time.h>
#include "glpfhv.h"
#include "glprsm.h"

typedef struct AFI AFI;

struct AFI
{     /* advanced form of the inverse (AFI) */
      int m;
      /* order of the basis matrix */
      FHV *fhv;
      /* FHV-factorization of the basis matrix */
      MAT *A;
      /* augmented constraint matrix from RSM data structure */
      int *indb;
      /* indices of basic columns from RSM data structure */
      int t_build;
      /* total time in build_afi() */
      int t_ftran;
      /* total time in afi_ftran() */
      int t_btran;
      /* total time in afi_btran() */
      int t_update;
      /* total time in update_afi() */
};

/*----------------------------------------------------------------------
-- create_afi - create AFI.
--
-- *Synopsis*
--
-- #include "glpafi.h"
-- void *create_afi(int m);
--
-- *Description*
--
-- The routine create_afi creates AFI for the basis matrix of order m.
-- Initially the created AFI corresponds to the unity matrix.
--
-- *Returns*
--
-- The routine create_afi returns a pointer to the created AFI. */

void *create_afi(int m)
{     AFI *afi;
      if (m < 1)
         fault("create_afi: m = %d; invalid order", m);
      afi = umalloc(sizeof(AFI));
      afi->m = m;
      afi->fhv = fhv_create(m, 100);
      afi->A = NULL;
      afi->indb = NULL;
      afi->t_build = 0;
      afi->t_ftran = 0;
      afi->t_btran = 0;
      afi->t_update = 0;
      return (void *)afi;
}

/*----------------------------------------------------------------------
-- build_afi - build AFI for given basis matrix.
--
-- *Synopsis*
--
-- #include "glpafi.h"
-- int build_afi(void *afi, MAT *A, int indb[]);
--
-- *Description*
--
-- The build_afi routine builds AFI for the given basis matrix B (see
-- below) in the form B = F*H*V. This is used when: (a) the current AFI
-- became inaccurate, (b) the current AFI requires too much memory, and
-- (c) the basis matrix B was completely changed.
--
-- The given basis matrix B should be specified implicitly by the matrix
-- A and the array indb. The matrix A should have m rows, where m is the
-- order of the basis matrix B. The array indb should specify a list of
-- column numbers of the matrix A, which form the matrix B. These column
-- numbers should be placed in locations indb[1], indb[2], ..., indb[m].
--
-- *Returns*
--
-- If the AFI has been built, the routine returns zero. Otherwise, the
-- routine returns non-zero. The latter case can happen if the matrix B
-- is numerically singular or ill conditioned; for details see the gel
-- routine. */

static int column(void *info, int j, int rn[], double bj[])
{     /* build j-th column of the current basis matrix */
      AFI *afi = (AFI *)info;
      ELEM *e;
      int nz = 0;
      insist(afi->A != NULL);
      insist(afi->indb != NULL);
      for (e = afi->A->col[afi->indb[j]]; e != NULL; e = e->col)
         if (e->val != 0) nz++, rn[nz] = e->i, bj[nz] = e->val;
      return nz;
}

#if 1 /* 3.0.5; piv_tol = 0.03 is too small ! */
int build_afi(void *_afi, MAT *A, int indb[])
{     AFI *afi = (AFI *)_afi;
      static double tol[1+3] = { 0.00, 0.10, 0.30, 0.70 };
      clock_t timer;
      int try, ret;
      afi->A = A;
      afi->indb = indb;
      timer = clock();
      for (try = 1; try <= 3; try++)
      {  afi->fhv->piv_tol = tol[try];
         ret = fhv_decomp(afi->fhv, column, afi);
         if (ret == 0) break;
      }
      afi->t_build += clock() - timer;
      return ret;
}
#else
int build_afi(void *_afi, MAT *A, int indb[])
{     AFI *afi = (AFI *)_afi;
      clock_t timer;
      int ret;
      afi->A = A;
      afi->indb = indb;
      timer = clock();
loop: ret = fhv_decomp(afi->fhv, column, afi);
      if (ret != 0)
      {  error("build_afi: return code = %d", ret);
         if (afi->fhv->piv_tol < 0.29)
         {  afi->fhv->piv_tol = 0.30;
            print("build_afi: piv_tol set to %g", afi->fhv->piv_tol);
            goto loop;
         }
         if (afi->fhv->piv_tol < 0.69)
         {  afi->fhv->piv_tol = 0.70;
            print("build_afi: piv_tol set to %g", afi->fhv->piv_tol);
            goto loop;
         }
      }
      afi->t_build += clock() - timer;
      return ret;
}
#endif

/*----------------------------------------------------------------------
-- afi_ftran - perform forward transformation (FTRAN) using AFI.
--
-- *Synopsis*
--
-- #include "glpafi.h"
-- double *afi_ftran(void *afi, double z[], int save);
--
-- *Description*
--
-- The afi_ftran routine performs forward transformation of the vector
-- z using AFI which afi points to.
--
-- In order to perform this operation the routine solves the system
-- B*x = z, where B is the basis matrix defined by AFI, x is vector of
-- unknowns (transformed vector that should be computed), z is vector of
-- right-hand sides (given vector that should be transformed). On entry
-- the array z should contain elements of the vector z in locations
-- z[1], z[2], ..., z[m], where m is the order of the matrix B. On exit
-- this array will contain the vector x in the same locations.
--
-- The parameter save is a flag. If this flag is set, it means that the
-- vector z is a column corresponding to that non-basis variable, which
-- has been chosen to enter the basis. And the afi_ftran routine saves
-- this column after partial transformation in order that the update_afi
-- routine could use it to update AFI for adjacent basis matrix. It is
-- assumed that the simplex method routine should perform at least one
-- call to the afi_ftran routine with the save parameter set.
--
-- *Returns*
--
-- The afi_ftran routine returns a pointer to the array z. */

double *afi_ftran(void *_afi, double z[], int save)
{     AFI *afi = (AFI *)_afi;
      clock_t timer;
      timer = clock();
      fhv_ftran(afi->fhv, z, save);
      afi->t_ftran += clock() - timer;
      return z;
}

/*----------------------------------------------------------------------
-- afi_btran - perform backward transformation (BTRAN) using AFI.
--
-- *Synopsis*
--
-- #include "glpafi.h"
-- double *afi_btran(void *afi, double z[]);
--
-- *Description*
--
-- The afi_btran routine performs backward transformation of the vector
-- z using AFI which afi points to.
--
-- In order to perform this operation the routine solves the system
-- B'*x = z, where B' is a matrix transposed to the basis matrix B that
-- is defined by AFI, x is vector of unknowns (transformed vector that
-- should be computed), z is vector of right-hand sides (given vector
-- that should be transformed). On entry the array z should contain
-- elements of the vector z in locations z[1], z[2], ..., z[m], where
-- m is the order of the matrix B. On exit this array will contain the
-- vector x in the same locations.
--
-- *Returns*
--
-- The afi_btran routine returns a pointer to the array z. */

double *afi_btran(void *_afi, double z[])
{     AFI *afi = (AFI *)_afi;
      clock_t timer;
      timer = clock();
      fhv_btran(afi->fhv, z);
      afi->t_btran += clock() - timer;
      return z;
}

/*----------------------------------------------------------------------
-- update_afi - update AFI for adjacent basis matrix.
--
-- *Synopsis*
--
-- #include "glpafi.h"
-- int update_afi(void *afi, int p);
--
-- *Description*
--
-- The update_afi routine recomputes AFI corresponding to the current
-- basis matrix B, so that the updated AFI will correspond to the new
-- (adjacent) basis matrix Bnew, where Bnew is a result of change p-th
-- column of B by other column.
--
-- Note that new p-th column of the basis matrix is passed implicitly to
-- this routine: the update_afi routine assumes that the transformed new
-- p-th column was saved before by the afi_ftran routine.
--
-- *Returns*
--
-- The update_afi routine returns one of the following codes:
--
-- 0 - AFI has been successfully updated;
-- 1 - AFI became inaccurate;
-- 2 - AFI became too long.
--
-- If the returned code is non-zero, AFI should be rebuilt by the means
-- of the build_afi routine. */

int update_afi(void *_afi, int p)
{     AFI *afi = (AFI *)_afi;
      clock_t timer;
      int ret;
      timer = clock();
      ret = fhv_update(afi->fhv, p);
      afi->t_update += clock() - timer;
      switch (ret)
      {  case 0:
            ret = 0; break;
         case 1:
         case 2:
            ret = 1; break;
         case 3:
         case 4:
            ret = 2; break;
         default:
            insist(ret != ret);
      }
      return ret;
}

/*----------------------------------------------------------------------
-- delete_afi - delete AFI.
--
-- *Synopsis*
--
-- #include "glpafi.h"
-- void delete_afi(void *afi);
--
-- *Description*
--
-- The delete_afi routine deletes AFI, which afi points to, freeing all
-- the memory allocated to this object. */

void delete_afi(void *_afi)
{     AFI *afi = (AFI *)_afi;
#if 0
      print("build = %.3f; ftran = %.3f; btran = %.3f; update = %.3f",
         (double)afi->t_build / (double)CLOCKS_PER_SEC,
         (double)afi->t_ftran / (double)CLOCKS_PER_SEC,
         (double)afi->t_btran / (double)CLOCKS_PER_SEC,
         (double)afi->t_update / (double)CLOCKS_PER_SEC);
#endif
      fhv_delete(afi->fhv);
      ufree(afi);
      return;
}

/* eof */