/* glplp/extract_prob.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpk.h"
#include "glplp.h"

/*----------------------------------------------------------------------
-- extract_prob - extract LP problem data from LPI object.
--
-- *Synopsis*
--
-- #include "glplp.h"
-- LP *extract_prob(void *lpi);
--
-- *Description*
--
-- The routine extract_prob extracts linear programming problem data
-- from the LPI object (for details see GLPK API).
--
-- *Returns*
--
-- The routine returns a pointer to the LP data block. */

LP *extract_prob(void *lpi)
{     LP *lp;
      int m = glp_get_num_rows(lpi), n = glp_get_num_cols(lpi), i, j;
      int *cn = ucalloc(1+n, sizeof(int));
      double *ai = ucalloc(1+n, sizeof(double));
      if (m == 0)
         fault("extract_prob: problem has no rows");
      if (n == 0)
         fault("extract_prob: problem has no columns");
      lp = create_lp(m, n, glp_get_num_int(lpi) == 0 ? 0 : 1);
      for (i = 1; i <= m; i++)
      {  int nz, t;
         glp_get_row_bnds(lpi, i, &lp->type[i], &lp->lb[i],
            &lp->ub[i]);
         nz = glp_get_row_coef(lpi, i, cn, ai);
         for (t = 1; t <= nz; t++)
            if (ai[t] != 0.0) new_elem(lp->A, i, cn[t], ai[t]);
      }
      for (j = 1; j <= n; j++)
      {  if (glp_get_col_kind(lpi, j) == 'I') lp->kind[j] = 1;
         glp_get_col_bnds(lpi, j, &lp->type[m+j], &lp->lb[m+j],
            &lp->ub[m+j]);
      }
      lp->dir = glp_get_obj_sense(lpi);
      for (j = 0; j <= n; j++) lp->c[j] = glp_get_obj_coef(lpi, j);
      ufree(cn);
      ufree(ai);
      check_lp(lp);
      if (check_mplets(lp->A) != NULL)
         fault("extract_prob: constraint matrix has multiplets");
      return lp;
}

/* eof */
