/* glpapi/glp_get_col_coef.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_get_col_coef - get column of constraint matrix.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_col_coef(LPI *lp, int j, int rn[], double aj[]);
--
-- *Description*
--
-- The routine glp_get_col_coef scans elements of the j-th column of
-- the constraint matrix and stores their row indices and values to
-- locations rn[1], ..., rn[nz'] and aj[1], ..., aj[nz'] respectively,
-- where 0 <= nz' <= m is number of elements in the j-th column, m is
-- number of rows.
--
-- *Returns*
--
-- The routine returns nz', which is the number of stored elements. */

int glp_get_col_coef(LPI *lp, int j, int rn[], double aj[])
{     ELEM *e;
      int nz = 0;
      if (!(1 <= j && j <= lp->n))
         fault("glp_get_col_coef: j = %d; invalid column number", j);
      for (e = lp->col[j]->ptr; e != NULL; e = e->col)
      {  nz++;
         if (nz > lp->m)
            fault("glp_get_col_coef: j = %d; nultiplets detected", j);
         rn[nz] = e->i;
         aj[nz] = e->val;
      }
      return nz;
}

/* eof */
