/* glpset/get_atomv.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpset.h"

/*----------------------------------------------------------------------
-- get_atomv - allocate atom of variable size.
--
-- *Synopsis*
--
-- #include "glpset.h"
-- void *get_atomv(POOL *pool, int size);
--
-- *Description*
--
-- The get_atomv routine allocates an atom (i.e. continuous space of
-- memory) using the specified memory pool. It is assumed that the pool
-- was created by the create_pool routine with size = 0. The actual
-- size (in bytes) of atom which should be allocated is specified by
-- size (it should be positive and not greater than 256).
--
-- Note that being allocated the atom initially contains arbitrary data
-- (not binary zeros).
--
-- *Returns*
--
-- The get_atomv routine returns a pointer to the allocated atom. */

void *get_atomv(POOL *pool, int size)
{     void *ptr;
      if (pool->size != 0)
         fault("get_atomv: pool cannot be used to allocate an atom of v"
            "ariable size");
      if (!(1 <= size && size <= 256))
         fault("get_atomv: invalid atom size");
      /* actual atom size should be not less than sizeof(void *) and
         should be properly aligned */
      if (size < sizeof(void *)) size = sizeof(void *);
      size = align_datasize(size);
      /* if the last allocated block does not exist or if it has not
         enough space, we need a new block */
      if (pool->link == NULL || pool->used + size > POOL_SIZE)
      {  /* we can pull a new block from the list of free blocks, or if
            this list is empty, we need to allocate such block */
         if (pool->stock != NULL)
         {  ptr = pool->stock;
            pool->stock = *(void **)ptr;
         }
         else
            ptr = umalloc(POOL_SIZE);
         /* the new block becomes the last allocated block */
         *(void **)ptr = pool->link;
         pool->link = ptr;
         /* now only few bytes in the new block are used to hold a
            pointer to the previous allocated block */
         pool->used = align_datasize(sizeof(void *));
      }
      /* the last allocated block exists and it has enough space to
         allocate an atom */
      ptr = (void *)((char *)pool->link + pool->used);
      pool->used += size;
      pool->count++;
#if 0
      memset(ptr, '\0', size);
#else
      memset(ptr, '?', size);
#endif
      return ptr;
}

/* eof */
