/* glpmat/sum_mplets.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- sum_mplets - sum multiplets of sparse matrix.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- int sum_mplets(MAT *A, double eps);
--
-- *Description*
--
-- The sum_mplets routine sums multuplets (i.e. elements which have
-- identical row and column indices) of the sparse matrix A in order to
-- remove all multiplets from the matrix replacing them by their sums.
--
-- The parameter eps is used to remove all zero and tiny elements which
-- might appear in the matrix as the result of summation. If eps = 0.0,
-- only zero elements (including multiplets) are removed.
--
-- *Returns*
--
-- The sum_mplets routine returns total number of elements which have
-- been removed from the matrix. */

int sum_mplets(MAT *A, double eps)
{     int i;
      sort_mat(A);
      for (i = 1; i <= A->m; i++)
      {  ELEM *ee = NULL, *e;
         /* ee points to the first element of multiplet; e points to
            the current element */
         for (e = A->row[i]; e != NULL; e = e->row)
         {  if (ee == NULL || ee->j < e->j)
               ee = e;
            else
            {  insist(ee != e && ee->j == e->j);
               ee->val += e->val, e->val = 0.0;
            }
         }
      }
      /* remove all zero and tiny elements */
      return scrape_mat(A, eps);
}

/* eof */
