/* glpmat/load_mat.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpmat.h"
#include "glptext.h"

/*----------------------------------------------------------------------
-- load_mat - read sparse matrix from text file using plain format.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *load_mat(char *fname);
--
-- *Description*
--
-- The load_mat routine reads a sparse matrix from the text file, whose
-- name is the character string fname, using plain format (this format
-- is described in comments to the routine save_mat; note that the order
-- of matrix elements may be arbitrary).
--
-- *Returns*
--
-- If the operation was successful, the load_mat routine returns a
-- pointer to the loaded matrix. Otherwise the routine returns NULL. */

MAT *load_mat(char *fname)
{     TEXT *text;
      MAT *A = NULL;
      int m, n, nz, i, j, k;
      double s, aij;
      print("load_mat: reading sparse matrix from `%s'...", fname);
      text = open_text(fname);
      if (text == NULL) goto fail;
      scan_token(text); if (text->err > 0) goto fail;
      if (!(text->token == T_NAME && strcmp(text->image, "MAT") == 0))
      {  error("%s:%d: missing marker `MAT'",
            text->file, text->line);
         goto fail;
      }
      scan_token(text); if (text->err > 0) goto fail;
      if (!(text->token == T_INT && text->t_int > 0))
      {  error("%s:%d: number of rows missing or invalid",
            text->file, text->line);
         goto fail;
      }
      m = text->t_int;
      scan_token(text); if (text->err > 0) goto fail;
      if (!(text->token == T_INT && text->t_int > 0))
      {  error("%s:%d: number of columns missing or invalid",
            text->file, text->line);
         goto fail;
      }
      n = text->t_int;
      scan_token(text); if (text->err > 0) goto fail;
      if (!(text->token == T_INT && text->t_int >= 0))
      {  error("%s:%d: number of non-zeros missing or invalid",
            text->file, text->line);
         goto fail;
      }
      nz = text->t_int;
      A = create_mat(m, n);
      for (k = 1; k <= nz; k++)
      {  scan_token(text); if (text->err > 0) goto fail;
         if (!(text->token == T_INT && 1 <= text->t_int &&
               text->t_int <= m))
         {  error("%s:%d: row number missing or invalid",
               text->file, text->line);
            goto fail;
         }
         i = text->t_int;
         scan_token(text); if (text->err > 0) goto fail;
         if (!(text->token == T_INT && 1 <= text->t_int &&
               text->t_int <= n))
         {  error("%s:%d: column number missing or invalid",
               text->file, text->line);
            goto fail;
         }
         j = text->t_int;
         scan_token(text); if (text->err > 0) goto fail;
         if (text->token == T_SPEC && text->image[0] == '+')
         {  s = +1.0;
            scan_token(text); if (text->err > 0) goto fail;
         }
         else if (text->token == T_SPEC && text->image[0] == '-')
         {  s = -1.0;
            scan_token(text); if (text->err > 0) goto fail;
         }
         else
            s = +1.0;
         if (!(text->token == T_INT || text->token == T_REAL))
         {  error("%s:%d: element value missing or invalid",
               text->file, text->line);
            goto fail;
         }
         aij = s * text->t_real;
         new_elem(A, i, j, aij);
      }
      scan_token(text); if (text->err > 0) goto fail;
      if (!(text->token == T_NAME && strcmp(text->image, "END") == 0))
      {  error("%s:%d: missing marker `END'",
            text->file, text->line);
         goto fail;
      }
      scan_token(text); if (text->err > 0) goto fail;
      if (text->token != T_EOF)
      {  error("%s:%d: extra symbols behind `END'",
            text->file, text->line);
         goto fail;
      }
      close_text(text);
      return A;
fail: if (text != NULL) close_text(text);
      if (A != NULL) delete_mat(A);
      return NULL;
}

/* eof */
