/* glpmat/clear_lines.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- clear_lines() -- clear rows and columns of sparse matrix.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *clear_lines(MAT *A, int rs[], int cs[]);
--
-- *Description*
--
-- The routine clear_lines() erases elements in the rows and columns of
-- the sparse matrix, which belong to the specified subsets.
--
-- The array rs specifies a subset of rows and should have at least 1+m
-- locations, where m is number of rows in the matrix A. The location
-- rs[0] is not used. If rs[i] is non-zero, the routine erases elements
-- in the i-th row. Otherwise, if rs[i] is zero, the i-th row is not
-- changed.
--
-- The array cs specifies a subset of columns and should have at least
-- 1+n locations, where n is number of columns in the matrix A. The
-- location cs[0] is not used. If cs[j] is non-zero, the routine erases
-- elements in the j-th columns. Otherwise, if cs[j] is zero, the j-th
-- column is not changed.
--
-- It is allowed to specify rs = NULL or cs = NULL, in which case the
-- subset of rows or columns is assumed to be empty.
--
-- *Complexity*
--
-- Independently on how many rows and columns should be nullified the
-- time complexity is O(nz), where nz is the number of elements in the
-- matrix A before the operation.
--
-- *Returns*
--
-- The routine clear_lines() returns a pointer to the matrix A. */

MAT *clear_lines(MAT *A, int rs[], int cs[])
{     ELEM *ptr, *e;
      int m = A->m, n = A->n, i, j;
      /* clear rows */
      if (rs != NULL)
      {  for (i = 1; i <= m; i++)
            if (rs[i]) A->row[i] = NULL;
         for (j = 1; j <= n; j++)
         {  ptr = NULL;
            while (A->col[j] != NULL)
            {  e = A->col[j];
               A->col[j] = e->col;
               if (rs[e->i])
                  free_atom(A->pool, e);
               else
                  e->col = ptr, ptr = e;
            }
            A->col[j] = ptr;
         }
      }
      /* clear columns */
      if (cs != NULL)
      {  for (j = 1; j <= n; j++)
            if (cs[j]) A->col[j] = NULL;
         for (i = 1; i <= m; i++)
         {  ptr = NULL;
            while (A->row[i] != NULL)
            {  e = A->row[i];
               A->row[i] = e->row;
               if (cs[e->j])
                  free_atom(A->pool, e);
               else
                  e->row = ptr, ptr = e;
            }
            A->row[i] = ptr;
         }
      }
      return A;
}

/* eof */
