/* glpapi/glp_new_row.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_new_row - create new row.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_new_row(LPI *lp, char *name);
--
-- *Description*
--
-- The routine glp_new_row creates a new row and adds it to the end of
-- the row list (thus, sequential numbers of existing rows remain
-- unchanged). Initially the created row is empty and corresponds to an
-- equality constraint with zero right-hand side.
--
-- The parameter name specifies a symbolic name, which is assigned to
-- the new row. If this parameter is NULL, no name is assigned.
--
-- *Complexity*
--
-- If the parameter name is not NULL, the time complexity is O(log m'),
-- where m' is number of named rows in the row list. Otherwise, the time
-- complexity is O(1). */

void glp_new_row(LPI *lp, char *name)
{     LPIROW *row;
      /* create new row */
      row = get_atom(lp->row_pool);
      row->seqn = lp->m+1;
      row->name = NULL;
      row->type = 'S';
      row->lb = 0.0;
      row->ub = 0.0;
      row->fctr = 1.0;
      row->ptr = NULL;
      row->tagx = 'B';
      row->valx = 0.0;
      row->dx = 0.0;
      /* add new row to the end of the row list */
      if (lp->m == lp->m_max)
      {  /* enlarge the row list */
         LPIROW **temp;
         int i;
         lp->m_max += lp->m_max;
         temp = ucalloc(1+lp->m_max, sizeof(LPIROW *));
         for (i = 1; i <= lp->m; i++) temp[i] = lp->row[i];
         ufree(lp->row);
         lp->row = temp;
      }
      lp->m++;
      lp->row[lp->m] = row;
      /* assign row name */
      if (name != NULL)
      {  AVLNODE *node;
         if (glp_check_name(name))
            fault("glp_new_row: invalid row name");
         lp->row[lp->m]->name = create_str(lp->str_pool);
         set_str(lp->row[lp->m]->name, name);
         if (find_by_key(lp->row_tree, lp->row[lp->m]->name) != NULL)
            fault("glp_new_row: duplicate row name `%s'", name);
         node = insert_by_key(lp->row_tree, lp->row[lp->m]->name);
         node->link = lp->row[lp->m];
      }
      return;
}

/* eof */
