/* glprsm/build_basis.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- build_basis - build advanced basis.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int build_basis(RSM *rsm, LPI *lp);
--
-- *Description*
--
-- The routine build_basis() obtains information about advanced basis
-- specified in a problem object lp and uses this information in order
-- to build the basis in the RSM block.
--
-- *Returns*
--
-- The routine build_basis() returns one of the following codes:
--
-- 0 - no errors;
-- 1 - type of a row/column is not compatible with its status;
-- 2 - basis has invalid structure;
-- 3 - unable to invert the basis matrix.
--
-- Should note that in case of error statuses of the variables as well
-- as representation of the basis matrix become invalid, therefore the
-- RSM block should not be used until necessary corrections have been
-- made. */

int build_basis(RSM *rsm, LPI *lp)
{     int m = rsm->m, n = rsm->n, i, j, k;
      insist(m == glp_get_num_rows(lp));
      insist(n == glp_get_num_cols(lp));
      i = 0; /* number of basic variables */
      j = 0; /* number of non-basic variables */
      for (k = 1; k <= m+n; k++)
      {  int type, tagx;
         /* obtain type and status of the k-th variable */
         if (k <= m)
         {  glp_get_row_bnds(lp, k, &type, NULL, NULL);
            glp_get_row_soln(lp, k, &tagx, NULL, NULL);
         }
         else
         {  glp_get_col_bnds(lp, k-m, &type, NULL, NULL);
            glp_get_col_soln(lp, k-m, &tagx, NULL, NULL);
         }
         /* check for compatibility between type and status */
         if (!(tagx == 'B' ||
            type == 'F' && tagx == 'F' ||
            type == 'L' && tagx == 'L' ||
            type == 'U' && tagx == 'U' ||
            type == 'D' && tagx == 'L' ||
            type == 'D' && tagx == 'U' ||
            type == 'S' && tagx == 'S')) return 1;
         /* store information about basis into RSM */
         if (tagx == 'B')
         {  /* x[k] is basic variable xB[i] */
            i++;
            if (i > m) return 2;
            rsm->posx[k] = +i;
            rsm->indb[i] =  k;
         }
         else
         {  /* x[k] is non-basic variable xN[j] */
            j++;
            if (j > n) return 2;
            rsm->posx[k] = -j;
            rsm->indn[j] =  k;
            rsm->tagn[j] = tagx;
         }
      }
      insist(i == m && j == n);
      /* try to invert the basis matrix */
      if (invert_b(rsm)) return 3;
      /* hmm... the basis seems to be ok */
      return 0;
}

/* eof */
