/* glpapi/glp_get_col_bnds.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_get_col_bnds - determine column bounds.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_get_col_bnds(LPI *lp, int j, int *type, double *lb,
--    double *ub);
--
-- *Description*
--
-- The routine glp_get_col_bnds stores the type, the lower bound, and
-- the upper bound of the j-th column to locations, which the parameters
-- type, lb, and ub point to, respectively.
--
-- If some of the parameters type, lb, or ub is NULL, the corresponding
-- value is not stored.
--
-- Types and bounds have the following meaning:
--
--    Type        Bounds            Note
--    ----------------------------------------
--    'F'   -inf <  x <  +inf   free variable
--    'L'     lb <= x <  +inf   lower bound
--    'U'   -inf <  x <=  ub    upper bound
--    'D'     lb <= x <=  ub    double bound
--    'S'           x  =  lb    fixed variable
--
-- where x is the corresponding structural variable.
--
-- If the column has no lower bound, *lb is set to zero. If the column
-- has no upper bound, *ub is set to zero. If the column is of fixed
-- type, *lb and *ub are set to the same value. */

void glp_get_col_bnds(LPI *lp, int j, int *type, double *lb, double *ub)
{     if (!(1 <= j && j <= lp->n))
         fault("glp_get_col_bnds: j = %d; invalid column number", j);
      if (type != NULL) *type = lp->col[j]->type;
      if (lb != NULL) *lb = lp->col[j]->lb;
      if (ub != NULL) *ub = lp->col[j]->ub;
      return;
}

/* eof */
