/* glpapi/glp_delete_rows.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_delete_rows - delete rows from problem instance.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_delete_rows(LPI *lp, int flag[]);
--
-- *Description*
--
-- The routine glp_delete_rows deletes specified subset of rows from
-- the problem object.
--
-- The array flag should have at least 1+m locations, where m is number
-- of rows in the problem object. The location flag[0] is not used.
-- If flag[i] (1 <= i <= n) is non-zero, the routine deletes the i-th
-- row. Otherwise, if flag[i] is zero, the i-th row is kept.
--
-- *Complexity*
--
-- The time complexity is O(nz), where nz is total number of elements
-- in the constraint matrix before operation. */

void glp_delete_rows(LPI *lp, int flag[])
{     int i, m;
      glp_clear_mat_rows(lp, flag);
      m = 0;
      for (i = 1; i <= lp->m; i++)
      {  if (flag[i])
         {  glp_set_row_name(lp, i, NULL);
            free_atom(lp->row_pool, lp->row[i]);
         }
         else
         {  ELEM *e;
            m++;
            lp->row[m] = lp->row[i];
            lp->row[m]->seqn = m;
            for (e = lp->row[i]->ptr; e != NULL; e = e->row) e->i = m;
         }
      }
      lp->m = m;
      return;
}

/* eof */
