/* glplp/save_lp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <float.h>
#include <stdio.h>
#include <string.h>
#include "glplp.h"

/*----------------------------------------------------------------------
-- save_lp - write LP data block to text file using plain format.
--
-- *Synopsis*
--
-- #include "glplp.h"
-- int save_lp(LP *lp, char *fname);
--
-- *Description*
--
-- The save_lp routine writes the LP problem data block, which lp points
-- to, to the text file, whose name is the character string fname, using
-- the following plain format:
--
--    'LP'           \n
--    m n nz dir nc  \n
--    type lb ub     \n    |
--    . . .                |-> m+n lines
--    type lb ub     \n    |
--    i j aij        \n    |
--    . . .                |-> nz lines
--    i j aij        \n    |
--    j cj           \n    |
--    . . .                |-> nc lines
--    j cj           \n    |
--    'EOD'          \n
--
-- where:
-- m     - number of rows (auxiliary variables);
-- n     - number of columns (structural variables);
-- nz    - number of elements in constraint matrix;
-- dir   - optimization direction:
--         '-' - minimization;
--         '+' - maximization.
-- nc    - number of the objective function coefficients (including the
--         constant term);
-- type  - type of auxiliary or structural variable:
--         'F' - free variable:    -inf <  x[k] < +inf
--         'L' - lower bound:      l[k] <= x[k] < +inf
--         'U' - upper bound:      -inf <  x[k] <= u[k]
--         'D' - double bound:     l[k] <= x[k] <= u[k]
--         'S' - fixed variable:   l[k]  = x[k]  = u[k]
-- lb    - lower bound of auxiliary or structural variable;
-- ub    - upper bound of auxiliary or structural variable;
-- i     - row index;
-- j     - column index;
-- aij   - value of element of constraint matrix;
-- cj    - value of the objective function coefficient (j = 0 means the
--         constant term).
--
-- The save_lp routine writes the constraint matrix in row-wise manner.
-- In order that its elements follow in the order of increasing column
-- indices the sort_mat routine may be used before.
--
-- *Returns*
--
-- If the operation was successful, the save_lp routine returns zero.
-- Otherwise the routine returns non-zero. */

int save_lp(LP *lp, char *fname)
{     FILE *fp;
      int nz, nc, i, j, k;
      print("save_lp: writing LP data block to `%s'...", fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("save_lp: unable to create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* header */
      fprintf(fp, "'LP'\n");
      nz = count_nz(lp->A, 0);
      nc = 0;
      for (j = 0; j <= lp->n; j++) if (lp->c[j] != 0) nc++;
      fprintf(fp, "%d %d %d '%c' %d\n", lp->m, lp->n, nz, lp->dir, nc);
      /* auxiliary and structural variables */
      for (k = 1; k <= lp->m+lp->n; k++)
      {  fprintf(fp, "'%c' %.*g %.*g\n", lp->type[k],
            DBL_DIG, lp->lb[k], DBL_DIG, lp->ub[k]);
      }
      /* constraint matrix */
      for (i = 1; i <= lp->m; i++)
      {  ELEM *e;
         for (e = lp->A->row[i]; e != NULL; e = e->row)
            fprintf(fp, "%d %d %.*g\n", e->i, e->j, DBL_DIG, e->val);
      }
      /* objective coefficients */
      for (j = 0; j <= lp->n; j++)
      {  if (lp->c[j] != 0.0)
            fprintf(fp, "%d %.*g\n", j, DBL_DIG, lp->c[j]);
      }
      fprintf(fp, "'EOD'\n");
      fflush(fp);
      if (ferror(fp))
      {  error("save_lp: writing error on `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
