/* glpstr/compare_str.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpstr.h"

/*----------------------------------------------------------------------
-- compare_str - compare segmented character strings.
--
-- *Synopsis*
--
-- #include "glpstr.h"
-- int compare_str(STR *str1, STR *str2);
--
-- *Returns*
--
-- The compare_str compares segmented character strings str1 and str2,
-- and returns one of the following values:
--
-- < 0, if str1 is lexicographically less than str2;
-- = 0, if str1 and str2 are identical;
-- > 0, if str1 is lexicographically greater than str2.
--
-- If compared strings have different length, the shorter string is
-- considered as if it would be padded by binary zeros to the length of
-- the longer string. */

int compare_str(STR *str1, STR *str2)
{     SQE *sqe1 = str1->head, *sqe2 = str2->head;
      int len1 = str1->len, len2 = str2->len, ret;
      while (len1 > 0 || len2 > 0)
      {  int n1 = (len1 <= SQE_SIZE ? len1 : SQE_SIZE);
         int n2 = (len2 <= SQE_SIZE ? len2 : SQE_SIZE);
         char buf1[SQE_SIZE], buf2[SQE_SIZE];
         memset(buf1, '\0', SQE_SIZE);
         if (n1 > 0)
         {  insist(sqe1 != NULL);
            memcpy(buf1, sqe1->data, n1);
            len1 -= n1, sqe1 = sqe1->next;
         }
         memset(buf2, '\0', SQE_SIZE);
         if (n2 > 0)
         {  insist(sqe2 != NULL);
            memcpy(buf2, sqe2->data, n2);
            len2 -= n2, sqe2 = sqe2->next;
         }
         ret = memcmp(buf1, buf2, SQE_SIZE);
         if (ret != 0) break;
      }
      return ret;
}

/* eof */
