/* glprsm/eval_pi.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glprsm.h"

/*----------------------------------------------------------------------
-- eval_pi - compute simplex multipliers.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void eval_pi(RSM *rsm, double c[], double pi[]);
--
-- *Description*
--
-- The eval_pi routine computes simplex multipliers pi = (pi_1, ...,
-- pi_m), i.e. Lagrange multipliers for the equality constraints, that
-- correspond to the current basis and stores pi_1, ..., pi_m into
-- locations pi[1], ..., pi[m] respectively.
--
-- On entry the array c should contain the vector of coefficients of the
-- objective function in locations c[1], ..., c[m+n]. The array c is not
-- changed on exit.
--
-- The vector pi is computed using the following formula:
--
--    pi = inv(B') * cB,
--
-- where B' is a matrix transposed to the basis matrix B, cB is the
-- subvector of coefficients of the objective function at the basic
-- variables. */

void eval_pi(RSM *rsm, double c[], double pi[])
{     int m = rsm->m, i, k;
      double *cB = pi;
      /* pi = inv(BT) * cB */
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         cB[i] = c[k];
      }
      btran(rsm, pi);
      return;
}

/* eof */
