/* glprsm/dual_col.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- dual_col - choose non-basic variable (dual, standard technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int dual_col(RSM *rsm, int tagp, double ap[], double cbar[],
--    double tol);
--
-- *Description*
--
-- The dual_col routine chooses non-basic variable (xN)q (i.e. pivot
-- column of the simplex table) which should enter the basis on the next
-- iteration of the dual simplex method. The routine is based on the
-- standard (textbook) technique.
--
-- The parameter tagp is a tag that specifies to what subset xL or xU
-- the chosen basic variable (xB)p should be attributed after it has
-- left the basis. This tag is set by the dual_row routine.
--
-- The array ap should contain p-th (pivot) row if the simplex table,
-- i.e. p-th row of the matrix A~ in locations ap[1], ..., ap[n]. This
-- array is not changed on exit.
--
-- The array cbar should contain reduced costs of non-basic variables in
-- locations cbar[1], ..., cbar[n]. This array is not changed on exit.
--
-- The parameter tol is a relative tolerance (see below).
--
-- The dual_col routine implements the standard (textbook) ratio test
-- for choosing non-basic variable, i.e. the routine determines that
-- non-basic variable whose dual variable reaches its (lower or upper)
-- zero bound first when the dual variable (lambda_B)p that corresponds
-- to the chosen basic variable (xB)p is changing in the feasible
-- direction (increasing if (xB)p goes on its lower bound or decreasing
-- if (xB)p goes on its upper bound). Besides, the following additional
-- rules are used:
--
-- if |ap[j]| < tol * max|ap[*]|, i.e. if the influence coefficient
-- ap[j] is relatively close to zero, it is assumed that the
-- corresponding non-basic variable (xN)j doesn't affect on the basic
-- variable (xB)p and therefore such non-basic variable is not
-- considered to be chosen;
--
-- if the reduced cost cbar[j] of some non-basic variable (xN)j violates
-- its (zero) bound, it is assumed that this happens dur to round-off
-- errors and actually the reduced cost is exactly zero (because the
-- current basis solution should be primal feasible);
--
-- if several dual variables that correspond to non-basic variables
-- reach their (zero) bounds first at the same time, the routine prefers
-- that variable which has the largest (in absolute value) influence
-- coefficient.
--
-- For further details see the program documentation.
--
-- *Returns*
--
-- If the choice has been made, the dual_col routine returns q which is
-- a number of the chosen non-basic variable (xN)q, 1 <= q <= n.
-- Otherwise, if the dual variable (lambda_B)p that corresponds to the
-- chosen basic variable (xB)p can unlimitedly change in the feasible
-- direction and therefore the choice us impossible, the routine returns
-- zero. */

int dual_col(RSM *rsm, int tagp, double ap[], double cbar[], double tol)
{     int n = rsm->n, j, q;
      double big, eps, temp, teta;
      /* compute the absolute tolerance eps using the given relative
         tolerance tol */
      big = 0.0;
      for (j = 1; j <= n; j++)
         if (big < fabs(ap[j])) big = fabs(ap[j]);
      eps = tol * big;
      /* turn to the case of increasing xB[p] in order to simplify
         program logic */
      if (tagp == 'U') for (j = 1; j <= n; j++) ap[j] = - ap[j];
      /* initial settings */
      q = 0, teta = DBL_MAX, big = 0.0;
      /* look through the list of non-basic variables */
      for (j = 1; j <= n; j++)
      {  /* if the coefficient ap[j] is too small, it is assumed that
            xB[p] doesn't depend on xN[j] */
         if (ap[j] == 0.0 || fabs(ap[j]) < eps) continue;
         /* analyze main cases */
         if (rsm->tagn[j] == 'F')
         {  /* xN[j] is free variable */
            temp = 0.0;
         }
         else if (rsm->tagn[j] == 'L')
         {  /* xN[j] is on its lower bound */
            if (ap[j] < 0.0) continue;
            temp = cbar[j] / ap[j];
         }
         else if (rsm->tagn[j] == 'U')
         {  /* xN[j] is on its upper bound */
            if (ap[j] > 0.0) continue;
            temp = cbar[j] / ap[j];
         }
         else if (rsm->tagn[j] == 'S')
         {  /* xN[j] is fixed variable */
            continue;
         }
         else
            insist(rsm->tagn[j] != rsm->tagn[j]);
         /* if reduced cost of xN[j] (i.e. dual variable) slightly
            violates its bound, temp is negative; in this case it is
            assumed that reduced cost is exactly on its bound (i.e.
            equal to zero), therefore temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply minimal ratio test */
         if (teta > temp || teta == temp && big < fabs(ap[j]))
         {  q = j;
            teta = temp;
            big = fabs(ap[j]);
         }
      }
      /* restore original signs of the coefficients ap[j] */
      if (tagp == 'U') for (j = 1; j <= n; j++) ap[j] = - ap[j];
      /* returns to the calling program */
      return q;
}

/* eof */
