/* glprsm/harris_col.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- harris_col - choose non-basic variable (dual, Harris technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int harris_col(RSM *rsm, int tagp, double ap[], double c[],
--    double cbar[], double tol, double tol1);
--
-- *Description*
--
-- The harris_col routine chooses non-basic variable (xN)q (i.e. pivot
-- column of the simplex table) which should enter the basis on the next
-- iteration of the dual simplex method. The routine is based on the
-- two-pass ratio test proposed by P.Harris.
--
-- The harris_col routine has analogous program specification as the
-- dual_col routine.
--
-- The first additional parameter is the array c which should contain
-- the expanded vector c of coefficients of the objective function in
-- locations c[1], ..., c[m+n]. It is assumed that the reduced costs are
-- computed using the vector c passed to this routine.
--
-- The second additional parameter is tol1 which is a relative tolerance
-- used for relaxing zero bounds of dual variables lambda_N (that
-- correspond to primal non-basic variables xN) on the first pass. The
-- routine replaces original zero bounds of dual variables by their
-- relaxed bounds:
--
--    (lambda_N)j >= -eps  or  (lambda_N)j <= +eps
--
-- where
--
--    eps = tol1 * max(1, |(cN)j|)
--
-- (the absolute tolerance eps reflects that the reduced cost on
-- non-basic variable (xN)j is the difference dj = (cN)j - pi'*Nj).
--
-- For futher details see the program specification. */

int harris_col(RSM *rsm, int tagp, double ap[], double c[],
      double cbar[], double tol, double tol1)
{     int n = rsm->n, j, k, q;
      double big, eps, temp, teta;
#     define gap (tol1 * (fabs(c[k]) > 1.0 ? fabs(c[k]) : 1.0))
      /* compute the absolute tolerance eps using the given relative
         tolerance tol */
      big = 0.0;
      for (j = 1; j <= n; j++)
         if (big < fabs(ap[j])) big = fabs(ap[j]);
      eps = tol * big;
      /* turn to the case of increasing xB[p] in order to simplify
         program logic */
      if (tagp == 'U') for (j = 1; j <= n; j++) ap[j] = - ap[j];
      /* initial settings for the first pass */
      teta = DBL_MAX;
      /* the first look through the list of non-basic variables */
      for (j = 1; j <= n; j++)
      {  /* if the coefficient ap[j] is too small, it is assumed that
            xB[p] doesn't depend on xN[j] */
         if (ap[j] == 0.0 || fabs(ap[j]) < eps) continue;
         /* analyze main cases */
         k = rsm->indn[j]; /* x[k] = xN[j] */
         if (rsm->tagn[j] == 'F')
         {  /* xN[j] is free variable */
            if (ap[j] > 0.0) goto lo; else goto up;
         }
         else if (rsm->tagn[j] == 'L')
         {  /* xN[j] is on its lower bound */
            if (ap[j] < 0.0) continue;
lo:         temp = (cbar[j] + gap) / ap[j];
         }
         else if (rsm->tagn[j] == 'U')
         {  /* xN[j] is on its upper bound */
            if (ap[j] > 0.0) continue;
up:         temp = (cbar[j] - gap) / ap[j];
         }
         else if (rsm->tagn[j] == 'S')
         {  /* xN[j] is fixed variable */
            continue;
         }
         else
            insist(rsm->tagn[j] != rsm->tagn[j]);
         /* if reduced costs of xN[j] (i.e. dual variable) slightly
            violates its bound, temp is negative; in this case it is
            assumed that reduced cost is exactly on its bound (i.e.
            equal to zero), therefore temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* compute maximal allowable change of reduced cost of the
            basis variable xB[p] */
         if (teta > temp) teta = temp;

      }
      /* initial settings for the second pass */
      q = 0, big = 0.0;
      /* the second look through the list of non-basis variables */
      for (j = 1; j <= n; j++)
      {  /* if the coefficient ap[j] is too small, it is assumed that
            xB[p] doesn't depend on xN[j] */
         if (ap[j] == 0.0 || fabs(ap[j]) < eps) continue;
         /* analyze main cases */
         if (rsm->tagn[j] == 'F')
         {  /* xN[j] is free variable */
            temp = 0.0;
         }
         else if (rsm->tagn[j] == 'L')
         {  /* xN[j] is on its lower bound */
            if (ap[j] < 0.0) continue;
            temp = cbar[j] / ap[j];
         }
         else if (rsm->tagn[j] == 'U')
         {  /* xN[j] is on its upper bound */
            if (ap[j] > 0.0) continue;
            temp = cbar[j] / ap[j];
         }
         else if (rsm->tagn[j] == 'S')
         {  /* xN[j] is fixed variable */
            continue;
         }
         else
            insist(rsm->tagn[j] != rsm->tagn[j]);
         /* if reduced costs of xN[j] (i.e. dual variable) slightly
            violates its bound, temp is negative; in this case it is
            assumed that reduced cost is exactly on its bound (i.e.
            equal to zero), therefore temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply the dual version of Harris' rule */
         if (temp <= teta &&  big < fabs(ap[j]))
         {  q = j;
            big = fabs(ap[j]);
         }
      }
      /* restore original signs of the coefficients ap[j] */
      if (tagp == 'U') for (j = 1; j <= n; j++) ap[j] = - ap[j];
      /* returns to the calling program */
      return q;
#     undef gap
}

/* eof */
