/* glprsm/eval_bbar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- eval_bbar - compute values of basic variables.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void eval_bbar(RSM *rsm, double bbar[]);
--
-- *Description*
--
-- The eval_bbar routine computes values of basic variables xB = beta =
-- = (beta_1, ..., beta_m) that correspond to the current basis solution
-- and stores beta_1, ..., beta_m into locations bbar[1], ..., bbar[m]
-- respectively.
--
-- The vector beta is computed using the following formula:
--
--    beta = - inv(B) * (N * xN) =
--         = inv(B) * (- N[1]*xN[1] - ... - N[n]*xN[n]),
--
-- where N[j] is the column of the expanded constraint matrix A, which
-- corresponds to the non-basic variable xN[j]. */

void eval_bbar(RSM *rsm, double bbar[])
{     ELEM *e;
      int m = rsm->m, n = rsm->n, i, j, k;
      double *u = bbar, t;
      /* u := - N*xN = - N[1]*xN[1] - ... - N[n]*xN[n] */
      for (i = 1; i <= m; i++) u[i] = 0.0;
      for (j = 1; j <= n; j++)
      {  t = eval_xn(rsm, j); /* current value of xN[j] */
         if (t == 0.0) continue;
         k = rsm->indn[j]; /* x[k] = xN[j] */
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            u[e->i] -= e->val * t;
      }
      /* bbar := inv(B)*u */
      ftran(rsm, u, 0);
      return;
}

/* eof */
