/* glprsm/check_bbar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glprsm.h"

/*----------------------------------------------------------------------
-- check_bbar - check basis solution for primal feasibility.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int check_bbar(RSM *rsm, double bbar[], double tol);
--
-- *Description*
--
-- The check_bbar routine checks the given basis solution for primal
-- feasibility.
--
-- The array bbar should contain the given values of basic variables
-- beta = (beta_1, ..., beta_m) in locations bbar[1], ..., bbar[m]
-- respectively.
--
-- The parameter tol > 0 is a relative tolerance.
--
-- In order to see if the given basis solution is primal feasible the
-- routine checks the primal feasibility conditions which using the
-- relative tolerance are the following:
--
--    (lB)i - eps <= beta_i <= (uB)i + eps, i = 1,...,m
--
-- where
--
--    eps = tol * max(1, |(lB)i|) in case of lower bound
--    eps = tol * max(1, |(uB)i|) in case of upper bound
--
-- *Returns*
--
-- If the given basis solution is primal feasible (i.e. it satisfies to
-- the conditions above) the check_bbar routine returns zero. Otherwise
-- the routine returns non-zero. */

int check_bbar(RSM *rsm, double bbar[], double tol)
{     int m = rsm->m, i, k;
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == 'L' || rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
         {  /* xB[i] has lower bound */
            if (check_rr(bbar[i], rsm->lb[k], tol) < -1) return 1;
         }
         if (rsm->type[k] == 'U' || rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
         {  /* xB[i] has upper bound */
            if (check_rr(bbar[i], rsm->ub[k], tol) > +1) return 1;
         }
      }
      return 0;
}

/* eof */
